//****************************************************************************
//
// Copyright (c) 2021 Broadcom Corporation
//
// This program is the proprietary software of Broadcom Corporation and/or
// its licensors, and may only be used, duplicated, modified or distributed
// pursuant to the terms and conditions of a separate, written license
// agreement executed between you and Broadcom (an "Authorized License").
// Except as set forth in an Authorized License, Broadcom grants no license
// (express or implied), right to use, or waiver of any kind with respect to
// the Software, and Broadcom expressly reserves all rights in and to the
// Software and all intellectual property rights therein.  IF YOU HAVE NO
// AUTHORIZED LICENSE, THEN YOU HAVE NO RIGHT TO USE THIS SOFTWARE IN ANY WAY,
// AND SHOULD IMMEDIATELY NOTIFY BROADCOM AND DISCONTINUE ALL USE OF THE
// SOFTWARE.
//
// Except as expressly set forth in the Authorized License,
//
// 1.     This program, including its structure, sequence and organization,
// constitutes the valuable trade secrets of Broadcom, and you shall use all
// reasonable efforts to protect the confidentiality thereof, and to use this
// information only in connection with your use of Broadcom integrated circuit
// products.
//
// 2.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, THE SOFTWARE IS PROVIDED
// "AS IS" AND WITH ALL FAULTS AND BROADCOM MAKES NO PROMISES, REPRESENTATIONS
// OR WARRANTIES, EITHER EXPRESS, IMPLIED, STATUTORY, OR OTHERWISE, WITH
// RESPECT TO THE SOFTWARE.  BROADCOM SPECIFICALLY DISCLAIMS ANY AND ALL
// IMPLIED WARRANTIES OF TITLE, MERCHANTABILITY, NONINFRINGEMENT, FITNESS FOR
// A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR COMPLETENESS, QUIET
// ENJOYMENT, QUIET POSSESSION OR CORRESPONDENCE TO DESCRIPTION. YOU ASSUME
// THE ENTIRE RISK ARISING OUT OF USE OR PERFORMANCE OF THE SOFTWARE.
//
// 3.     TO THE MAXIMUM EXTENT PERMITTED BY LAW, IN NO EVENT SHALL BROADCOM
// OR ITS LICENSORS BE LIABLE FOR (i) CONSEQUENTIAL, INCIDENTAL, SPECIAL,
// INDIRECT, OR EXEMPLARY DAMAGES WHATSOEVER ARISING OUT OF OR IN ANY WAY
// RELATING TO YOUR USE OF OR INABILITY TO USE THE SOFTWARE EVEN IF BROADCOM
// HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES; OR (ii) ANY AMOUNT IN
// EXCESS OF THE AMOUNT ACTUALLY PAID FOR THE SOFTWARE ITSELF OR U.S. $1,
// WHICHEVER IS GREATER. THESE LIMITATIONS SHALL APPLY NOTWITHSTANDING ANY
// FAILURE OF ESSENTIAL PURPOSE OF ANY LIMITED REMEDY.
//
//*****************************************************************************
//
//  Filename:       eventfd.cxx
//  Author:         Ignatius Cheng
//  Creation Date:  July 12, 2021
//
//****************************************************************************
//  Description:
//      This is BRCM implmenetation of eventfd() for eCos
//
//****************************************************************************
//==========================================================================
//
//      eventfd.cxx
//
//      Fileio eventfd operations
//
//==========================================================================


#include <pkgconf/hal.h>
#include <pkgconf/kernel.h>
#include <pkgconf/io_fileio.h>

#include <pkgconf/hal.h>
#include <pkgconf/kernel.h>
#include <pkgconf/io_fileio.h>

#include <cyg/kernel/ktypes.h>         // base kernel types
#include <cyg/infra/cyg_trac.h>        // tracing macros
#include <cyg/infra/cyg_ass.h>         // assertion macros

#include "fio.h"                       // Private header

#include <cyg/fileio/eventfd.h>
#include <cyg/error/codes.h>
#include <sys/sockio.h>
#include <stdio.h>
#include <stdlib.h>

//==========================================================================
// Forward definitions

// File operations
static int eventfd_fo_read    (struct CYG_FILE_TAG *fp, struct CYG_UIO_TAG *uio);
static int eventfd_fo_write   (struct CYG_FILE_TAG *fp, struct CYG_UIO_TAG *uio);
static int eventfd_fo_lseek   (struct CYG_FILE_TAG *fp, off_t *pos, int whence );
static int eventfd_fo_ioctl   (struct CYG_FILE_TAG *fp, CYG_ADDRWORD cmd,
                               CYG_ADDRWORD data);
static int eventfd_fo_select  (struct CYG_FILE_TAG *fp, int which, CYG_ADDRWORD info);
static int eventfd_fo_fsync   (struct CYG_FILE_TAG *fp, int mode );
static int eventfd_fo_close   (struct CYG_FILE_TAG *fp);
static int eventfd_fo_fstat   (struct CYG_FILE_TAG *fp, struct stat *buf );
static int eventfd_fo_getinfo (struct CYG_FILE_TAG *fp, int key, void *buf, int len );
static int eventfd_fo_setinfo (struct CYG_FILE_TAG *fp, int key, void *buf, int len );


//==========================================================================
// Tables and local variables
static cyg_fileops eventfd_fileops =
{
    eventfd_fo_read,
    eventfd_fo_write,
    eventfd_fo_lseek,
    eventfd_fo_ioctl,
    eventfd_fo_select,
    eventfd_fo_fsync,
    eventfd_fo_close,
    eventfd_fo_fstat,
    eventfd_fo_getinfo,
    eventfd_fo_setinfo
};

//==========================================================================
// Initialization


//==========================================================================
// eventfd API calls

// -------------------------------------------------------------------------
int eventfd (int __count, int __flags)
{
    FILEIO_ENTRY();
    int fd;
    cyg_file *file;
    struct eventfd *efd;

    fd = cyg_fd_alloc(0);
    if( fd < 0 )
        FILEIO_RETURN (EMFILE);

    file = cyg_file_alloc();
    if( file == NULL ) {
        cyg_fd_free(fd);
        FILEIO_RETURN (ENFILE);
    }

    efd = (struct eventfd*) malloc(sizeof(struct eventfd));
    if( efd == NULL ) {
        cyg_fd_free(fd);
        cyg_file_free(file);
        FILEIO_RETURN (ENOMEM);
    }

    cyg_mutex_init(&efd->mutex);
    efd->flags = __flags;
    efd->count = __count;
    cyg_cond_init(&efd->r_pend, &efd->mutex);
    cyg_cond_init(&efd->w_pend, &efd->mutex);
    cyg_selinit(&efd->eventfd_sel_r);
    cyg_selinit(&efd->eventfd_sel_w);
    efd->efd_nr_pend = 0;
    efd->efd_nw_pend = 0;

    file->f_flag   |= CYG_FREAD|CYG_FWRITE;
    file->f_type    = CYG_FILE_TYPE_EVENTFD;
    file->f_ops     = &eventfd_fileops;
    file->f_offset  = 0;
    file->f_data    = (CYG_ADDRWORD)efd;
    file->f_xops    = 0;

    cyg_fd_assign( fd, file );
    FILEIO_RETURN_VALUE(fd);
}

/* Read event counter and possibly wait for events.  */
int eventfd_read (int __fd, eventfd_t *__value)
{
    if ( read(__fd, __value, sizeof(eventfd_t)) == sizeof(eventfd_t) )
        return 0;
    return -1;
}

/* Increment event counter.  */
int eventfd_write (int __fd, eventfd_t __value)
{
    if ( write(__fd, &__value, sizeof(eventfd_t)) == sizeof(eventfd_t) )
        return 0;
    return -1;
}

//==========================================================================
// File operations
static int eventfd_fo_read      (struct CYG_FILE_TAG *fp, struct CYG_UIO_TAG *uio)
{
    eventfd_t count;
    cyg_iovec *iov = &uio->uio_iov[0];
    cyg_uint32 len = iov->iov_len;

    if (len < sizeof(eventfd_t))
        return (EINVAL);

    struct eventfd *efd = (struct eventfd *)fp->f_data;

    cyg_mutex_lock(&efd->mutex);
    if (efd->count == 0) {
        if (efd->flags & EFD_NONBLOCK) {
            cyg_mutex_unlock(&efd->mutex);
            return (EAGAIN);
        }
    }

    while (efd->count == 0) {
        efd->efd_nr_pend++;
        if ( cyg_cond_wait(&efd->r_pend) ==  false ) {
            cyg_mutex_unlock(&efd->mutex);
            return (EINTR);
        }
        efd->efd_nr_pend--;
    }

    if (efd->flags & EFD_SEMAPHORE) {
        efd->count--;
        count = 1;
    } else {
        count = efd->count;
        efd->count = 0;
    }
    /* wake any blocked write */
    if (efd->efd_nw_pend)
        cyg_cond_signal(&efd->w_pend);
    if (efd->flags & EFD_SEL_W_PENDING) {
        cyg_selwakeup(&efd->eventfd_sel_w);
        efd->flags &= ~EFD_SEL_W_PENDING;
    }
    cyg_mutex_unlock(&efd->mutex);

    memcpy(iov->iov_base, &count, sizeof(eventfd_t));
    uio->uio_resid -= sizeof(eventfd_t);

    return (0);
}

static int eventfd_fo_write     (struct CYG_FILE_TAG *fp, struct CYG_UIO_TAG *uio)
{
    eventfd_t count;
    cyg_iovec *iov = &uio->uio_iov[0];
    cyg_uint32 len = iov->iov_len;
    int overflow;

    if (len < sizeof(eventfd_t))
        return (EINVAL);

    memcpy(&count, iov->iov_base, sizeof(eventfd_t));
    if (count == (~0ULL))
        return (EINVAL);

    struct eventfd *efd = (struct eventfd *)fp->f_data;

    cyg_mutex_lock(&efd->mutex);
    overflow = ( (efd->count + count) == (~0ULL) ||
                 ((efd->count + count) < (efd->count)) );

    if ((efd->flags & EFD_NONBLOCK) && overflow) {
        cyg_mutex_unlock(&efd->mutex);
        return (EAGAIN);
    }

    while (overflow) {
        efd->efd_nw_pend++;
        if ( cyg_cond_wait(&efd->w_pend) ==  false ) {
            cyg_mutex_unlock(&efd->mutex);
            return (EINTR);
        }
        efd->efd_nw_pend--;
        overflow = ( (efd->count + count) == (~0ULL) ||
                     ((efd->count + count) < (efd->count)) );
    }

    efd->count += count;
    /* wake any blocked read */
    if (efd->efd_nr_pend)
        cyg_cond_signal(&efd->r_pend);
    if (efd->flags & EFD_SEL_R_PENDING) {
        cyg_selwakeup(&efd->eventfd_sel_r);
        efd->flags &= ~EFD_SEL_R_PENDING;
    }

    cyg_mutex_unlock(&efd->mutex);
    uio->uio_resid -= sizeof(eventfd_t);

    return (0);
}

static int eventfd_fo_select    (struct CYG_FILE_TAG *fp, int which, CYG_ADDRWORD info)
{
    struct eventfd *efd = (struct eventfd *)fp->f_data;
    cyg_mutex_lock(&efd->mutex);
    switch (which) {
        case CYG_FREAD:
            if (efd->count) {
                cyg_mutex_unlock(&efd->mutex);
                return (1);
            }
            cyg_selrecord(info, &efd->eventfd_sel_r);
            efd->flags |= EFD_SEL_R_PENDING;
            break;

        case CYG_FWRITE:
            if (efd->count < ((~0ULL)-1)) {
                cyg_mutex_unlock(&efd->mutex);
                return (1);
            }
            cyg_selrecord(info, &efd->eventfd_sel_w);
            efd->flags |= EFD_SEL_W_PENDING;
            break;
    }
    cyg_mutex_unlock(&efd->mutex);

    return ENOERR;
}

static int eventfd_fo_close     (struct CYG_FILE_TAG *fp)
{
    int error = 0;
    struct eventfd *efd;

    if (fp->f_data) {
        efd = (struct eventfd *)fp->f_data;
        cyg_cond_destroy(&efd->r_pend);
        cyg_cond_destroy(&efd->w_pend);
        cyg_mutex_destroy(&efd->mutex);
        free (efd);
    }

    return (error);
}

static int eventfd_fo_lseek   (struct CYG_FILE_TAG *fp, off_t *pos, int whence )
{
    // All current devices have no notion of position. Just return zero
    // as the new position.
    *pos = 0;

    return ENOERR;
}

static int eventfd_fo_ioctl   (struct CYG_FILE_TAG *fp, CYG_ADDRWORD cmd,
                               CYG_ADDRWORD data)
{
    int error = 0;
    struct eventfd *efd = (struct eventfd *)fp->f_data;

    cyg_mutex_lock(&efd->mutex);
    if (cmd == FIONBIO) {
        if (*(int *)data) {
            efd->flags |= EFD_NONBLOCK;
            fp->f_flag |= O_NONBLOCK;
        } else {
            efd->flags &= ~EFD_NONBLOCK;
            fp->f_flag &= ~O_NONBLOCK;
        }
    }
    else
        error = ENOSYS;
    cyg_mutex_unlock(&efd->mutex);

    return error;
}

static int eventfd_fo_fsync   (struct CYG_FILE_TAG *fp, int mode )
{
    return ENOERR;
}

static int eventfd_fo_fstat   (struct CYG_FILE_TAG *fp, struct stat *buf )
{
    // Just fill in the stat buffer with some constant values.
    buf->st_mode = __stat_mode_FIFO;

    buf->st_ino         = (ino_t)fp->f_data;    // map dev handle to inode
    buf->st_dev         = (dev_t)fp->f_data;    // same with dev id
    buf->st_nlink       = 0;
    buf->st_uid         = 0;
    buf->st_gid         = 0;
    buf->st_size        = 0;
    buf->st_atime       = 0;
    buf->st_mtime       = 0;
    buf->st_ctime       = 0;

    return ENOERR;
}

static int eventfd_fo_getinfo (struct CYG_FILE_TAG *fp, int key, void *buf, int len )
{
    return ENOSYS;
}

static int eventfd_fo_setinfo (struct CYG_FILE_TAG *fp, int key, void *buf, int len )
{
    return ENOSYS;
}

// -------------------------------------------------------------------------
// EOF eventfd.cxx
