//=============================================================================
//
//      ser33xxsio.c
//
//      Simple driver for the serial controllers on the Broadcom 33xx
//
//=============================================================================
//####ECOSGPLCOPYRIGHTBEGIN####
// -------------------------------------------
// This file is part of eCos, the Embedded Configurable Operating System.
// Copyright (C) 1998, 1999, 2000, 2001, 2002 Red Hat, Inc.
//
// eCos is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 2 or (at your option) any later version.
//
// eCos is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
// for more details.
//
// You should have received a copy of the GNU General Public License along
// with eCos; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. However the source code for this file must still be made available
// in accordance with section (3) of the GNU General Public License.
//
// This exception does not invalidate any other reasons why a work based on
// this file might be covered by the GNU General Public License.
//
// Alternative licenses for eCos may be arranged by contacting Red Hat, Inc.
// at http://sources.redhat.com/ecos/ecos-license/
// -------------------------------------------
//####ECOSGPLCOPYRIGHTEND####
//=============================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):   Mike Sieweke
// Contributors:Mike Sieweke
// Date:        2003-06-09
// Description: Simple driver for the Broadcom 33xx serial controller
//
//####DESCRIPTIONEND####
// ====================================================================
// Portions of this software Copyright (c) 2003-2010 Broadcom Corporation
// ====================================================================
//=============================================================================

#include <pkgconf/hal.h>
#include <pkgconf/system.h>
#include CYGBLD_HAL_PLATFORM_H

#include <cyg/hal/hal_arch.h>           // SAVE/RESTORE GP macros
#include <cyg/hal/hal_io.h>             // IO macros
#include <cyg/hal/hal_if.h>             // interface API
#include <cyg/hal/hal_intr.h>           // HAL_ENABLE/MASK/UNMASK_INTERRUPTS
#include <cyg/hal/hal_misc.h>           // Helper functions
#include <cyg/hal/drv_api.h>            // CYG_ISR_HANDLED
#include <cyg/hal/bcm33xx_regs.h>
#include <cyg/hal/debug_uart_sdw.h>
#include <cyg/io/serial.h>

extern void bcm33940_serial_debug_uart_init( uint32 baudDivider, DbgUartSdwRegistersType * UART );
extern void bcm33940_hal_serial_putc( volatile DbgUartSdwRegistersType *UART, unsigned char c  );
extern bool bcm33940_hal_serial_getc_nonblock(volatile DbgUartSdwRegistersType *UART, cyg_uint8* ch);
extern unsigned char bcm33940_hal_serial_getc(volatile DbgUartSdwRegistersType *UART);
extern unsigned char bcm33940_serial_getc(serial_channel *chan);

/* Register Macros to handle shadow register writing for
   bitfields.  This macro handles read modify writes.    */
#define ReadModWrField(reg, type, field, value) \
{                                               \
	type local_##type;				   	        \
	local_##type.Reg32 = reg.Reg32;             \
	local_##type.Bits.field = value;            \
	reg.Reg32 = local_##type.Reg32;             \
}

#define CHECK_NULL_CHANNEL_DATA  do {                                   \
                                        if( __ch_data == NULL)          \
                                        {                               \
                                            __ch_data = (void*)UART;    \
                                        }                               \
                                    }while(0)
//-----------------------------------------------------------------------------
typedef struct {
    cyg_uint8 *base;
    cyg_int32  msec_timeout;
    int        isr_vector;
} channel_data_t;

extern cyg_uint8 gSerialPortEnabled0;
extern cyg_uint8 gIkosFlag;

//-----------------------------------------------------------------------------
// The minimal init, get and put functions. All by polling.

void cyg_hal_plf_serial_33940_init_channel(void* __ch_data)
{
    static bool            doneOnce = 0;

    // Only initialize the serial device once.  If we do it again, it sends
    // garbage characters to the port.
    if (doneOnce == 1)
    {
        return;
    }

    cyg_uint32  baud_divisor = 0x2c; /* default 115 */

    CHECK_NULL_CHANNEL_DATA;
    bcm33940_serial_debug_uart_init(baud_divisor, (DbgUartSdwRegistersType *) __ch_data);

    doneOnce = 1;
}

void cyg_hal_plf_serial_33940_putc(void* __ch_data, cyg_uint8 __ch)
{
    if ( !gSerialPortEnabled0 )
    {
        return;
    }

    CHECK_NULL_CHANNEL_DATA;
    bcm33940_hal_serial_putc((DbgUartSdwRegistersType *) __ch_data, __ch);

    return;

}

cyg_bool cyg_hal_plf_serial_33940_getc_nonblock(void* __ch_data, cyg_uint8* ch)
{
    if ( !gSerialPortEnabled0 )
    {
        *ch = 0;
        return true;
    }

    CHECK_NULL_CHANNEL_DATA;
    bcm33940_hal_serial_getc_nonblock((DbgUartSdwRegistersType *) __ch_data,ch);
}

cyg_uint8 cyg_hal_plf_serial_33940_getc(void* __ch_data)
{
    cyg_uint8 ch;

    if ( !gSerialPortEnabled0 )
    {
        return 0;
    }

    CHECK_NULL_CHANNEL_DATA;
    return bcm33940_hal_serial_getc((DbgUartSdwRegistersType *) __ch_data);
}

static void cyg_hal_plf_serial_33940_write(void* __ch_data, const cyg_uint8* __buf,
                         cyg_uint32 __len)
{
    if ( !gSerialPortEnabled0 )
    {
        return;
    }

    CHECK_NULL_CHANNEL_DATA;
    while(__len-- > 0)
        bcm33940_hal_serial_putc((DbgUartSdwRegistersType *)__ch_data, *__buf++);

}

static void cyg_hal_plf_serial_33940_read(void* __ch_data, cyg_uint8* __buf, cyg_uint32 __len)
{
    if ( !gSerialPortEnabled0 )
    {
        return;
    }

    CHECK_NULL_CHANNEL_DATA;
    while(__len-- > 0)
        *__buf++ = bcm33940_hal_serial_getc((DbgUartSdwRegistersType *) __ch_data);
}


cyg_bool
cyg_hal_plf_serial_33940_getc_timeout(void* __ch_data, cyg_uint8* ch)
{
    int             delay_count;
    channel_data_t *chan;
    cyg_bool        charFound;

    if ( !gSerialPortEnabled0 )
    {
        return false;
    }

    chan = (channel_data_t*)__ch_data;

    delay_count = chan->msec_timeout * 10; // delay in .1 ms steps

    for(;;) {
        charFound = cyg_hal_plf_serial_33940_getc_nonblock((DbgUartSdwRegistersType *) UART, ch);
        if (charFound || ( delay_count-- == 0 ))
            break;
        CYGACC_CALL_IF_DELAY_US(100);
    }

    return charFound;
}

extern cyg_uint32 PeripheralClockFrequency;

static int
cyg_hal_plf_serial_33940_control(void *__ch_data, __comm_control_cmd_t __func, ...)
{
    channel_data_t      *chan;
    DbgUartSdwIerType localUartSdwIerType;
    int                  returnValue        = 0;

    if ( !gSerialPortEnabled0 )
    {
        return 0;
    }

    chan = (channel_data_t*)__ch_data;

    switch (__func) {
        case __COMMCTL_IRQ_ENABLE:
            // Enable interrupts in the UART and interrupt controller.
            // jb-uart
            // interrupt control is done thru bcm33xx driver
            break;
        case __COMMCTL_IRQ_DISABLE:
            // jb-uart
            // interrupt control is done thru bcm33xx driver
            break;
        case __COMMCTL_DBG_ISR_VECTOR:
            returnValue = chan->isr_vector;
            break;
        case __COMMCTL_SET_TIMEOUT:
        {
            va_list ap;

            returnValue = chan->msec_timeout;

            va_start(ap, __func);
            chan->msec_timeout = va_arg(ap, cyg_uint32);
            va_end(ap);
            break;
        }
        case __COMMCTL_SETBAUD:
        {
            // jb-uart
            // baud changes are done thru bcm33xx driver
            break;
        }

        case __COMMCTL_GETBAUD:
            break;
        default:
            break;
    }
    return returnValue;
}

static int
cyg_hal_plf_serial_33940_isr(void *__ch_data, int* __ctrlc,
                       CYG_ADDRWORD __vector, CYG_ADDRWORD __data)
{
    int             returnVal   = 0;
    DbgUartSdwIirType localUartSdwIirType;
    cyg_uint8       c;

    // isr unused
    // refer to bcm33940_serial_ISR in bcm33940_serial.c for ISR function
    return returnVal;
}

void cyg_hal_plf_serial_33940_init(void)
{
    hal_virtual_comm_table_t *comm;
    int                       currentID;

    currentID = CYGACC_CALL_IF_SET_CONSOLE_COMM(CYGNUM_CALL_IF_SET_COMM_ID_QUERY_CURRENT);

    // Init channels
    cyg_hal_plf_serial_33940_init_channel((void*)UART);

    // Setup procs in the vector table

    // Set channel 0
    CYGACC_CALL_IF_SET_CONSOLE_COMM(0);
    comm = CYGACC_CALL_IF_CONSOLE_PROCS();
    CYGACC_COMM_IF_CH_DATA_SET(     *comm, UART);
    CYGACC_COMM_IF_WRITE_SET(       *comm, cyg_hal_plf_serial_33940_write);
    CYGACC_COMM_IF_READ_SET(        *comm, cyg_hal_plf_serial_33940_read);
    CYGACC_COMM_IF_PUTC_SET(        *comm, cyg_hal_plf_serial_33940_putc);
    CYGACC_COMM_IF_GETC_SET(        *comm, cyg_hal_plf_serial_33940_getc);
    CYGACC_COMM_IF_CONTROL_SET(     *comm, cyg_hal_plf_serial_33940_control);
    CYGACC_COMM_IF_DBG_ISR_SET(     *comm, cyg_hal_plf_serial_33940_isr);
    CYGACC_COMM_IF_GETC_TIMEOUT_SET(*comm, cyg_hal_plf_serial_33940_getc_timeout);

}

//-----------------------------------------------------------------------------
// end of ser33940.c
