//==========================================================================
//
//      bcm33940_serial.c
//
//      Serial device driver for Broadcom 33940 reference platform on-chip serial devices
//
//==========================================================================
//####ECOSGPLCOPYRIGHTBEGIN####
// -------------------------------------------
// This file is part of eCos, the Embedded Configurable Operating System.
// Copyright (C) 1998, 1999, 2000, 2001, 2002 Red Hat, Inc.
//
// eCos is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free
// Software Foundation; either version 2 or (at your option) any later version.
//
// eCos is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
// for more details.
//
// You should have received a copy of the GNU General Public License along
// with eCos; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
//
// As a special exception, if other files instantiate templates or use macros
// or inline functions from this file, or you compile this file and link it
// with other works to produce a work based on this file, this file does not
// by itself cause the resulting work to be covered by the GNU General Public
// License. However the source code for this file must still be made available
// in accordance with section (3) of the GNU General Public License.
//
// This exception does not invalidate any other reasons why a work based on
// this file might be covered by the GNU General Public License.
//
// Alternative licenses for eCos may be arranged by contacting Red Hat, Inc.
// at http://sources.redhat.com/ecos/ecos-license/
// -------------------------------------------
//####ECOSGPLCOPYRIGHTEND####
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    jbarroso
// Contributors: gthomas, jskov, dmoseley, tmichals, msieweke
// Date:         2003-07-02
// Purpose:      Broadcom 33940 reference platform serial device driver
// Description:  Broadcom 33940 serial device driver
//
//
//####DESCRIPTIONEND####
//
//==========================================================================
// =========================================================================
// Portions of this software Copyright (c) 2003-2010 Broadcom Corporation
// =========================================================================

#include <pkgconf/io_serial.h>
#include <pkgconf/io.h>

#include <cyg/io/io.h>
#include <cyg/hal/hal_intr.h>
#include <cyg/io/devtab.h>
#include <cyg/infra/diag.h>
#include <cyg/io/serial.h>
#include <cyg/hal/bcm33xx_regs.h>
#include <cyg/hal/debug_uart_sdw.h>

#ifdef CYGPKG_IO_SERIAL_MIPS_BCM33XX

#include "bcm33xx_serial.h"
#include "bcm33940_serial.h"

/* Register Macros to handle shadow register writing for
   bitfields.  This macro handles read modify writes.    */
#define ReadModWrField(reg, type, field, value) \
{                                               \
	type local_##type;				   	        \
	local_##type.Reg32 = reg.Reg32;             \
	local_##type.Bits.field = value;            \
	reg.Reg32 = local_##type.Reg32;             \
}

static uint32 cmd_read_C0_COUNT(void)
{
	uint32 countValue;
	__asm volatile("mfc0    %0, $9" : "=r"(countValue));
	return countValue;
}

cyg_uint8 gIkosFlag  __attribute__ ((weak)) = 0; /* control IKOS specific logic from app code */
extern CYG_WORD uart_irq_bits[];

#define UART_SERIAL_CLOCK_FREQUENCY (81000000) /* 81MHz */

//-----------------------------------------------------------------------------
// The minimal init, get and put functions. All by polling.

/* This force read function ensures that a register read is not optmized away
   by the compiler when the return result is not being used */
static uint32 _forceRd(volatile uint32 *addr)
{
    return *addr;
}

/* DBG  UPG UART init code */
void bcm33940_serial_debug_uart_init( uint32 baudDivider, volatile DbgUartSdwRegistersType * UART )
{
    static uint8 done = 0;

    if(done)
    {
        return;
    }
    done = 1;

    if(gIkosFlag)
    {
        baudDivider = 1;
    }

    DbgUartSdwLcrType localUartSdwLcrType;
    DbgUartSdwFcrType localUartSdwFcrType;
    DbgUartSdwMcrType localUartSdwMcrType;
    DbgUartSdwIerType localUartSdwIerType;
    DbgUartSdwIerType localUartSdwDlhType;

    // Need to open the divisor latch to set the baud rate
    ReadModWrField(UART->Lcr, DbgUartSdwLcrType, Dlab, 1);
    // Set the baud rate with setting divisor latch high and low registers
    // baud rate = (serial clock freq) / (16 * divisor)
    // baud rate = 81Mhz / 704 (16 * 44) = 115,056 (within 2%)
    ReadModWrField(UART->Rbr_Thr_Dll, DbgUartSdwDllType, Dll, baudDivider);

    // After setting the baud rate, we need to wait at least 8 81Mhz clock
    // cycles before reading/writing characters.
    // MIPS freq(800MHz) is ~10 times the UART (80MHz) freq.  Use Cp0 counter to wait
    unsigned long startTicks = cmd_read_C0_COUNT();
    volatile unsigned long currentTicks = cmd_read_C0_COUNT();

    // cp0 count increases every other tick.  So waiting 40 ticks should
    // be enough.  Double the wait just in case
    // rollover - doing the substraction takes care of rollover events
    while( (currentTicks-startTicks) < 80 )
    {
        currentTicks = cmd_read_C0_COUNT();
    }

    // Setup 8 bits, no parity, and 1 stop bit (8-N-1)
    localUartSdwLcrType.Reg32       = UART->Lcr.Reg32;
    localUartSdwLcrType.Bits.Dlab   = 0;   // Close the divisor latch
    localUartSdwLcrType.Bits.Bc     = 0;   // Break Control Bit
    localUartSdwLcrType.Bits.Eps    = 0;   // Even Parity Select (1=even parity, 0=odd)
    localUartSdwLcrType.Bits.Pen    = 0;   // Parity Enable (0=no parity)
    localUartSdwLcrType.Bits.Stop   = 0;   // Number of stop bits:0=1 stop bit,1=1.5 stop bits
                                           // when DLS (LCR[1:0]) is zero, else 2 stop bit
    localUartSdwLcrType.Bits.Dls    = 0x3; // Data Length:00=5 bits,01=6 bits,10=7 bits,11=8 bits
    UART->Lcr.Reg32                 = localUartSdwLcrType.Reg32;

    // Setup RX/TX FIFO triggers and FIFO enable
    localUartSdwFcrType.Reg32       = 0;   // FCR is write only, don't read current value before setting
    localUartSdwFcrType.Bits.Rt     = 1;   // RCVR Trigger:00=1 char in FIFO,01=FIFO 1/4 full,
                                           // 10=FIFO 1/2 full,11=FIFO 2 less than fullk
                                           // = 1: 8 characters in FIFO
    localUartSdwFcrType.Bits.Tet    = 0;   // TX Empty Trigger:00=FIFO empty,01=2 chars in the FIFO,
                                           // 10=FIFO 1/4 full,11=FIFO 1/2 full
                                           // = 1: 2 characters in FIFO
    localUartSdwFcrType.Bits.Dmam   = 0;   // DMA signaling mode:0=mode 0, 1=mode 1
    localUartSdwFcrType.Bits.Xfifor = 1;   // XMIT FIFO Reset:Resets the control portion of the
                                           // transmit FIFO and treats FIFO as empty
    localUartSdwFcrType.Bits.Rfifor = 1;   // RCVR FIFO Reset:Resets the control portion of the
                                           // receive FIFO and treats the FIFO as empty
    localUartSdwFcrType.Bits.Fifoe  = 1;   // FIFO Enable:enables/disables the transmit (XMIT)
                                           // and receive (RCVR) FIFOs
    UART->Fcr.Reg32                 = localUartSdwFcrType.Reg32;

    // Enable DTR, RTS, and disable flow control
    localUartSdwMcrType.Reg32       = UART->Mcr.Reg32;
    localUartSdwMcrType.Bits.Afce   = 0;   // Auto Flow Control Enable:0=disabled,1=enabled
    localUartSdwMcrType.Bits.Rts    = 0;   // Request to Send
    localUartSdwMcrType.Bits.Dtr    = 0;   // Data Terminal Ready
    UART->Mcr.Reg32                 = localUartSdwMcrType.Reg32;

    // flush status registers
    // Read the status registers to flush them
    _forceRd( (volatile uint32 *)&UART->Lsr.Reg32 );
    _forceRd( (volatile uint32 *)&UART->Msr.Reg32 );

    // read Rbr to clear any character timeout errors
    _forceRd( (volatile uint32 *)&UART->Rbr_Thr_Dll.Reg32 );

    // Setup Interrupt Enable
    localUartSdwIerType.Reg32       = UART->Dlh_Ier.Reg32;
    localUartSdwIerType.Bits.Ptime  = 1;   // Programmable THRE Interrupt Mode Enable
    localUartSdwIerType.Bits.Edssi  = 0;   // Enable Modem Status Interrupt
    localUartSdwIerType.Bits.Elsi   = 0;   // Enable Receiver Line Status Interrupt
    localUartSdwIerType.Bits.Etbei  = 0;   // Enable Transmit Holding Register Empty Interrupt
                                           // Etbei will be enabled by start_xmit
    localUartSdwIerType.Bits.Erbfi  = 1;   // Enable Received Data Available Interrupt
    UART->Dlh_Ier.Reg32             = localUartSdwIerType.Reg32;

    /* read IIR register and warn about any outstanding interrupts */
    DbgUartSdwIirType iir;
    iir.Reg32 = UART->Fcr.Reg32;  /* Iir is RO if Fcr */
    if( iir.Bits.Iid != 1)
    {
        diag_printf(" WARNING: Iid not 0 at init - 0x%x",iir.Bits.Iid);
    }
}

// Internal function to actually configure the hardware to desired baud rate, etc.
bool bcm33940_serial_config_port(serial_channel    *chan,
                           cyg_serial_info_t *new_config,
                           bool               initialize)
{
    bcm33xx_serial_info *bcm33940_chan   = (bcm33xx_serial_info *) chan->dev_priv;
    int                  baud_rate;
    cyg_uint32           baud_divisor;

#ifdef CYGDBG_IO_INIT
    diag_printf("BCM UPG SERIAL config - UART = %08x, chan = %08x\n\n",(cyg_uint32)bcm33940_chan->base,(cyg_uint32)chan);
#endif

    // Translate baud rate from an enum to an integer.
    baud_rate = select_baud[new_config->baud];
    if (baud_rate == 0)
        return false;    // Invalid baud rate selected

    //diag_printf("baud_rate = %d\n\n",baud_rate);
    // Calculate baudword by dividing serial frequency
    // baud rate = (serial clock freq) / (16 * divisor)
    // Ex: 115200: baud rate = 81Mhz / (16 * 44) = 115,056 (within 2%)
    // baud_divisor = (serial clock freq)/(16(baud rate))
    // this formula rounds up the result:
    baud_divisor = ((UART_SERIAL_CLOCK_FREQUENCY + (baud_rate*16) - 1 )/(baud_rate*16));

    //diag_printf("baud divisor = %d 0x%x\n\n",baud_divisor,baud_divisor);
    bcm33940_serial_debug_uart_init(baud_divisor,(DbgUartSdwRegistersType*)bcm33940_chan->base);

    if(initialize)
    {
        if (chan->out_cbuf.len != 0)
        {
            // Enable general UART interrupt.
            hal_enable_periph_interrupt( uart_irq_bits[bcm33940_chan->channelNumber] );
        }
    }
    if (new_config != &chan->config) {
        chan->config = *new_config;
    }

    return true;
}

// UART non-blocking HW writes
bool bcm33940_hal_serial_putc_non_blocking( volatile DbgUartSdwRegistersType *UART, unsigned char c  )
{
    // Local copy guarantees 32-bit aligned accesses
    // return false is buffer is full
    DbgUartSdwLsrType localDbgUartLsrThrType;
    DbgUartSdwThrType localDbgUartSdwThrType;
    localDbgUartLsrThrType.Reg32 = UART->Lsr.Reg32;

    // check for fifo full
    // this assumes Ptime is set to 1
    if(localDbgUartLsrThrType.Bits.Thre == 1)
    {
        return false;
    }

    // write Thr
    localDbgUartSdwThrType.Bits.Thr = c;
    UART->Rbr_Thr_Dll.Reg32 = localDbgUartSdwThrType.Reg32;

    return true;
}

// UART HW writes
void bcm33940_hal_serial_putc( volatile DbgUartSdwRegistersType *UART, unsigned char c  )
{
    // Local copy guarantees 32-bit aligned accesses
    // Delay writes if the buffer is full
    DbgUartSdwLsrType localDbgUartLsrThrType;
    DbgUartSdwIerType localUartSdwIerType;
    DbgUartSdwThrType localDbgUartSdwThrType;
    localDbgUartLsrThrType.Reg32 = UART->Lsr.Reg32;

    // check for fifo full
    // this assumes Ptime is set to 1
    while (localDbgUartLsrThrType.Bits.Thre == 1)
    {
        localDbgUartLsrThrType.Reg32 = UART->Lsr.Reg32;
    }

    // write Thr
    localDbgUartSdwThrType.Bits.Thr = c;
    UART->Rbr_Thr_Dll.Reg32 = localDbgUartSdwThrType.Reg32;

    return;
}

// Send a character to the device output buffer.
// Return 'true' if character is sent to device
bool bcm33940_serial_putc(serial_channel *chan,
                    unsigned char   c)
{

    bcm33xx_serial_info *bcm33940_chan = (bcm33xx_serial_info *) chan->dev_priv;

    bcm33940_hal_serial_putc(( DbgUartSdwRegistersType *) bcm33940_chan->base, c );

    return true;
}

// Send a character to the device output buffer if space is available.
// Return 'true' if character is sent to device
bool bcm33940_serial_putc_non_blocking(serial_channel *chan,
                    unsigned char   c)
{

    bcm33xx_serial_info *bcm33940_chan = (bcm33xx_serial_info *) chan->dev_priv;

    return bcm33940_hal_serial_putc_non_blocking(( DbgUartSdwRegistersType *) bcm33940_chan->base, c );
}

bool bcm33940_hal_serial_getc_nonblock(volatile DbgUartSdwRegistersType *UART, cyg_uint8* ch)
{
    DbgUartSdwLsrType localDbgUartLsrThrType;
    DbgUartSdwRbrType localUartSdwRbrType;

    localDbgUartLsrThrType.Reg32 = UART->Lsr.Reg32;
    if ( localDbgUartLsrThrType.Bits.Dr == 0)
        return false;

    localUartSdwRbrType.Reg32 = UART->Rbr_Thr_Dll.Reg32;
    *ch = localUartSdwRbrType.Bits.Rbr;

    return true;
}

unsigned char bcm33940_hal_serial_getc(volatile DbgUartSdwRegistersType *UART)
{
    cyg_uint8 ch;

    while(!bcm33940_hal_serial_getc_nonblock(UART, &ch));

    return ch;
}

// Fetch a character from the device input buffer - non blocking
unsigned char bcm33940_serial_getc(serial_channel *chan)
{

    bcm33xx_serial_info *bcm33940_chan = (bcm33xx_serial_info *) chan->dev_priv;
    unsigned char ch = 0;

    //while(!bcm33940_hal_serial_getc_nonblock(( DbgUartSdwRegistersType *) bcm33940_chan->base, &ch));
    bcm33940_hal_serial_getc_nonblock(( DbgUartSdwRegistersType *) bcm33940_chan->base, &ch);

    return ch;

}

// Enable the transmitter on the device
void bcm33940_serial_start_xmit(serial_channel *chan)
{
    bcm33xx_serial_info *bcm33940_chan = (bcm33xx_serial_info *) chan->dev_priv;
    DbgUartSdwRegistersType * UART = (DbgUartSdwRegistersType *) bcm33940_chan->base;
    DbgUartSdwIerType localUartSdwIerType;

    // enable txmit interrupt
    localUartSdwIerType.Reg32       = UART->Dlh_Ier.Reg32;
    localUartSdwIerType.Bits.Etbei  = 1;   // Enable Transmit Holding Register Empty Interrupt
    UART->Dlh_Ier.Reg32             = localUartSdwIerType.Reg32;
}

// Disable the transmitter on the device
void bcm33940_serial_stop_xmit(serial_channel *chan)
{
    bcm33xx_serial_info *bcm33940_chan = (bcm33xx_serial_info *) chan->dev_priv;

    volatile DbgUartSdwRegistersType * UART = (DbgUartSdwRegistersType *) bcm33940_chan->base;
    DbgUartSdwIerType localUartSdwIerType;

    // disable txmit interrupt
    localUartSdwIerType.Reg32       = UART->Dlh_Ier.Reg32;
    localUartSdwIerType.Bits.Etbei  = 0;   // Transmit Holding Register Empty Interrupt
    UART->Dlh_Ier.Reg32             = localUartSdwIerType.Reg32;
}

#define IIR_IID_MODEM_STATUS        (0)
#define IIR_IID_NO_INTERRUPT        (1)
#define IIR_IID_THR_EMPTY           (2)
#define IIR_IID_RCV_DATA_AVAIL      (4)
#define IIR_IID_RCV_LINE_STATUS     (6)
#define IIR_IID_CHARACTER_TIMEOUT (0xc)

// Serial I/O - low level interrupt handler (ISR)
void bcm33940_serial_ISR( void *data )
{
    serial_channel      *chan                = (serial_channel *)data;
    bcm33xx_serial_info *bcm33940_chan        = (bcm33xx_serial_info *)chan->dev_priv;
    volatile  DbgUartSdwRegistersType * UART = ( DbgUartSdwRegistersType *)bcm33940_chan->base;
    DbgUartSdwLsrType localDbgUartLsrThrType;
    //uint32* dbg = (uint32*)0xec20471c;
    DbgUartSdwRbrType localUartSdwRbrType;
    cyg_uint8 ch;

    DbgUartSdwIirType iir;
    char * error=NULL;

    iir.Reg32 = UART->Fcr.Reg32;  /* Iir is RO if Fcr */

    //*dbg = 0xEA000000 | iir.Reg32;

    /* check for Break indication */
    localDbgUartLsrThrType.Reg32 = UART->Lsr.Reg32;
    if(localDbgUartLsrThrType.Bits.Bi)
    {
        // If there's a break callout function for this channel, call it.
        if ( bcm33940_chan->break_callout != NULL )
        {
            (bcm33940_chan->break_callout)();
        }
    }

    switch (iir.Bits.Iid)
    {
        /* no interrupt */
        case IIR_IID_NO_INTERRUPT:
        break;

        /* Receive line status */
        case  IIR_IID_RCV_LINE_STATUS:
            error = "overrun/parity/framing";
            _forceRd( (volatile uint32 *)&UART->Lsr.Reg32 );
        break;

        /* Received data available */
        case  IIR_IID_RCV_DATA_AVAIL:
        /* Character timeout */
        case IIR_IID_CHARACTER_TIMEOUT:
            localDbgUartLsrThrType.Reg32 = UART->Lsr.Reg32;

            /* if data ready is not set, read Thr register once
               to clear the interrupt */
            if(localDbgUartLsrThrType.Bits.Dr == 0)
            {
                error = "character timeout";
                _forceRd( (volatile uint32 *)&UART->Rbr_Thr_Dll.Reg32 );
            }
            else
            {
                /* read characters until data ready clears */
                while( localDbgUartLsrThrType.Bits.Dr == 1)
                {
                    /* new character available */
                    /* 32 bit read from register */
                    localUartSdwRbrType.Reg32 = UART->Rbr_Thr_Dll.Reg32;
                    /* save new character and call the callback */
                    ch = localUartSdwRbrType.Bits.Rbr;
                    (chan->callbacks->rcv_char)(chan, ch);

                    /* re-read Lsr register */
                    localDbgUartLsrThrType.Reg32 = UART->Lsr.Reg32;
                }
            }
        break;

        case IIR_IID_MODEM_STATUS:
            _forceRd( (volatile uint32 *)&UART->Msr.Reg32 );
        break;

        case IIR_IID_THR_EMPTY:
                (chan->callbacks->xmt_char)(chan);
        break;

        default:
            error = "unknown source";
        break;
    }

    if(error != NULL)
    {
        diag_printf("cliIsr: error %s\n",error);
    }

    //diag_printf("cliIsr done 0x%x 0x%x\n",iir.Bits.Iid, UART->Fcr.Reg32);
    // The interrupt was disabled by the upper ISR.  Need to re-enable.
    hal_enable_periph_interrupt( uart_irq_bits[bcm33940_chan->channelNumber] );
}

#endif

//-------------------------------------------------------------------------
// EOF bcm33xx_serial.c
