#!/bin/bash

#
# As CM Implemented a framework to report security violations and events of interest using BfcEventLog,
# this script is used to monitor the remote connections (ssh, telent). If connection increases, it will
# report security event log via Lattice API to CM.
#
# Author:         Chris Zacker
# Creation Date:  Jan 29, 2019
#

# To prevent report to many event logs
maxLogEventCount=3

# Loop timeout, by seconds
timeout=60
# Once event log is reported, wait timeout to send a new one, by seconds
reportTimeout=600
# After reported maxLogEventCount events, freeze until timeout
freezeReport=false
freezeReportTimeout=0
# EventId sent by RG
systemSecurityEventId="1.3.6.1.4.1.4413.2.2.2.1.9.1.3.1.1.5"
loggedEvent=0


echo "Start monitoring Remote Connections"

# monitoring loop
while [ true ]
do

telnetSessions=""
sshSessions=""

  sleep $timeout

  # check if report is in freeze
  if [ $freezeReport ] && [ $freezeReportTimeout -gt 0 ]; then
    freezeReportTimeout=`expr $freezeReportTimeout - $timeout`
    if [ $freezeReportTimeout -le 0 ]; then
      freezeReport=false
      freezeReportTimeout=0
    fi
    continue
  fi

  netstatOutput=$(netstat -a -t | grep telnet | grep ESTABLISHED)

  while read line ; do
    telnetSessions=$telnetSessions"_"$(echo $line | awk 'END {print$5}')
  done <<< "$(echo -e "$netstatOutput")"

  netstatOutput=$(netstat -a -t | grep ssh | grep ESTABLISHED)

  while read line ; do
    sshSessions=$sshSessions"_"$(echo $line | awk 'END {print$5}')
  done <<< "$(echo -e "$netstatOutput")"

  if [ $sshSessions != "_" ] || [ $telnetSessions != "_" ]; then
    loggedEvent=`expr $loggedEvent + 1`

    # prevent report too many log events to CM
    if [ $loggedEvent -le $maxLogEventCount ]; then
      # report event
      systemSecurityEventText="Abnormal"

      if [ $sshSessions != "_" ]; then
      systemSecurityEventText=$systemSecurityEventText"_SSH_Connections_from:_"$sshSessions
      fi
      if [ $telnetSessions != "_" ]; then
      systemSecurityEventText=$systemSecurityEventText"_***_Telnet_Connections_from:_"$telnetSessions
      fi

      latticecli -n "set Cm.LogEventId "$systemSecurityEventId
      latticecli -n "set Cm.LogEventText "$systemSecurityEventText
    else
      # report maximum events, freeze reporting, reset loggedEvent
      freezeReport=true
      # set freeze report timeout to 1 day ( 60 * 60 * 24 )
      freezeReportTimeout=`expr $timeout \* 1440`
      loggedEvent=0
    fi

    sleep $reportTimeout
  fi
done
