#!/bin/bash
#
# As CM Implemented a framework to report security violations and events of interest using BfcEventLog,
# this script is used to monitor the connection tracker. If sessions exceeds a certain count, it will
# report security event log via Lattice API to CM.
#
# Author:         Andy Yin
# Creation Date:  Dec 12, 2018
#

# Connction track table sessions log event report threshold
# Linx conntrack table size can get by "cat /proc/sys/net/nf_conntrack_max"
sessionThreshold=5000
sessionIncrement=1000
lastsessionValue=0
fastsessionThreshold=0
# Loop timeout, by seconds
timeout=300
# Once event log is reported, wait timeout to send a new one, by seconds
reportTimeoutNormal=300
reportTimeoutFast=60
# EventId sent by RG
systemSecurityEventId="1.3.6.1.4.1.4413.2.2.2.1.9.1.3.1.1.5"
loggedEvent=0

# check if nf_conntrack_max exist, if yes, set sessionThreshold = nf_conntrack_max * 80%
if [ -f "/proc/sys/net/nf_conntrack_max" ]; then
  conntrack_max=$(cat /proc/sys/net/nf_conntrack_max)
  sessionThreshold=$(expr $conntrack_max \* 8 / 10)
fi

echo "Start monitoring connection track sessions, set threshold to "$sessionThreshold" ..."

# monitoring loop
while [ true ]
do
  sleep $timeout
  sessions=0


  # check if conntrack file is exist or not. nf_conntrack replaces ip_conntrack after kernel 2.6.15
  if [ -f "/proc/net/nf_conntrack" ]; then
    sessions=$(cat /proc/sys/net/netfilter/nf_conntrack_count)
  elif [ -f "/proc/net/ip_conntrack" ]; then
    sessions=$(awk '{print NR}' /proc/net/ip_conntrack|tail -n1)
  else
    echo "Linux nf_conntrack or ip_conntrack is not exist, exist ..."
    break
  fi

  if [ ! $sessions ]; then
    continue
  fi

  # check if session entries reach the preset threshold
  if [ $sessions -gt $sessionThreshold ]; then
    # report event
    if [ $timeout -eq $reportTimeoutNormal ]; then
       echo "Connection tracker sessions are "$sessions", exceeds "$sessionThreshold" sessions, trigger event log..."
       systemSecurityEventText="Connection_tracker_sessions_are_"$sessions",exceeds_"$sessionThreshold"_sessions!"

       latticecli -n "set Cm.LogEventId "$systemSecurityEventId
       latticecli -n "set Cm.LogEventText "$systemSecurityEventText
       timeout=$reportTimeoutFast
    else
       if [ $sessions -gt $fastsessionThreshold ]; then
          echo "Connection tracker sessions are "$sessions", exceeds "$sessionThreshold" sessions, trigger event log..."
          systemSecurityEventText="Connection_tracker_sessions_are_"$sessions",exceeds_"$sessionThreshold"_sessions!"

          latticecli -n "set Cm.LogEventId "$systemSecurityEventId
          latticecli -n "set Cm.LogEventText "$systemSecurityEventText
       fi
    fi
    lastsessionValue=$sessions;
    fastsessionThreshold=$(( $lastsessionValue + $sessionIncrement ))
  else
    # if we were in FAST reporting interval then we need to send event saying we are back below
    # our normal threshold
    if [ $timeout -eq $reportTimeoutFast ]; then
       echo "Connection tracker sessions are "$sessions", below "$sessionThreshold" sessions, trigger event log..."
       systemSecurityEventText="Connection_tracker_sessions_are_"$sessions",below_"$sessionThreshold"_sessions!"

       latticecli -n "set Cm.LogEventId "$systemSecurityEventId
       latticecli -n "set Cm.LogEventText "$systemSecurityEventText
    fi
    fastsessionThreshold=0
    timeout=$reportTimeoutNormal
  fi
done
