#!/bin/bash

#
# As CM Implemented a framework to report security violations and events of interest using BfcEventLog,
# this script is used to monitor Denial of Service attacks.
#
# Author:         Chris Zacker
# Creation Date:  Feb 4, 2019
#

# To prevent report to many event logs
maxLogEventCount=3


# We can get the # of times a dqm queue is FULL using dqm. this would indicate that we
# are dropping packets and that a denial of service attack may be occurring.  We check
# both the WAN side and LAN side.
# Also we can check CPU load using mpstat. If idle drops below 10%
# we know cpu is being bogged down

minimumIdlePercent=10
savedWanDQMnumfull=0
savedLanDQMnumfull=0

# If we see this number of queue fulls in the time between checking
dqmFullTrigger=100
dqmFullTriggerGfap=10

# Loop timeout, by seconds
timeout=60
# Once event log is reported, wait timeout to send a new one, by seconds
reportTimeout=600
# After reported maxLogEventCount events, freeze until timeout
freezeReport=false
freezeReportTimeout=0
# EventId sent by RG
systemSecurityEventId="1.3.6.1.4.1.4413.2.2.2.1.9.1.3.1.1.5"
loggedEvent=0
devicefound=0

echo "Start monitoring Denial of Service "

#test to see if we are a 3384

devicecheck=$(cat /proc/driver/dqm/mib | grep "gfap")
if [ "$devicecheck" == "DEVICE: gfap" ]; then
   devicefound=1;
fi

oneExceptionQueue=$(cat /proc/driver/dqm/mib | grep "RG Exception WAN")


# monitoring loop
while [ true ]
do

  sleep $timeout

  # check if report is in freeze
  if [ $freezeReport ] && [ $freezeReportTimeout -gt 0 ]; then
    freezeReportTimeout=`expr $freezeReportTimeout - $timeout`
    if [ $freezeReportTimeout -le 0 ]; then
      freezeReport=false
      freezeReportTimeout=0
    fi
    continue
  fi


  percentIdle=$(mpstat | grep all | awk 'END {print$11}' | grep -o -E '[0-9]+'  | head -1 )

  # check for 3384 device (we cant use the queues for the 3384)
  if [ $devicefound -eq 1 ]; then
     wanDQMnumfull=$(cat /proc/driver/dqm/mib | grep -A100 "DEVICE: gfap" | grep "\[23\]" | awk 'END {print$7}' | grep -o -E '[0-9]+')
     lanDQMnumfull=$wanDQMnumfull
     wanDQMTrigger=$(($savedWanDQMnumfull + $dqmFullTriggerGfap))
     lanDQMTrigger=$(($savedLanDQMnumfull + $dqmFullTriggerGfap))
     if [ $percentIdle -le $minimumIdlePercent ] || [ $wanDQMnumfull -gt $wanDQMTrigger ] || [ $lanDQMnumfull -gt $lanDQMTrigger ]; then
       loggedEvent=`expr $loggedEvent + 1`

       # prevent report too many log events to CM
       if [ $loggedEvent -le $maxLogEventCount ]; then
         # report event
         systemSecurityEventText="Abnormal_DoS_Attack:_IdlePercent="$percentIdle"_WAN_DQM_FULL="$wanDQMnumfull"_LAN_DQM_FULL="$lanDQMnumfull

         latticecli -n "set Cm.LogEventId "$systemSecurityEventId
         latticecli -n "set Cm.LogEventText "$systemSecurityEventText
       else
         # report maximum events, freeze reporting, reset loggedEvent
         freezeReport=true
         # set freeze report timeout to 1 day ( 60 * 60 * 24 )
         freezeReportTimeout=`expr $timeout \* 1440`
         loggedEvent=0
       fi
       sleep $reportTimeout
     fi
  else
     if [[ ! -z $oneExceptionQueue ]]; then
      RxExcWan=$(cat /proc/driver/dqm/mib | grep "RG Exception WAN")
      [ -z "${RxExcWan}" ] && wanDQMnumfull=0 || wanDQMnumfull=$(echo ${RxExcWan} | awk 'END {print$11}' | grep -o -E '[0-9]+')
      RxExcLan=$(cat /proc/driver/dqm/mib | grep "RG Exception LAN")
      [ -z "${RxExcLan}" ] && lanDQMnumfull=0 || lanDQMnumfull=$(echo ${RxExcLan} | awk 'END {print$11}' | grep -o -E '[0-9]+')
     else
      RxExcWan=$(cat /proc/driver/dqm/mib | grep "Rx-exc-cm0")
      [ -z "${RxExcWan}" ] && wanDQMnumfull=0 || wanDQMnumfull=$(echo ${RxExcWan} | awk 'END {print$8}' | grep -o -E '[0-9]+')
      lanDQMnumfullMax=0
      for file in /sys/class/net/eth*
      do
        eth=${file##*/}
        RxExcEth=$(cat /proc/driver/dqm/mib | grep "Rx-exc-${eth}")
        [ -z "${RxExcEth}" ] && continue
        lanDQMnumfull=$(echo ${RxExcEth} | awk 'END {print$8}' | grep -o -E '[0-9]+')
        if [ $lanDQMnumfullMax -le "${lanDQMnumfull}" ]; then
          lanDQMnumfullMax="${lanDQMnumfull}"
        fi
      done
      lanDQMnumfull=$lanDQMnumfullMax
     fi
     wanDQMTrigger=$(($savedWanDQMnumfull + $dqmFullTrigger))
     lanDQMTrigger=$(($savedLanDQMnumfull + $dqmFullTrigger))

     if [ $percentIdle -le $minimumIdlePercent ] || [ $wanDQMnumfull -gt $wanDQMTrigger ] || [ $lanDQMnumfull -gt $lanDQMTrigger ]; then
       loggedEvent=`expr $loggedEvent + 1`

       # prevent report too many log events to CM
       if [ $loggedEvent -le $maxLogEventCount ]; then
         # report event
         systemSecurityEventText="Abnormal_DoS_Attack:_IdlePercent="$percentIdle"_WAN_DQM_FULL="$wanDQMnumfull"_LAN_DQM_FULL="$lanDQMnumfull

         latticecli -n "set Cm.LogEventId "$systemSecurityEventId
         latticecli -n "set Cm.LogEventText "$systemSecurityEventText
       else
         # report maximum events, freeze reporting, reset loggedEvent
         freezeReport=true
         # set freeze report timeout to 1 day ( 60 * 60 * 24 )
         freezeReportTimeout=`expr $timeout \* 1440`
         loggedEvent=0
       fi
       sleep $reportTimeout
     fi

     savedWanDQMnumfull=$wanDQMnumfull
     savedLanDQMnumfull=$lanDQMnumfull
  fi
done
