#!/bin/bash
#
# As CM Implemented a framework to report security violations and events of interest using BfcEventLog,
# this script is used to monitor the thermal level. If thermal exceeds a certain level, it will
# report security event log via Lattice API to CM.
#
# Author:         Andy Yin
# Creation Date:  Dec 12, 2018
#

# To prevent report to many event logs
maxLogEventCount=3
# If thermal reachs this threshold, reprot log event
thermalThreshold=70
# Loop timeout, by seconds
timeout=60
# Once event log is reported, wait timeout to send a new one, by seconds
reportTimeout=600
# After reported maxLogEventCount events, freeze until timeout
freezeReport=false
freezeReportTimeout=0
# EventId sent by RG
systemSecurityEventId="1.3.6.1.4.1.4413.2.2.2.1.9.1.3.1.1.5"
loggedEvent=0

echo "Start monitoring thermal level, set threshold to "$thermalThreshold" ..."

# monitoring loop
while [ true ]
do
  sleep $timeout
  thermal=0

  # check if report is in freeze
  if [ $freezeReport ] && [ $freezeReportTimeout -gt 0 ]; then
    freezeReportTimeout=`expr $freezeReportTimeout - $timeout`
    if [ $freezeReportTimeout -le 0 ]; then
      freezeReport=false
      freezeReportTimeout=0
    fi
    continue
  fi

  # get thermal from lattice API, bypass error output
  result=$(latticecli -n "get Thermal.CurrentTemperature" 2>/dev/null)
  thermal=${result##*: }

  # check if thermal reach the preset threshold, bypass error output
  if [ $thermal -gt $thermalThreshold ] 2>/dev/null; then
    loggedEvent=`expr $loggedEvent + 1`

    # prevent report too many log events to CM
    if [ $loggedEvent -le $maxLogEventCount ]; then
      # report event
      echo "Thermal value is "$thermal", exceeds "$thermalThreshold", trigger event log..."
      systemSecurityEventText="Thermal_value_is_"$thermal",exceeds_threshold_"$thermalThreshold"!"

      latticecli -n "set Cm.LogEventId "$systemSecurityEventId
      latticecli -n "set Cm.LogEventText "$systemSecurityEventText
    else
      # report maximum events, freeze reporting, reset loggedEvent
      freezeReport=true
      # set freeze report timeout to 1 day ( 60 * 60 * 24 )
      freezeReportTimeout=`expr $timeout \* 1440`
      loggedEvent=0
      echo "reported "$maxLogEventCount" events, freeze reporting until timeout reached "$freezeReportTimeout
    fi

    sleep $reportTimeout
  fi
done
