#!/bin/bash
#
# As CM Implemented a framework to report security violations and events of interest using BfcEventLog,
# this script is used to monitor avialable memory. If available memory drops below a certain count, it will
# report security event log via Lattice API to CM.
#
# Author:         Chris Zacker
# Creation Date:  Jan 29, 2019
#

# To prevent report to many event logs
maxLogEventCount=3

# Memory log event report threshold in KB
# memory size available can be gotten by "cat /proc/meminfo"
memoryThreshold=1000

# Loop timeout, by seconds
timeout=60
# Once event log is reported, wait timeout to send a new one, by seconds
reportTimeout=600
# After reported maxLogEventCount events, freeze until timeout
freezeReport=false
freezeReportTimeout=0
# EventId sent by RG
systemSecurityEventId="1.3.6.1.4.1.4413.2.2.2.1.9.1.3.1.1.5"
loggedEvent=0

if [ -f "/proc/meminfo" ]; then
   systemMemTotal=$(cat /proc/meminfo | grep MemTotal | grep -o -E '[0-9]+')
   memoryThreshold=$(expr $systemMemTotal \* 1 / 10)
fi

echo "Start monitoring available memory, threshold set to "$memoryThreshold" ..."

# monitoring loop
while [ true ]
do
  sleep $timeout
  currentMemAvailable=0

  # check if report is in freeze
  if [ $freezeReport ] && [ $freezeReportTimeout -gt 0 ]; then
    freezeReportTimeout=`expr $freezeReportTimeout - $timeout`
    if [ $freezeReportTimeout -le 0 ]; then
      freezeReport=false
      freezeReportTimeout=0
    fi
    continue
  fi

  currentMemAvailable=$(cat /proc/meminfo | grep MemAvailable | grep -o -E '[0-9]+')

  # check if session entries reach the preset threshold
  if [ $currentMemAvailable -lt $memoryThreshold ]; then
    loggedEvent=`expr $loggedEvent + 1`

    # prevent report too many log events to CM
    if [ $loggedEvent -le $maxLogEventCount ]; then
      # report event
      systemSecurityEventText="Current_memory_available_"$currentMemAvailable",lessthan_"$memoryThreshold

      latticecli -n "set Cm.LogEventId "$systemSecurityEventId
      latticecli -n "set Cm.LogEventText "$systemSecurityEventText
    else
      # report maximum events, freeze reporting, reset loggedEvent
      freezeReport=true
      # set freeze report timeout to 1 day ( 60 * 60 * 24 )
      freezeReportTimeout=`expr $timeout \* 1440`
      loggedEvent=0
    fi

    sleep $reportTimeout
  fi
done
