#!/bin/bash

#
# As CM Implemented a framework to report security violations and events of interest using BfcEventLog,
# this script is used to monitor the jtag registers.
#
# Author:         Chris Zacker
# Creation Date:  Feb 13, 2019
#

# To prevent report to many event logs
maxLogEventCount=3

# To see if the jtag is being used the script will monitor the jtag registers
# within the processor and see if they ever change from their default values
# devmem is used to read the individual registers

jtag_general_cntrl0_addr=0xd3890100
jtag_general_cntrl0_default=$(devmem $jtag_general_cntrl0_addr 32 )

jtag_general_cntrl1_addr=0xd3890104
jtag_general_cntrl1_default=$(devmem $jtag_general_cntrl1_addr 32 )

jtag_general_cntrl2_addr=0xd3890108
jtag_general_cntrl2_default=$(devmem $jtag_general_cntrl2_addr 32 )

jtag_general_cntrl3_addr=0xd389010c
jtag_general_cntrl3_default=$(devmem $jtag_general_cntrl3_addr 32 )

jtag_general_cntrl4_addr=0xd3890110
jtag_general_cntrl4_default=$(devmem $jtag_general_cntrl4_addr 32 )

jtag_general_status1_addr=0xd3890118
jtag_general_status1_default=$(devmem $jtag_general_status1_addr 32 )

# Loop timeout, by seconds
timeout=60
# Once event log is reported, wait timeout to send a new one, by seconds
reportTimeout=600
# After reported maxLogEventCount events, freeze until timeout
freezeReport=false
freezeReportTimeout=0
# EventId sent by RG
systemSecurityEventId="1.3.6.1.4.1.4413.2.2.2.1.9.1.3.1.1.5"
loggedEvent=0

echo "Start monitoring Abnormal JTAG usage"

# monitoring loop
while [ true ]
do

  sleep $timeout

  trigger_event=0

  # check if report is in freeze
  if [ $freezeReport ] && [ $freezeReportTimeout -gt 0 ]; then
    freezeReportTimeout=`expr $freezeReportTimeout - $timeout`
    if [ $freezeReportTimeout -le 0 ]; then
      freezeReport=false
      freezeReportTimeout=0
    fi
    continue
  fi

  register_value=$(devmem $jtag_general_cntrl0_addr 32 )
  if [[ $register_value != $jtag_general_cntrl0_default ]]; then
    trigger_event=1
    jtag_general_cntrl0_default=$register_value
  fi

  register_value=$(devmem $jtag_general_cntrl1_addr 32 )
  if [ $register_value != $jtag_general_cntrl1_default ]; then
    trigger_event=1
    jtag_general_cntrl1_default=$register_value
  fi

  register_value=$(devmem $jtag_general_cntrl2_addr 32 )
  if [ $register_value != $jtag_general_cntrl2_default ]; then
    trigger_event=1
    jtag_general_cntrl2_default=$register_value
  fi

  register_value=$(devmem $jtag_general_cntrl3_addr 32 )
  if [ $register_value != $jtag_general_cntrl3_default ]; then
    trigger_event=1
    jtag_general_cntrl3_default=$register_value
  fi

  register_value=$(devmem $jtag_general_cntrl4_addr 32 )
  if [ $register_value != $jtag_general_cntrl4_default ]; then
    trigger_event=1
    jtag_general_cntrl4_default=$register_value
  fi

  register_value=$(devmem $jtag_general_status1_addr 32 )
  if [ $register_value != $jtag_general_status1_default ]; then
    trigger_event=1
    jtag_general_status1_default=$register_value
  fi


  if [ $trigger_event -ne 0 ]; then
    loggedEvent=`expr $loggedEvent + 1`

    # prevent report too many log events to CM
    if [ $loggedEvent -le $maxLogEventCount ]; then
      # report event
      systemSecurityEventText="Abnormal_JTAG_Usage"

      latticecli -n "set Cm.LogEventId "$systemSecurityEventId
      latticecli -n "set Cm.LogEventText "$systemSecurityEventText
    else
      # report maximum events, freeze reporting, reset loggedEvent
      freezeReport=true
      # set freeze report timeout to 1 day ( 60 * 60 * 24 )
      freezeReportTimeout=`expr $timeout \* 1440`
      loggedEvent=0
    fi
    sleep $reportTimeout
  fi
done
