#!/bin/bash

function find_mtd_partition()
{
	local arg=$1
	local out=$2
	for x in /sys/class/mtd/mtd*; do
		if [ -e $x/name ]; then
			name=$(cat $x/name)

			# 0x400 = MTD_WRITEABLE
			flags=$(cat $x/flags)
			if [ "$name" = "$arg" \
					-a $(($flags & 0x400)) != 0 ]; then
				eval $out=${x#*mtd/mtd}
				return 0
			fi
		fi
	done
	return 1
}

function get_mtd_attr()
{
	arg0=$1
	arg1=$2
	out=$3
	file=/sys/class/mtd/mtd${arg0}/$arg1
	if [ -e $file ]; then
		retval=$(cat $file)
		eval $out=$retval
	else
		echo $file
		return 1
	fi
	return 0
}

# Enable panic on rcu stall
[ -f /proc/sys/kernel/panic_on_rcu_stall ] && echo 1 > /proc/sys/kernel/panic_on_rcu_stall

mtdoops=0
if find_mtd_partition flash1.debug mtdoops_part; then
	OOPS_DEV=/dev/mtd$mtdoops_part
	get_mtd_attr $mtdoops_part size OOPS_DEV_SIZE
	mtdoops=1
fi
if find_mtd_partition flash0.debug mtdoops_part; then
	OOPS_DEV=/dev/mtd$mtdoops_part
	get_mtd_attr $mtdoops_part size OOPS_DEV_SIZE
	mtdoops=1
fi

OOPS_RECORD_SIZE=65536

if [ $mtdoops = 1 ]; then
	echo mtdoops: device=$OOPS_DEV size=$OOPS_DEV_SIZE
	data=$(hexdump -e '"%08x"' -n 4 $OOPS_DEV)
	if [ "$data" != "ffffffff" ]; then
		echo  "==== Save Previous crash log dump ===="
		echo  "==== Start: Previous crash log dump ====" > /data/oops.log
		for ((i = 0 ; i <= $OOPS_DEV_SIZE ; i+=$OOPS_RECORD_SIZE)); do
			data=$(hexdump -e '"%08x"' -s $i -n 4 $OOPS_DEV)
			if [ "$data" != "ffffffff" ]; then
				if [ -d /data ]; then
					dd if=$OOPS_DEV bs=$OOPS_RECORD_SIZE skip=$(($i/$OOPS_RECORD_SIZE)) count=1 2> /dev/null >> /data/oops.log
				else
					dd if=$OOPS_DEV bs=$OOPS_RECORD_SIZE skip=$(($i/$OOPS_RECORD_SIZE)) count=1 2> /dev/null
				fi
			else
				i=$OOPS_DEV_SIZE
			fi
		done
		echo  "==== End: Previous crash log dump ====" >> data/oops.log

		flash_erase $OOPS_DEV 0 0
	fi
	# Load mtdoops driver
	insmod $(echo $(find /lib/modules/ -name mtdoops.ko)) mtddev=$mtdoops_part record_size=$OOPS_RECORD_SIZE
fi

MMCBLK=mmcblk0
function find_mmc_partition()
{
	local arg=$1
	local out=$2
	for x in /sys/class/block/${MMCBLK}*; do
		if [ -e $x/uevent ]; then
			name=$(cat $x/uevent | grep PARTNAME | sed s/PARTNAME=//)

			if [ "$name" = "$arg" ]; then
				eval $out=$(cat $x/partition)
				return 0
			fi
		fi
	done
	return 1
}

function get_mmc_attr()
{
	arg0=$1
	arg1=$2
	out=$3
	file=/sys/class/block/${MMCBLK}p${arg0}/$arg1
	if [ -e $file ]; then
		retval=$(cat $file)
		eval $out=$retval
	else
		echo $file
		return 1
	fi
	return 0
}

mmcoops=0
if find_mmc_partition debug mmcoops_part; then
	OOPS_DEV=/dev/${MMCBLK}p${mmcoops_part}
	get_mmc_attr $mmcoops_part size OOPS_DEV_SIZE
	get_mmc_attr $mmcoops_part start OOPS_DEV_START
	OOPS_DEV_SECTOR_SIZE=$(cat /sys/class/block/$MMCBLK/queue/logical_block_size)
	mmcoops=1
fi

if [ $mmcoops = 1 ]; then
	echo mmcoops: device=$OOPS_DEV start=$OOPS_DEV_START size=$OOPS_DEV_SIZE sector_size=$OOPS_DEV_SECTOR_SIZE
	insmod $(echo $(find /lib/modules/ -name oops_mmc.ko)) partition_start_block=$OOPS_DEV_START partition_size=$(($OOPS_DEV_SIZE*$OOPS_DEV_SECTOR_SIZE)) mmc_blksz=$OOPS_DEV_SECTOR_SIZE rw_blocks=$(($OOPS_RECORD_SIZE/$OOPS_DEV_SECTOR_SIZE)) mmcdev=/dev/$MMCBLK dump_file_path=/data/oops.log
fi

vfoops=0
if [ -e /dev/flash-debugb ]; then
	OOPS_DEV=flash-debugb
	vfoops=1
fi
if [ -e /dev/flash-debugo ]; then
	OOPS_DEV=flash-debugo
	vfoops=1
fi

if [ $vfoops = 1 ]; then
	OOPS_DEV_SECTOR_SIZE=$(cat /sys/class/block/$OOPS_DEV/queue/logical_block_size)
	OOPS_DEV_SIZE=$(cat /sys/class/block/$OOPS_DEV/size)
	echo vfoops: device=$OOPS_DEV size=$OOPS_DEV_SIZE block_size=$OOPS_DEV_SECTOR_SIZE
	insmod $(echo $(find /lib/modules/ -name vfbio_oops.ko)) rw_blocks=$(($OOPS_RECORD_SIZE/$OOPS_DEV_SECTOR_SIZE)) blkdev=/dev/$OOPS_DEV dump_file_path=/data/oops.log
fi

if [ -e /proc/driver/sramdump/show_prev ]; then
	cat /proc/driver/sramdump/show_prev | grep "Kernel crash detected: Yes" > /dev/null
	if [ $? -eq 0 ]; then
		cat /proc/driver/sramdump/show_prev >> /data/oops.log
	else
		cat /data/oops.log | grep "Broadcom SRAM: No Crash Detected" > /dev/null
		if [ $? -eq 0 ]; then
			# Remove all lines inclusing and following this match
			sed -i '/Broadcom SRAM: No Crash Detected/,$d' /data/oops.log
		fi
		echo "======> Broadcom SRAM: No Crash Detected <======" >> /data/oops.log
		cat /proc/driver/sramdump/show_now >> /data/oops.log
	fi
fi
