#!/bin/bash
CONSOLE=/dev/console
DEBUG=0
ROUTER_PERM_CONFIG_DEFAULT=/usr/local/etc/lattice/router-perm.config.default
ROUTER_PERM_CONFIG=/data/router-perm.config
ROUTER_DYN_CONFIG=/data/router-dyn.config
LATTICE_DYN_CONFIG=/data/lattice-dyn.config
LATTICE_DEFAULT_CFG_FILE="/usr/local/etc/lattice/lattice-config.xml"
LATTICE_ROUTER_CFG_FILE="/usr/local/etc/lattice/lattice-config-router.xml"
LATTICE_BRIDGE_CFG_FILE="/usr/local/etc/lattice/lattice-config-bridge.xml"
LATTICE_PSEUDO_BRIDGE_CFG_FILE="/usr/local/etc/lattice/lattice-config-pseudobridge.xml"

# Validate Router provisioned data
if [ ! -f $ROUTER_PERM_CONFIG ]; then
    echo "/*******************************************************************/"
    echo "!!! WARNING: This device needs to be provisioned !!!"
    echo "/*******************************************************************/"
    cp -f $ROUTER_PERM_CONFIG_DEFAULT $ROUTER_PERM_CONFIG
fi

# /data/wanbridge_ip if exists. The ip address and netmask are set based on the file.
# else default 192.168.0.1/24 255.255.255.0 are set.
if [ -f "/data/wanbridge_ip" ]; then
    wanbridge_ip=`cat /data/wanbridge_ip | cut -f 1 -d "/"`
    if [ -z "${wanbridge_ip}" ]; then
       wanbridge_ip="192.168.0.1"
    fi

    wanbridge_mask=`cat /data/wanbridge_ip`
    if [[ "$wanbridge_mask" == *'/'* ]]; then
       wanbridge_mask=`cat /data/wanbridge_ip | cut -f 2 -d "/"`
    else
       wanbridge_mask=24
    fi
else
    wanbridge_ip="192.168.0.1"
    wanbridge_mask="24"
fi

#LAN ifs
lanif_list=`ls -d /sys/class/net/eth* | cut -f5 -d'/'`
ls /proc/device-tree | grep stb0 > /dev/null
if [ $? -eq 0 ]; then
    lanif_list="$lanif_list stb0"
fi

ls /proc/device-tree | grep moca0 > /dev/null
if [ $? -eq 0 ]; then
    mocaif="moca0"
fi

# VirtNiHal
grep VirtNiHal /usr/local/etc/lattice/lattice-config.xml|grep Yes > /dev/null
noVirtNiHal=$?
# Don't configure VirtNiHal mode with LAG enabled
if [[ $noVirtNiHal -eq 0 ]]; then
    cat $ROUTER_DYN_CONFIG 2> /dev/null | grep '<LAG>' -w -A 7 | grep Enable | grep 1 -q
    if [ $? -eq 0 ]; then
        noVirtNiHal=1
    fi
fi

# Bridge
grep Bridge /usr/local/etc/lattice/lattice-config.xml|grep Yes > /dev/null
noBridge=$?

if [[ $noVirtNiHal -eq 0 ]]; then
    erouter_mode="virtnihal"
elif [[ $noBridge -eq 0 ]]; then
    erouter_mode="disabled"
else
    erouter_mode="dual"
fi


lattice_create_dyn_config () {
    cat <<EOF > $LATTICE_DYN_CONFIG
WANMode=$1
WANInterface=$2
ErouterInitControlMode=$3
ErouterAdminMode=$3
EOF
}

# check and source lattice-dyn.config
if [ ! -f $LATTICE_DYN_CONFIG ]; then
    echo "$LATTICE_DYN_CONFIG does not exist, generate it."
    lattice_create_dyn_config Docsis cm0 $erouter_mode
fi
. $LATTICE_DYN_CONFIG
if [ -z $WANMode ] || [ -z $WANInterface ]; then
    echo "$LATTICE_DYN_CONFIG is corrupted, generate it."
    lattice_create_dyn_config Docsis cm0 $erouter_mode
fi
. $LATTICE_DYN_CONFIG

print_msg() {
	echo > $CONSOLE "/*******************************************************************/"
	echo > $CONSOLE " $1"
	echo > $CONSOLE "/*******************************************************************/"
}

# Print debug messages
print_msg_d() {
	if [ $DEBUG -ne 1 ]; then return; fi
	echo > $CONSOLE "/*******************************************************************/"
	echo > $CONSOLE " $1"
	echo > $CONSOLE "/*******************************************************************/"
}

config_macaddr() {
    value=$(awk -F '[<>]' '/'$1'/{print $3}' $ROUTER_PERM_CONFIG)
    if [ ! -z $value ] && [ "$value" != "(null)" ]; then
        cat /proc/net/dev|grep $2 > /dev/null
        if [ $? -eq 0 ]; then
            ifconfig $2 hw ether $value
        fi
    fi;
}

lattice_rpcd_wait () {
    x=0;
    y=0;
    # Wait until latticerpcd runs, timeout after 20sec
    until [ $x -eq 1 ] || [ $y -ge 20 ]
    do
        ps aux | grep latticerpcd | grep -v grep > /dev/null
        if [ $? -eq 0 ]; then
            x=1;
        fi
        sleep 1
        y=$(($y+1))
    done
}

lattice_flowmgr_wait () {
    x=0;
    y=0;
    # Wait until flowmgr configured, timeout after 20sec
    until [ $x -eq 1 ] || [ $y -ge 20 ]
    do
        lsmod|grep flowmgr > /dev/null
        if [ $? -eq 0 ]; then
            x=1;
        fi
        sleep 1
        y=$(($y+1))
    done
}

lattice_snmp_wait () {
    y=0;
    local altwan="Unknown"
    x=$(snmpget -t 1 -c public -v 2c 172.31.255.45 -Oqv 1.3.6.1.4.1.4413.2.2.2.1.2.12161.1.1.19.0)
    # Wait untile get AltWan mode from CM by snmp
    until [ "$x" = "1" ] || [ "$x" = "2" ] || [ $y -ge 30 ]
    do
        y=$(($y+1))
        x=$(snmpget -t 1 -c public -v 2c 172.31.255.45 -Oqv 1.3.6.1.4.1.4413.2.2.2.1.2.12161.1.1.19.0)
    done
    if [ "$x" = "1" ]; then
        altwan="Ethernet"
    elif [ "$x" = "2" ]; then
        altwan="Docsis"
    fi
    echo $altwan
}

lattice_network_cmds () {
    local wan_mode=`cat $LATTICE_DYN_CONFIG | grep WANMode | awk -F '=' '{print $2}'`
    local wanif=`cat $LATTICE_DYN_CONFIG | grep WANInterface | awk -F '=' '{print $2}'`
    local lanifs=${lanif_list//$wanif/}
    local wanbridge=""
    local cm_wan_mode=$wan_mode

    print_msg "Enter lattice_network_cmds $wan_mode $wanif"
    config_macaddr MocaMacAddress moca0
    brctl addbr br0
    config_macaddr LanBridgeMacAddress br0

    if [[ $1 = "no_latticeframework" ]]; then
        /mnt/apps/etc/init.d/lattice.framework
    else
        pgrep -f latticeframework &> /dev/null
        if [ $? -ne 0 ]; then
            latticeframework -d &
        fi
    fi

    cm_wan_mode=$(lattice_snmp_wait)

    # if EthWAN
    if [ $wan_mode = "Ethernet" ]; then
        lattice_flowmgr_wait
        latticecli -n "set AltWan.EthWanInterface $wanif"
    elif [ $cm_wan_mode !=  $wan_mode ]; then
        echo "AltWAN mode on CM and RG mismatch, force CM AltWAN mode to Docsis"
        latticecli -n "set AltWan.EthWanInterface disable"
    fi

    for lanif in $lanifs
    do
        brctl addif br0 $lanif
        ifconfig $lanif up
    done
    if [ "$mocaif" ]; then
        brctl addif br0 $mocaif
        ifconfig $mocaif up
    fi

    if [ "${wanbridge_mask}" -eq "24" ]; then
        ifconfig br0 ${wanbridge_ip} netmask 255.255.255.0 up
    elif [ "${wanbridge_mask}" -eq "16" ]; then
        ifconfig br0 ${wanbridge_ip} netmask 255.255.0.0 up
    fi

    ls /proc/device-tree | grep pon0 > /dev/null
    if [ $? -eq 0 ] || [ $wan_mode = "Ethernet" ]; then
        wanbridge="wanbridge"
    fi
    if [ $wanbridge ]; then
        brctl addbr $wanbridge
        config_macaddr WanBridgeMacAddress $wanbridge
        brctl addif $wanbridge $wanif
        ifconfig $wanif up
        ifconfig $wanbridge up
        ip route add default dev $wanbridge
    else
        ifconfig $wanif down
        config_macaddr WanBridgeMacAddress $wanif
        ifconfig $wanif up
        ip route add default dev $wanif
    fi

    # Bridge pon0 and emta0 on epon board
    ls /proc/device-tree | grep pon0 > /dev/null
    if [ $? -eq 0 ]; then
        echo "Bridge pon0 and emta0 on epon board..."
        brctl addif $wanbridge pon0
        brctl addif $wanbridge emta0
        ifconfig pon0 up
    fi

    #Run the script to bring up the bare minimum WiFi configuration
    if [ -e /data/wifi.sh ]; then
        sleep 10
        echo "Run the script to bring up the bare minimum WiFi configuration"
        . /data/wifi.sh
    elif [ -e /mnt/apps/etc/wlan/wifi.sh ]; then
        sleep 10
	cp -f /mnt/apps/etc/wlan/wifi.sh /data/wifi.sh
        echo "Run the script to bring up the bare minimum WiFi configuration"
        . /data/wifi.sh
    fi

    if [ -e /usr/local/bin/wifi_setup.sh ] || [ -e /usr/local/sbin/wifi_setup.sh ] || [ -e /sbin/wifi_setup.sh ]; then
        echo "########## Starting Wifi ##########"
        wifi_setup.sh init
        wifi_setup.sh start
        wifi_setup.sh rg_network_add
    fi

    # Start DHCP server
    mkdir -p /var/run /var/udhcpd
    cp /etc/udhcpd-static.conf /var/udhcpd/udhcpd.conf
    echo > /var/udhcpd/udhcpd.leases
    /sbin/udhcpd /etc/udhcpd.conf

    # Start DHCP client
    ls /proc/device-tree | grep pon0 > /dev/null
    if [ $? -eq 0 ]; then
        echo "Wait latticerpcd runs..."
        lattice_rpcd_wait
        /usr/local/bin/oamd &
    fi
    if [ $wanbridge ]; then
        echo "Start DHCP client on $wanbridge..."
        /sbin/udhcpc -i $wanbridge -s /etc/udhcpc.script.rg -p /tmp/udhcpc.$wanbridge.pid >& /dev/null &
        odhcp6c -s /mnt/apps/etc/odhcp6c_script.sh -P0 -t120 $wanbridge -d
        echo 0 > /proc/sys/net/ipv6/conf/$wanbridge/disable_ipv6
    else
        echo "Start DHCP client on $wanif..."
        /sbin/udhcpc -i $wanif -s /etc/udhcpc.script.rg -p /tmp/udhcpc.$wanif.pid >& /dev/null &
        odhcp6c -s /mnt/apps/etc/odhcp6c_script.sh -P0 -t120 $wanif -d
        echo 0 > /proc/sys/net/ipv6/conf/$wanif/disable_ipv6
    fi

    echo 1 > /proc/sys/net/ipv4/conf/all/forwarding

    ip route add 192.168.100.1 dev privbr
    # IP rule to prevent LAN to privbr flow
    iptables -I FORWARD -i br0 -o privbr ! -d 192.168.100.1 -m comment --comment "REJECT LAN to privbr traffic, except to CM LAN addr" -j REJECT --reject-with icmp-net-unreachable
    ip6tables -I FORWARD -i br0 -o privbr -m comment --comment "REJECT LAN to privbr traffic, except to CM LAN addr" -j REJECT --reject-with icmp6-no-route

    # Enable ipv6 conntrack hooks
    ip6tables -t raw -I PREROUTING -m conntrack --ctstate INVALID

    # mcast monitor
    if [ $wanbridge ]; then
        /mnt/apps/etc/mcast_mon start $wanif $wanbridge br0 &
    else
        /mnt/apps/etc/mcast_mon start $wanif $wanif br0 &
    fi

    # start basl
    if [ -f /mnt/apps/etc/basl_launch.sh ]; then
        /bin/sh /mnt/apps/etc/basl_launch.sh &
    fi

    print_msg "Leave lattice_network_cmds"
}

lattice_network_bridged_cmds () {
    local wan_mode=`cat $LATTICE_DYN_CONFIG | grep WANMode | awk -F '=' '{print $2}'`
    local wanif=`cat $LATTICE_DYN_CONFIG | grep WANInterface | awk -F '=' '{print $2}'`
    local lanifs=${lanif_list//$wanif/}
    local cm_wan_mode=$wan_mode

    print_msg "Enter lattice_network_bridged_cmds $wan_mode $wanif"
    brctl addbr wanbridge
    config_macaddr WanBridgeMacAddress wanbridge
    brctl addif wanbridge $wanif

    pgrep -f latticeframework &> /dev/null
    if [ $? -ne 0 ]; then
        latticeframework -d &
    fi

    cm_wan_mode=$(lattice_snmp_wait)

    # if EthWAN
    if [ $wan_mode = "Ethernet" ]; then
        lattice_flowmgr_wait
        latticecli -n "set AltWan.EthWanInterface $wanif"
        latticecli -n "set AltWan.EthWanBridgeModeEnable $wanif"
        latticecli -n "set Ethernet.VlanEnable 0"
    elif [ $cm_wan_mode !=  $wan_mode ]; then
        echo "AltWAN mode on CM and RG mismatch, force CM AltWAN mode to Docsis"
        latticecli -n "set AltWan.EthWanInterface disable"
    fi

    for lanif in $lanifs
    do
        brctl addif wanbridge $lanif
        ifconfig $lanif up
    done

    # Justin, 20180221, add it base on CS4257093
    if [ -f "/data/VirtNiHal_NO" ]; then
        noVirtNiHal=1
    fi
    # justin, end

    if [[ $noVirtNiHal -ne 0 ]] && [[ "$mocaif" ]]; then
        brctl addif wanbridge $mocaif
        ifconfig $mocaif up
    fi

    # Create virtual link to connect private and wan bridge
    ip link add name vethpriv type veth peer name vethwan
    brctl addif privbr vethpriv
    local wanif_running=`cat /sys/class/net/$wanif/carrier 2> /dev/null`
    if [[ $noVirtNiHal -ne 0 ]] && [[ "X$wanif_running" != "X1" ]]; then
        brctl addif wanbridge vethwan
        ifconfig vethwan up
        echo 1 > /sys/class/net/wanbridge/bridge/flush
    fi
    ifconfig vethpriv up
    ebtables --concurrent -t nat -I PREROUTING -i vethpriv -s $(ifconfig wanbridge | head -n1 | tr -s ' ' | cut -d' ' -f5) -j DROP
    ebtables --concurrent -I FORWARD -i veth+ -o cm0 -j DROP
    ebtables --concurrent -I FORWARD -i cm0 -o veth+ -j DROP

    if [ "${wanbridge_mask}" -eq "24" ]; then
         ifconfig wanbridge ${wanbridge_ip} netmask 255.255.255.0 up
         baseip=`echo ${wanbridge_ip} | cut -d"." -f1-3`
         baseip=`echo ${baseip}.0`
         ip route del ${baseip}/${wanbridge_mask} dev wanbridge
    elif [ "${wanbridge_mask}" -eq "16" ]; then
         ifconfig wanbridge ${wanbridge_ip} netmask 255.255.0.0 up
         baseip=`echo ${wanbridge_ip} | cut -d"." -f1-2`
         baseip=`echo ${baseip}.0.0`
         ip route del ${baseip}/${wanbridge_mask} dev wanbridge
    fi

    ip addr add ${wanbridge_ip} dev wanbridge
    ip rule add from ${wanbridge_ip} table bridge_http
    ip route add default dev wanbridge table bridge_http
    ebtables --concurrent -t broute -A BROUTING  -i ! cm0 -p 0x800 --ip-dst ${wanbridge_ip} -j redirect --redirect-target ACCEPT

    iptables -t raw -I PREROUTING  -i privbr  -j CT --notrack
    iptables -t raw -I OUTPUT  -o privbr -j CT --notrack

    # Enable ipv6 conntrack hooks
    ip6tables -t raw -I PREROUTING -m conntrack --ctstate INVALID

    # setup LAG configurations
    which latticelagctl
    if [ $? -eq 0 ]; then
        echo "Wait latticerpcd runs..."
        lattice_rpcd_wait
        latticelagctl
    fi

    #Run the script to bring up the bare minimum WiFi configuration
    if [ -e /data/wifi.sh ]; then
        sleep 10
        echo "Run the script to bring up the bare minimum WiFi configuration"
        . /data/wifi.sh
    elif [ -e /mnt/apps/etc/wlan/wifi.sh ]; then
        sleep 10
        cp -f /mnt/apps/etc/wlan/wifi.sh /data/wifi.sh
        echo "Run the script to bring up the bare minimum WiFi configuration"
        . /data/wifi.sh
    fi

    if [ -e /usr/local/bin/wifi_setup.sh ] || [ -e /usr/local/sbin/wifi_setup.sh ] || [ -e /sbin/wifi_setup.sh ]; then
        echo "########## Starting Wifi ##########"
        wifi_setup.sh init
        wifi_setup.sh start
        wifi_setup.sh rg_network_bridged_add
    fi

    if [[ $noVirtNiHal -eq 0 ]]; then
        lattice_flowmgr_wait
        modprobe rtf
        echo host_inf wanbridge > /proc/driver/rtf/cmd
        ls /proc/device-tree | grep runner -q
        if [ $? -eq 0 ]; then
            echo brcmtag_opcode uc 1 > /proc/driver/dqnet/cmd
        fi
        echo enable 1 > /proc/driver/rtf/cmd
        ps aux | grep httpd_wlan | grep -v grep
        if [ $? -eq 0 ]; then
            echo filter_port add 8000 6 > /proc/driver/rtf/cmd
        fi
        #disable flow manager for the CMIM mode.
        if [ -e /usr/local/bin/wifi_setup.sh ] || [ -e /usr/local/sbin/wifi_setup.sh ] || [ -e /sbin/wifi_setup.sh ]; then
            wifi_setup.sh disable_flowmgr
        fi
    fi

    # If runner nat cache not allocated, disable flowmgr to avoid flow promotion
    ls /proc/device-tree | grep runner -q
    if [ $? -eq 0 ]; then
        natcache=$(hexdump /proc/device-tree/runner*/reg -v | grep 0000030 | awk '{print $(NF-1)$NF}')
        if [ $natcache = "00000000" ]; then
            echo "Runner in low memory config. Disabling flowmgr..."
            echo 0 > /proc/sys/net/flowmgr/enable
            echo 0 > /proc/sys/net/flowmgr/enable_expected_flow
        fi
    fi

    ifconfig $wanif up

    # start basl
    if [ -f /mnt/apps/etc/basl_launch.sh ]; then
        /bin/sh /mnt/apps/etc/basl_launch.sh &
    fi

    print_msg "Leave lattice_network_bridged_cmds"
}

lattice_unconfigure_network_cmds () {
    local wan_mode=`cat $LATTICE_DYN_CONFIG | grep WANMode | awk -F '=' '{print $2}'`
    local wanif=`cat $LATTICE_DYN_CONFIG | grep WANInterface | awk -F '=' '{print $2}'`
    local lanifs=${lanif_list//$wanif/}
    local wanbridge=""

    print_msg "Enter lattice_unconfigure_network_cmds $wan_mode $wanif"
    ls /proc/device-tree | grep pon0 > /dev/null
    if [ $? -eq 0 ] || [ $wan_mode = "Ethernet" ]; then
        wanbridge="wanbridge"
    fi

    # stop mcast monitor
    /mnt/apps/etc/mcast_mon stop

    # stop basl
    if [ -f /mnt/apps/etc/basl_launch.sh ]; then
        /mnt/apps/etc/basl_launch.sh stop
    fi

    # ip route and iptables rule
    ip route del 192.168.100.1 dev privbr
    iptables -D FORWARD -i br0 -o privbr ! -d 192.168.100.1 -m comment --comment "REJECT LAN to privbr traffic, except to CM LAN addr" -j REJECT --reject-with icmp-net-unreachable
    ip6tables -D FORWARD -i br0 -o privbr -m comment --comment "REJECT LAN to privbr traffic, except to CM LAN addr" -j REJECT --reject-with icmp6-no-route
    ip6tables -t raw -D PREROUTING -m conntrack --ctstate INVALID

    # Stop DHCP client
    echo "Stop DHCP client"
    kill -s USR2 `pidof udhcpc`
    ls /proc/device-tree | grep pon0 > /dev/null
    if [ $? -eq 0 ]; then
        killall oamd &> /dev/null
    fi
    killall udhcpc &> /dev/null
    killall odhcp6c &> /dev/null
    if [ $wanbridge ]; then
        echo 1 > /proc/sys/net/ipv6/conf/$wanbridge/disable_ipv6
        ifconfig $wanbridge down
    else
        echo 1 > /proc/sys/net/ipv6/conf/$wanif/disable_ipv6
    fi

    echo 1 > /proc/sys/net/ipv4/conf/all/forwarding

    # Stop DHCP server
    echo "Stop DHCP server"
    killall udhcpd &> /dev/null

    # Teardown Bridge pon0 and emta0 on epon board
    ls /proc/device-tree | grep pon0 > /dev/null
    if [ $? -eq 0 ]; then
        echo "Teardown Bridge pon0 and emta0 on epon board..."
        ifconfig pon0 down
        brctl delif $wanbridge pon0
        brctl delif $wanbridge emta0
    fi

    if [ $wanbridge ]; then
        brctl delbr $wanbridge
    fi
    ifconfig $wanif down

    # if EthWAN
    if [ $wan_mode = "Ethernet" ]; then
        latticecli -n "set AltWan.EthWanInterface disable"
    fi

    # Teardown Bridge br0
    echo "Teardown Bridge br0..."
    for lanif in $lanifs
    do
        ifconfig $lanif down
        brctl delif br0 $lanif
    done
    if [[ $noVirtNiHal -ne 0 ]] && [[ "$mocaif" ]]; then
        ifconfig $mocaif down
        brctl delif br0 $mocaif
    fi
    ifconfig br0 down
    brctl delbr br0
    print_msg "Leave lattice_unconfigure_network_cmds"
}

lattice_unconfigure_network_bridged_cmds () {
    local wan_mode=`cat $LATTICE_DYN_CONFIG | grep WANMode | awk -F '=' '{print $2}'`
    local wanif=`cat $LATTICE_DYN_CONFIG | grep WANInterface | awk -F '=' '{print $2}'`
    local lanifs=${lanif_list//$wanif/}

    print_msg "Enter lattice_unconfigure_network_bridged_cmds $wan_mode $wanif"
    # Stop lagctl
    killall latticelagctl &> /dev/null

    # stop basl
    if [ -f /mnt/apps/etc/basl_launch.sh ]; then
        /mnt/apps/etc/basl_launch.sh stop
    fi

    # Teardown virtual link to connect private and wan bridge
    ebtables --concurrent -t nat -D PREROUTING -i vethpriv -s $(ifconfig wanbridge | head -n1 | tr -s ' ' | cut -d' ' -f5) -j DROP
    ebtables --concurrent -D FORWARD -i veth+ -o cm0 -j DROP
    ebtables --concurrent -D FORWARD -i cm0 -o veth+ -j DROP
    ifconfig vethpriv down
    brctl delif privbr vethpriv
    ip link del vethpriv

    # Remove ip route and iptables rules
    ebtables --concurrent -t broute -D BROUTING  -i ! cm0 -p 0x800 --ip-dst ${wanbridge_ip} -j redirect --redirect-target ACCEPT

    iptables -t raw -D PREROUTING  -i privbr  -j CT --notrack
    iptables -t raw -D OUTPUT  -o privbr -j CT --notrack
    ip6tables -t raw -D PREROUTING -m conntrack --ctstate INVALID
    ip route del default dev wanbridge table bridge_http
    ip rule del from ${wanbridge_ip} table bridge_http
    ip addr del ${wanbridge_ip}/${wanbridge_mask} dev wanbridge

    # Teardown Bridge wanbridge
    echo "Teardown Bridge wanbridge..."
    for lanif in $lanifs
    do
        ifconfig $lanif down
        brctl delif wanbridge $lanif
    done
    if [[ $noVirtNiHal -ne 0 ]] && [[ "$mocaif" ]]; then
        ifconfig $mocaif down
        brctl delif wanbridge $mocaif
    fi
    brctl delif wanbridge $wanif
    ifconfig $wanif down
    ifconfig wanbridge down
    brctl delbr wanbridge
    # if EthWAN
    if [ $wan_mode = "Ethernet" ]; then
        latticecli -n "set AltWan.EthWanInterface disable"
        latticecli -n "set AltWan.EthWanBridgeModeEnable disable"
    fi
    print_msg "Leave lattice_unconfigure_network_bridged_cmds"
}

lattice_mode_get () {
    echo "WANMode $WANMode, WANInterface $WANInterface, ErouterInitControlMode $ErouterInitControlMode, ErouterAdminMode $ErouterAdminMode"
}

lattice_mode_set () {
    WANMode_new=$1
    WANInterface_new=$2
    ErouterInitControlMode_new=$3

    if [ "$WANMode_new" == "Docsis" ]; then
        if [ "$WANInterface_new" != "cm0" ]; then
            echo "Error: Invalid WANInterface $WANInterface_new for WANMode $WANMode_new, valid WANInterface for WANMode $WANMode_new is cm0"
            exit 1
        fi
        if [ "$ErouterInitControlMode_new" != "disabled" -a "$ErouterInitControlMode_new" != "dual" -a "$ErouterInitControlMode_new" != "virtnihal" ]; then
            echo "Error: Invalid ErouterInitControlMode $ErouterInitControlMode_new for WANMode $WANMode_new, valid ErouterInitControlMode for WANMode $WANMode_new is disabled|dual|virtnihal"
            exit 1
        fi
    elif [ "$WANMode_new" == "Ethernet" ]; then
        local valid_lanif=0
        for lanif in $lanif_list
        do
            if [ $lanif == "$WANInterface_new" ]; then
                valid_lanif=1
                break;
            fi
        done
        if [ $valid_lanif -ne 1 ]; then
            echo "Error: Invalid WANInterface $WANInterface_new for WANMode $WANMode_new"
            exit 1
        fi
        if [ "$ErouterInitControlMode_new" != "disabled" -a "$ErouterInitControlMode_new" != "dual" ]; then
            echo "Error: Invalid ErouterInitControlMode $ErouterInitControlMode_new for WANMode $WANMode_new, valid ErouterInitControlMode for WANMode $WANMode_new is disabled|dual"
            exit 1
        fi
    else
        echo "Error: Invalid WANMode $WANMode_new, valid WANMode is Docsis|Ethernet"
        exit 1
    fi

    if [ "$WANMode_new" == "$WANMode" -a "$WANInterface_new" == "$WANInterface" -a "$ErouterInitControlMode_new" == "$ErouterInitControlMode" ]; then
        echo "Error: Same provision"
        exit 1;
    fi

    # unconfigure network
    if [ "$ErouterInitControlMode" == dual ]; then
        lattice_unconfigure_network_cmds
    else
        lattice_unconfigure_network_bridged_cmds
    fi

    # renew lattice-dyn-config
    lattice_create_dyn_config $WANMode_new $WANInterface_new $ErouterInitControlMode_new

    # change lattice configuration file
    if [ "$ErouterInitControlMode_new" != "$ErouterInitControlMode" ]; then
        case "$ErouterInitControlMode_new" in
            "disabled")
                cat $LATTICE_PSEUDO_BRIDGE_CFG_FILE > $LATTICE_DEFAULT_CFG_FILE
                ;;
            "dual")
                cat $LATTICE_ROUTER_CFG_FILE > $LATTICE_DEFAULT_CFG_FILE
                ;;
            "virtnihal")
                cat $LATTICE_BRIDGE_CFG_FILE > $LATTICE_DEFAULT_CFG_FILE
                ;;
        esac
    fi

    # configure network
    # virtnihal change require reboot
    if [ "$ErouterInitControlMode" != "$ErouterInitControlMode_new" ] && [ "$ErouterInitControlMode" == "virtnihal" -o "$ErouterInitControlMode_new" == "virtnihal" ]; then
        echo "The system is going to reboot..."
        sync
        local virt=0
        [ "$ErouterInitControlMode_new" == "virtnihal" ] && virt=1
        latticecli -n "set System.VirtNiHalEnable $virt"
        print_msg "Reboot in 10 seconds for configuration change..."
        reboot -d 10
    else
        print_msg "The system is swtiching to $WANMode_new $WANInterface_new $ErouterInitControlMode_new"
        sleep 2
    fi

    if [ "$ErouterInitControlMode_new" == dual ]; then
        lattice_network_cmds
    else
        lattice_network_bridged_cmds
    fi
}

action=${1}
shift
$action "$@"
