#!/bin/bash
#
# As CM MIB httpIpStackInterfaces controls what IP stack should the HTTP server is accessible,
# this script is used to set up IPTABLES rules to restrict HTTP server access
#
# Author:         Ignatius Cheng
# Creation Date:  Aug 12 , 2020
#

# delete previously added rules
iptables -D INPUT -p tcp -m multiport --dport 80,443 -j HTTP_ACCESS 2> /dev/null
# flush and delete previously added user target
iptables -F HTTP_ACCESS 2> /dev/null; iptables -X HTTP_ACCESS 2> /dev/null;
# create new user target
iptables -N HTTP_ACCESS
# allow redirected access coming from CM thru privbr, CM restricts its own HTTP access
iptables -A HTTP_ACCESS -i privbr -j ACCEPT
# drop all HTTP access that is not allowed.
iptables -A HTTP_ACCESS -j DROP
# add HTTP access rule in INPUT chain
iptables -I INPUT -p tcp -m multiport --dport 80,443 -j HTTP_ACCESS

# perform SNMP GET, finding out the HTTP IP stack MIB value
httpIpStackInterfacesOid="1.3.6.1.4.1.4413.2.2.2.1.1.3.5.0"
mib=$(snmpget -v2c -c private -Ov -Ox 172.31.255.45 $httpIpStackInterfacesOid)
value=$(echo $mib | cut -d' ' -f2)
value=$(echo "0x$value")

# we are only interested in the bit3 (WAN interface), and bit 5 (LAN interface)
bit2=$(( (value & 0x40) == 0x40 ))
bit3=$(( (value & 0x20) == 0x20 ))
bit5=$(( (value & 0x08) == 0x08 ))

# find out the WAN and LAN interface name
# expect WAN interface is either 'cm0' or 'wan'
if [ -d "/sys/class/net/cm0" ]; then
        wanifname="cm0"
elif [ -d "/sys/class/net/wan" ]; then
        wanifname="wan"
else
        echo "WARNING!  $0 cannot determine the WAN interface name!  HTTP access rule is not set for WAN."
fi

# set iptables rules for RG WAN and RG LAN HTTP access
echo "Set IPTABLES rules for RG WAN and RG LAN HTTP access"
if (( bit3 )) && [ -n "${wanifname}" ] ; then
        iptables -I HTTP_ACCESS -i $wanifname -j ACCEPT
fi
if (( bit5 )); then
        iptables -I HTTP_ACCESS -d 192.168.0.1 -j ACCEPT
fi

iptables -A HTTP_ACCESS -j DROP
