#!/bin/bash
#
# This script is used to set up CM Orignated Support in RG.
#   1) add generic CmNat Proxy Rule for CM Port Forwarder Snoop, allowing traffic originated from the RG_CmOrg Address to be proxied as CM traffic.
#   2) query CM DNS server list and add the list to RG DNS resolver's nameservers.
#   3) add low priority (high metric) default route, promoting CM Originiated path.
#         This low priority will allow RG applications to automatically bind to the RG_CmOrg Address, when there is not a better default route.
#
# Author:         Ignatius Cheng
# Creation Date:  Nov 16, 2020
#

doIpv4=0
doIpv6=0
doStart=0
doStop=0
retcode=0

if [ -z $1 ]; then echo "*** Error missing parameters#1 ***"; exit 1;
elif [ -z $2 ]; then echo "*** Error missing parameter#2 ***"; exit 1;
fi

if [ $1 == "ipv4" ]; then doIpv4=1;
elif [ $1 == "ipv6" ]; then doIpv6=1;
else echo "*** Error unknown operation type '$1'.  Expected types are 'ipv4', 'ipv6' ***"; exit 1;
fi

if [ $2 == "start" ]; then doStart=1;
elif [ $2 == "stop" ]; then doStop=1;
else echo "*** Error unknown operation type '$2'.  Expected types are 'start', 'stop' ***"; exit 1;
fi

echo "*** Setup CM Originated Proxy Rule - operation = '$1' '$2' ***"

# OIDs
cmRemotePortForwardProxyRuleOid="1.3.6.1.4.1.4413.2.2.2.1.2.12161.1.1.24.0"
cmRemoteDnsServersOid="1.3.6.1.4.1.4413.2.2.2.1.2.12161.1.2.13.0"
cmRemoteDhcpOptionsv6Oid="1.3.6.1.4.1.4413.2.2.2.1.2.12161.1.3.10.0"  # for DPCHv6 option 23 (Dns server list)

# RG's CM Orginated IP Stack.
PRIVATE_SOURCE_IP_CM="172.31.255.45"
PRIVATE_SOURCE_IPV6_CM="fe80::ffff:ac1f:ff2d"
PRIVATE_SOURCE_IP_RG_CMORG="172.31.255.32"
PRIVATE_SOURCE_IPV6_RG_CMORG="fe80::ffff:ac1f:ff20"
IPV4_NUL="0.0.0.0"
IPV6_NUL="::"
CMNAT_PROT="0x7"   # bitmask 0x1 UDP, 0x2 TCP, 0x4 ICMP, thus 0x7 for all
CMNAT_TUNNEL="3"
PRIV_IF_NAME="privbr"
RESOLV_CONF="/etc/resolv.conf"
OPENWRT="0"

# OpenWRT uses a different interface name
[ -f /etc/banner ] && grep "OpenWRT" < /etc/banner > /dev/null && OPENWRT="1"

if [ $OPENWRT == "1" ]; then
	PRIV_IF_NAME="br-privbr"
fi


# Get CM IPv4 DNS list
#  getDnsList
#  return - expanded array of the DNS addresses
function getDnsList
{
	local dnslist
	local output
	local snmpret
	#snmpget -v2c -c private -O v 172.31.255.45 1.3.6.1.4.1.4413.2.2.2.1.2.12161.1.2.13.0
	snmpret=$(snmpget -v2c -c private -O v 172.31.255.45 $cmRemoteDnsServersOid)
	# expected snmpget output if successful
	#    STRING: "10.1.0.42"
	#    STRING: "10.1.0.42,10.1.0.46"
	# check if snmpget matches an expected output pattern
	grep "^STRING:\s\".*\"" <<< $snmpret > /dev/null
	if [ $? == 0 ]; then
		output=$(sed 's/STRING: \"\(.*\)\"/\1/' <<< $snmpret)
		IFS=',' read -r -a dnslist <<< "$output"
	fi
	echo ${dnslist[@]}
}

# Count zeros, given the starting index in an array
# countRunningZero $parm1 ${parm2[@]]}
#    parm1 - starting index looking for zero
#    parm2 - expanded array of item
#  return - number of zero from the starting index
function countRunningZero
{
	local i=$1
	shift
	local item=("$@")
	local zlen=0
	while [ $i -lt ${#item[@]} ]; do
		if [ $((16#${item[$i]})) -eq 0 ]; then
			((zlen++))
			((i++))
		else
			break;
		fi
	done
	echo $zlen;
}

# Compress the IPv6 address string, given a full IPv6 address string notation
# compressIPv6Addr $parm1
#    parm1 - full IPv6 address string notation
#  return - compressed IPv6 address string if successful, empty string if failure
function compressIPv6Addr
{
	local ip6str=$1
	local len i num word
	local zeroLen=0 zeroIdx=0
	local output
	# updated to lowercase
	ip6str=$(awk '{print tolower($0)}' <<< "$ip6str")
	# take out leading zero in each word
	ip6str=$(sed -r 's/(^|:)(0+)([[:xdigit:]])/\1\3/g' <<< "$ip6str")
	# read the IPv6 addr to into array 'word'
	IFS=':' read -r -a word <<< "$ip6str"

	num=${#word[@]}
	# check if input string is valid
	[ $num -ne 8 ] && return

	# find the longest running zero and starting index
	i=0
	while [ $i -lt $num ]; do
		len=$(countRunningZero $i ${word[@]})
		if [ $len -gt $zeroLen ]; then
			zeroLen=$len
			zeroIdx=$i
		fi
		((i+=1+$len))
	done

	if [ $zeroLen -gt 1 ]; then
		# construct the compressed IPv6 addr string
		i=0
		while [ $i -lt $num ]; do
			if [ $i -eq $zeroIdx ]; then
				[ $zeroIdx -eq 0 ] && output+=":"
				((i+=$zeroLen))
				output+=":"
			else
				output+="${word[$i]}"
				((i++))
				[ $i -lt $num ] && output+=":"
			fi
		done
	else
		output=$ip6str
	fi
	echo $output
}

# Get CM DHCP6 Option
#  getDhcp6opts "$parm1" $parm2
#    parm1 - string of the CM DHCP6 Options NET-SNMP output
#    parm2 - option number of the wanted
#  return - expanded array of bytes of the wanted DHCP6 option
function getDhcp6opt
{
	local bytes opts optnumber optlen
	local i=0
	# take out the NET-SNMP output hex-string header
	opts=$(sed 's/Hex-STRING: \(.*\)/\1/' <<< $1)
	bytes=( $opts )
	while [ $i -lt ${#bytes[@]} ]; do
		let optnumber=0x"${bytes[i+0]}${bytes[i+1]}"
		let optlen=0x"${bytes[i+2]}${bytes[i+3]}"
		if [ $optnumber -eq $2 ]; then
			echo ${bytes[@]:((${i}+4)):${optlen}}	# output the options
			break;
		fi
		((i+=$optlen+4))	# move to next option
	done
}

# Get CM IPv6 DNS list
#  getDns6List
#  return - expanded array of the DNS6 addresses
function getDns6List
{
	local dhpc6opts optBytes numDns6 dns6 i
	local dns6list
	#snmpget -v2c -c private -O v 172.31.255.45 1.3.6.1.4.1.4413.2.2.2.1.2.12161.1.3.10.0
	local dhpc6opts=$(snmpget -v2c -c private -O v 172.31.255.45 $cmRemoteDhcpOptionsv6Oid)
	# expected snmpget output if successful
	#    Hex-STRING: 00 03 00 28 18 DE BE D0 00 00 00 3C 00 00 00 5A
	#    00 05 00 18 20 12 00 00 00 00 01 04 00 00 00 00
	# check if snmpget matches an expected output pattern
	grep "^Hex-STRING:\s.*" <<< $dhpc6opts > /dev/null
	if [ $? == 0 ]; then
		optBytes=( $(getDhcp6opt "${dhpc6opts}" 23) )  # get option 23 dns server list from dhcp6 options
		numDns6=$((${#optBytes[@]}/16))  # address list in multiple of 16
		if [ $numDns6 -gt 0 ]; then
			i=0
			while [ $i -lt $numDns6 ]; do
				dns6="${optBytes[(16*i)]}${optBytes[(16*i) +1]}:"
				dns6+="${optBytes[(16*i)+2]}${optBytes[(16*i)+3]}:"
				dns6+="${optBytes[(16*i)+4]}${optBytes[(16*i)+5]}:"
				dns6+="${optBytes[(16*i)+6]}${optBytes[(16*i)+7]}:"
				dns6+="${optBytes[(16*i)+8]}${optBytes[(16*i)+9]}:"
				dns6+="${optBytes[(16*i)+10]}${optBytes[(16*i)+11]}:"
				dns6+="${optBytes[(16*i)+12]}${optBytes[(16*i)+13]}:"
				dns6+="${optBytes[(16*i)+14]}${optBytes[(16*i)+15]}"
				# Compress the ipv6 address
				dns6=$(compressIPv6Addr $dns6)
				# add ipv6 address to the output list
				dns6list+=( $dns6 )
				((i++))
			done
		fi
	fi
	echo ${dns6list[@]}
}

# Update DNS Configuration file with the given DNS address
# updateDnsConf parm1 parm2
#    parm1 - ipv6 or ipv4
#    parm2 - DNS addresses list
#  return - nothing
function updateDnsConf
{
	local ip6=$1
	local dns=$2
	local proto metric c d
	local GEN_RESOLV_RUN_DIR="/var/gen_resolv"
	local CMORG_RESOLV_CONF
	# resolvconf default priority is sorted by name, when it is not in the interface-order
	if [ $ip6 == 1 ]; then
		metric="110"
		proto="6"
	else
		metric="120"
		proto="4"
	fi
	mkdir -p $GEN_RESOLV_RUN_DIR
	CMORG_RESOLV_CONF="$GEN_RESOLV_RUN_DIR/${metric}-cmOrg${proto}.resolv.conf"
	# clear file and create a new /etc/resolv.conf for cmOrg
	echo -n > $CMORG_RESOLV_CONF
	for c in $dns; do
		echo "nameserver $c" >> $CMORG_RESOLV_CONF
	done

	if [ $OPENWRT == "0" ]; then
		if command -v gen_resolv.sh > /dev/null; then
			gen_resolv.sh
			return
		elif [ -n "dns" ]; then
			echo "WARNING: Missing gen_resolv.sh tool!" >&2
			echo "cmOrg.sh will just append CM DNS setting to $RESOLV_CONF" >&2
			for c in $dns; do
				grep "^nameserver $c" < $RESOLV_CONF > /dev/null
				[ $? != 0 ] && echo "nameserver $c" >> $RESOLV_CONF
			done
		fi
	else
		# In OpenWRT, dnsmasq handles all dns queries
		local server_list
		local dup
		local updated=0
		server_list=$(uci get dhcp.@dnsmasq[0].server 2> /dev/null)
		for c in $dns; do
			dup=0
			for d in $server_list; do
				[ $c == $d ] && dup=1
			done
			if [ $dup -eq 0 ]; then
				updated=1
				$(uci add_list dhcp.@dnsmasq[0].server="$c")
			fi
		done
		if [ $updated -eq 1 ]; then
			uci commit dhcp
			/etc/init.d/dnsmasq restart
		fi
	fi
}

function StopIpv4Proxy
{
	# remove any previously added proxy rule
	snmpget -v2c -c private 172.31.255.45 $cmRemotePortForwardProxyRuleOid | grep "$PRIVATE_SOURCE_IP_RG_CMORG 0 $IPV4_NUL 0 $IPV4_NUL 0 $IPV4_NUL 0 $CMNAT_PROT $CMNAT_TUNNEL"
	if [ $? -eq 0 ]; then
		snmpset -v2c -c private -C q 172.31.255.45 $cmRemotePortForwardProxyRuleOid s "$PRIVATE_SOURCE_IP_RG_CMORG 0 $IPV4_NUL 0 $IPV4_NUL 0 $IPV4_NUL 0 $CMNAT_PROT $CMNAT_TUNNEL 0"
	fi

	# low metirc default route, enabling Cm Org support by default
	route=$(ip route show table default default via $PRIVATE_SOURCE_IP_CM dev $PRIV_IF_NAME src $PRIVATE_SOURCE_IP_RG_CMORG proto static 2> /dev/null)
	if [ -n "$route" ]; then
		ip route delete table default default via $PRIVATE_SOURCE_IP_CM dev $PRIV_IF_NAME src $PRIVATE_SOURCE_IP_RG_CMORG proto static
	fi

	rule=$(ip rule list from $PRIVATE_SOURCE_IP_RG_CMORG)
	if [ -n "$rule" ]; then
		ip rule del from $PRIVATE_SOURCE_IP_RG_CMORG table default
	fi
}

function StopIpv6Proxy
{
	# remove any previously added proxy rule
	snmpget -v2c -c private 172.31.255.45 $cmRemotePortForwardProxyRuleOid | grep "$PRIVATE_SOURCE_IPV6_RG_CMORG 0 $IPV6_NUL 0 $IPV6_NUL 0 $IPV6_NUL 0 $CMNAT_PROT $CMNAT_TUNNEL"
	if [ $? -eq 0 ]; then
		snmpset -v2c -c private -C q 172.31.255.45 $cmRemotePortForwardProxyRuleOid s "$PRIVATE_SOURCE_IPV6_RG_CMORG 0 $IPV6_NUL 0 $IPV6_NUL 0 $IPV6_NUL 0 $CMNAT_PROT $CMNAT_TUNNEL 0"
	fi

	# low metirc default route, enabling Cm Org support by default
	route=$(ip -6 route show table default default via $PRIVATE_SOURCE_IPV6_CM dev $PRIV_IF_NAME src $PRIVATE_SOURCE_IPV6_RG_CMORG proto static 2> /dev/null)
	if [ -n "$route" ]; then
		ip -6 route delete table default default via $PRIVATE_SOURCE_IPV6_CM dev $PRIV_IF_NAME src $PRIVATE_SOURCE_IPV6_RG_CMORG proto static
	fi

	rule=$(ip -6 rule list from $PRIVATE_SOURCE_IPV6_RG_CMORG)
	if [ -n "$rule" ]; then
		ip rule del from $PRIVATE_SOURCE_IPV6_RG_CMORG table default
	fi
}

#### Main ####
if [ $doIpv4 == 1 -a $doStart == 1 ]; then
	StopIpv4Proxy

	# add proxy rule
	snmpset -v2c -c private -C q 172.31.255.45 $cmRemotePortForwardProxyRuleOid s "$PRIVATE_SOURCE_IP_RG_CMORG 0 $IPV4_NUL 0 $IPV4_NUL 0 $IPV4_NUL 0 $CMNAT_PROT $CMNAT_TUNNEL 1"
	# get retcode of the snmpset add proxy rule operation
	retcode=$?

	ip route add table default default via $PRIVATE_SOURCE_IP_CM dev $PRIV_IF_NAME src $PRIVATE_SOURCE_IP_RG_CMORG proto static metric 2048
	if [ $? -ne 0 ]; then
		echo "WARNING!!! Failed to add CmOrg route \"ip route add table default default via $PRIVATE_SOURCE_IP_CM dev $PRIV_IF_NAME src $PRIVATE_SOURCE_IP_RG_CMORG metric 2048\""
	fi

	ip rule add from $PRIVATE_SOURCE_IP_RG_CMORG table default
	if [ $? -ne 0 ]; then
		echo "WARNING!!! Failed to add CmOrg rule \"ip rule add from $PRIVATE_SOURCE_IP_RG_CMORG table default\""
	fi

	dnsAddrs=$(getDnsList)
	updateDnsConf 0 "${dnsAddrs}"
fi

if [ $doIpv6 == 1 -a $doStart == 1 ]; then
	StopIpv6Proxy

	# add proxy rule
	snmpset -v2c -c private -C q 172.31.255.45 $cmRemotePortForwardProxyRuleOid s "$PRIVATE_SOURCE_IPV6_RG_CMORG 0 $IPV6_NUL 0 $IPV6_NUL 0 $IPV6_NUL 0 $CMNAT_PROT $CMNAT_TUNNEL 1"
	# get retcode of the snmpset add proxy rule operation
	retcode=$?

	ip -6 route add table default default via $PRIVATE_SOURCE_IPV6_CM dev $PRIV_IF_NAME src $PRIVATE_SOURCE_IPV6_RG_CMORG proto static metric 2048
	if [ $? -ne 0 ]; then
		echo "WARNING!!! Failed to add CmOrg route \"ip -6 route add table default default via $PRIVATE_SOURCE_IPV6_CM dev $PRIV_IF_NAME src $PRIVATE_SOURCE_IPV6_RG_CMORG metric 2048\""
	fi

	ip -6 rule add from $PRIVATE_SOURCE_IPV6_RG_CMORG table default
	if [ $? -ne 0 ]; then
		echo "WARNING!!! Failed to add CmOrg rule \"ip -6 rule add from $PRIVATE_SOURCE_IPV6_RG_CMORG table default\""
	fi

	dnsAddrs=$(getDns6List)
	updateDnsConf 1 "${dnsAddrs}"
fi

if [ $doIpv4 == 1 -a $doStop == 1 ]; then
	StopIpv4Proxy
	updateDnsConf 0 ""
fi

if [ $doIpv6 == 1 -a $doStop == 1 ]; then
	StopIpv6Proxy
	updateDnsConf 1 ""
fi
exit $retcode
