 /****************************************************************************
 *
 * Copyright (c) 2019 Broadcom Corporation
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/kthread.h>
#include <linux/timer.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <net/netevent.h>
#include <net/rtnetlink.h>
#include <linux/inet.h>
#include <linux/inetdevice.h>
#include <linux/netfilter.h>
#include <linux/netfilter_bridge.h>
#include "br_private.h"
#include "bcmnethooks.h"
#include "dqnet.h"
#include "dqnet_priv.h"
#include "dqnet_dbg.h"
#include "dqnet_brcmtag.h"
#include "ethsw.h"
#include "rtf.h"
#include "rdpa_types.h"
#include "rdpa_api.h"
#include "rdpa_cpu_helper.h"
#include "rdpa_ip_class.h"
#include "autogen/rdpa_ag_ip_class.h"

/*
  Runner Transparent Forwarding Module

  When enabled does following:
  SF2 -
   Disable ARL learning and all upstream packets are switched to IMP port 8
   Downstream packets from IMP ports are swithed based on BRCM tag
  Runner -
   Upstream packet processing is configured transparent bypass all mode.
   All traffic except ARP packets and IP packets destined to 192.168.100.1 and
   host Web IP interface (192.168.0.1) are swiched to DOCSIS WAN interface.
   ARP packets sent to host are futher filtered in SW based on destination MAC
   address of host Web interface. ARP packets not destined for host are
   switched to DOCSIS WAN interface (cm0).
   IP packets send host are further filtered based on destination port number.
   Packets for IP services (like HTTP port 80 or 443) running on host are sent
   to host. Rest of the packets are switched to DOCSIS WAN interface (cm0).

   Downstream packets from DOCSIS WAN interface are switched based on
   CMIM value, if INC_CM bit is set in DOCSIS PD. Runner will insert
   appropriate BRCM tag based in CMIM value when switching packets to SF2.
   Rest of the packets are sent to host for further processing.
*/

#define VERSION     "0.2"
#define VER_STR     "v" VERSION

static unsigned int rtf_enable = 0;
static char host_if[IFNAMSIZ] = {0};
static struct net_device *host_dev = NULL;
/* MAC address of Host Web interface for SW filtering of ARPs */
static unsigned char host_mac[ETH_ALEN] = {0};
/* IPv4 address of Host Web interface for setting HW filter in Runner */
static char host_ip[INET_ADDRSTRLEN] = {0};

struct rtf_table {
	u32 if_id; /* Runner RDPA Interface Number */
	int  profile;
	struct net_device *dev;
	const char *name;
	bool is_nethook;
	u32 packet_in_cnt;
	u32 packet_in_drop_cnt;
	u32 packet_in_host_cnt;
	u32 packet_out_cnt;
	u32 packet_out_fwd_cnt;
	u32 packet_out_drop_cnt;
};

/* Forwarding interface table */
struct rtf_table rtf_inf_tbl[RTF_TABLE_MAX] = {
	{ rdpa_if_wan0,  -1, NULL, NULL,  false }, /* 0 index should be wan port */
	{ rdpa_if_lan0,   0, NULL, NULL,  false },
	{ rdpa_if_lan1,   1, NULL, NULL,  false },
	{ rdpa_if_lan2,   2, NULL, NULL,  false },
	{ rdpa_if_lan3,   3, NULL, NULL,  false },
	{ rdpa_if_lan4,   4, NULL, NULL,  false },
	{ rdpa_if_lan5,   5, NULL, NULL,  false },
	{ rdpa_if_ssid0,  7, NULL, "wl0", false },
	{ rdpa_if_ssid16, 7, NULL, "wl1", false },
	{ rdpa_if_ssid32, 7, NULL, "wl2", false },
};

/* Forwarding match rule entry */
struct rtf_filter_entry {
	u16 port;
	u16 protocol;
};

#define MAX_RTF_FILTER_ENTRY 32
/* IP port filter list of Host Web interface for setting SW filter */
struct rtf_filter_entry rtf_filter_port_list[MAX_RTF_FILTER_ENTRY] = {
#if FOXCONN_CUSTOMIZE //Justin, 20171213, don't redirect http traffic to RG. We will use eCos http server.
	{ 1, IPPROTO_TCP },
#else
	{ 80, IPPROTO_TCP },
	{ 443, IPPROTO_TCP },
#endif
};

void filter_port_show(void)
{
	int i;
	pr_info("IP Dest Port Filter List: ");
	for (i = 0; i < MAX_RTF_FILTER_ENTRY; i++)
		if (rtf_filter_port_list[i].port)
			pr_info("%02d: %-5d %-5d\n", i, rtf_filter_port_list[i].port, rtf_filter_port_list[i].protocol);
	pr_info("\n-------------------------\n");
}

static inline bool is_filter_port(__be16 port, __be16 protocol)
{
	int i;
	for (i = 0; i < MAX_RTF_FILTER_ENTRY; i++) {
		if (rtf_filter_port_list[i].port &&
		    (rtf_filter_port_list[i].port == port) &&
		    (rtf_filter_port_list[i].protocol == protocol))
			return true;
	}
	return false;
}

int rtf_filter_port_add(__be16 port, __be16 protocol)
{
	int i;
	if (is_filter_port(port, protocol))
		return -1;
	for (i = 0; i < MAX_RTF_FILTER_ENTRY; i++) {
		if (rtf_filter_port_list[i].port == 0) {
			rtf_filter_port_list[i].port = port;
			rtf_filter_port_list[i].protocol = protocol;
			return 0;
		}
	}
	return -1;
}

int rtf_filter_port_del(__be16 port, __be16 protocol)
{
	int i;
	for (i = 0; i < MAX_RTF_FILTER_ENTRY; i++) {
		if ((rtf_filter_port_list[i].port == port) &&
		    (rtf_filter_port_list[i].protocol == protocol)) {
			rtf_filter_port_list[i].port = 0;
			rtf_filter_port_list[i].protocol = 0;
			return 0;
		}
	}
	return -1;
}

int rtf_filter_port_clear(void)
{
	int i;
	for (i = 0; i < MAX_RTF_FILTER_ENTRY; i++) {
		rtf_filter_port_list[i].port = 0;
		rtf_filter_port_list[i].protocol = 0;
   }
	return 0;
}

#define MAX_RTF_FILTER_IPV4 32
static __be32 rtf_filter_ipv4_list[MAX_RTF_FILTER_IPV4];

void filter_ipv4_show(void)
{
	int i;
	pr_info("IP Dest IP Filter List: ");
	for (i = 0; i < MAX_RTF_FILTER_IPV4; i++)
		if (rtf_filter_ipv4_list[i])
			pr_info("%02d: %pI4\n", i, &rtf_filter_ipv4_list[i]);
	pr_info("\n-------------------------\n");
}

static inline bool is_filter_ipv4(__be32 ip)
{
	int i;
	for (i = 0; i < MAX_RTF_FILTER_IPV4; i++) {
		if (rtf_filter_ipv4_list[i] &&
		    (rtf_filter_ipv4_list[i] == ip))
			return true;
	}
	return false;
}

int rtf_filter_ipv4_add(__be32 ip)
{
	int i;
	if (is_filter_ipv4(ip))
		return -1;
	for (i = 0; i < MAX_RTF_FILTER_IPV4; i++) {
		if (rtf_filter_ipv4_list[i] == 0) {
			rtf_filter_ipv4_list[i] = ip;
			return 0;
		}
	}
	return -1;
}

int rtf_filter_ipv4_del(__be32 ip)
{
	int i;
	for (i = 0; i < MAX_RTF_FILTER_IPV4; i++) {
		if (rtf_filter_ipv4_list[i] == ip) {
			rtf_filter_ipv4_list[i] = 0;
			return 0;
		}
	}
	return -1;
}

int rtf_filter_ipv4_clear(void)
{
	int i;
	for (i = 0; i < MAX_RTF_FILTER_IPV4; i++)
		rtf_filter_ipv4_list[i] = 0;
	return 0;
}

int rtf_interface(struct net_device *dev)
{
	int i;
	for (i = 0; i < RTF_TABLE_MAX; i++) {
		if (rtf_inf_tbl[i].dev == dev)
			return i;
	}
	return -1;
}

extern int
rdd_ingress_filter_ip_address_entry_add(bdmf_ip_t ip_address);
extern int
rdd_ingress_filter_ip_address_entry_delete(bdmf_ip_t ip_address);
extern int
rdd_ip_address_filter_cfg(uint32_t profile_idx,
			  rdpa_traffic_dir dir,
			  bdmf_boolean enable);
extern uint32_t rdpa_if2rdd_port_profile(rdpa_traffic_dir dir, rdpa_if port);

int rtf_ip_filter(int add, char *ip_str)
{
	int i, ret;
	bdmf_ip_t ip;
	struct net_device *dev;

	ip.family = bdmf_ip_family_ipv4;
	ip.addr.ipv4 = in_aton(ip_str);
	ip.addr.ipv4 = htonl(ip.addr.ipv4);

	if (!ip.addr.ipv4)
		return -1;

	if (add) {
		if ((ret = rdd_ingress_filter_ip_address_entry_add(ip))) {
			pr_err("Error %d adding ip filter for %s\n",
			       ret, ip_str);
			return ret;
		}
		rtf_filter_ipv4_add(ntohl(ip.addr.ipv4));
	} else {
		if ((ret = rdd_ingress_filter_ip_address_entry_delete(ip))) {
			pr_err("Error %d deleting ip filter for %s\n",
			       ret, ip_str);
			return ret;
		}
		rtf_filter_ipv4_del(ntohl(ip.addr.ipv4));
	}
	for (i = 0; i < RTF_TABLE_MAX; i++) {
		int dev_group_type;
		dev = rtf_inf_tbl[i].dev;
		if (!dev || !rtf_inf_tbl[i].is_nethook)
			continue;

		dev_group_type = BCM_NETDEVICE_GROUP_TYPE(dev->group);
		if (dev_group_type != BCM_NETDEVICE_GROUP_LAN)
			continue;

		if (rdd_ip_address_filter_cfg(
		   rtf_inf_tbl[i].profile,
		   rdpa_dir_us, add?true:false)) {
			pr_err("Fail rdd_ip_address_filter_cfg %d\n",
			       0);
			return -1;
		}
	}

	return 0;
}

static inline __be16 rtf_ether_type(u8 *data, bool brcm_tag)
{
	int offset = 2*ETH_ALEN+brcm_tag*BRCM_TAG_LEN;
	__be16 value = data[offset];
	value = (value << 8) | data[offset+1];
	return value;
}
static inline u8 rtf_ipv4_proto(u8 *data, bool brcm_tag)
{
	int offset = ETH_HLEN+brcm_tag*BRCM_TAG_LEN;
	offset += 9; /* IP protocol offset */
	return data[offset];
}
static inline __be16 rtf_ipv4_dst_port(u8 *data, bool brcm_tag)
{
	int offset = ETH_HLEN+brcm_tag*BRCM_TAG_LEN;
	char iplen = data[offset] & 0xF;
	__be16 value;
	offset += iplen<<2; /* IP header */
	offset += 2; /* DST Port offset */
	value = data[offset];
	value = (value << 8) | data[offset+1];
	return value;
}
static inline __be32 rtf_ipv4_dst_ip(u8 *data, bool brcm_tag)
{
	int offset = ETH_HLEN+brcm_tag*BRCM_TAG_LEN;
	__be32 value;
	offset += 16; /* DST IP offset */
	value = data[offset+3];
	value = (value << 8) | data[offset+2];
	value = (value << 8) | data[offset+1];
	value = (value << 8) | data[offset+0];
	return value;
}
enum bcm_nethook_result
ds_forwarding_no_nethook(struct bcm_nethooks *nethooks,
			 struct net_device *dev,
			 struct net_device *dev_out,
			 int rtf_idx,
			 struct fpm_buff *fb)
{
	struct sk_buff *skb;
	struct dqnet_netdev *ndev;

	if (!dev_out || !dev_out->netdev_ops->ndo_start_xmit)
		return BCM_NETHOOK_PASS;

	ndev = netdev_priv(dev);

	skb = dev_alloc_skb(fb->len + 2);
	if (!skb)
		return BCM_NETHOOK_DROP;

	/* Convert FPM to SKB */
	skb_reserve(skb, 2);
	memcpy(skb->data, fb->data, fb->len);
	skb->priority = fb->priority;
	skb_put(skb, fb->len);
	skb->dev = ndev->dev;

	netif_err(ndev, rx_status, ndev->dev,
		  "===============================================\n");
	netif_err(ndev, rx_status, ndev->dev, "RX packet\n");
	show_rx_pkt_desc(ndev, fb);
	show_rx_pkt(ndev, skb->data, skb->len);
	netif_err(ndev, rx_status, ndev->dev,
		  "===============================================\n");

	skb->protocol = eth_type_trans(skb, ndev->dev);
	skb_push(skb, ETH_HLEN);

	if (dev_out->netdev_ops->ndo_start_xmit(skb, dev_out) == 0) {
		rtf_inf_tbl[rtf_idx].packet_out_fwd_cnt++;
		return BCM_NETHOOK_DROP;
	}
	return BCM_NETHOOK_PASS;
}

enum bcm_nethook_result
ds_forwarding(struct bcm_nethooks *nethooks,
	      struct dqnet_netdev *ndev,
	      struct dqnet_netdev *ndev_out,
	      int rtf_idx,
	      struct fpm_buff *fb)
{
	if (nethooks->fpm_tx(fb, ndev_out->dev) == 0) {
		rtf_inf_tbl[rtf_idx].packet_out_fwd_cnt++;
		return BCM_NETHOOK_CONSUMED;
	}
	return BCM_NETHOOK_PASS;
}

enum bcm_nethook_result
us_forwarding(struct bcm_nethooks *nethooks,
	      struct dqnet_netdev *ndev,
	      struct dqnet_netdev *ndev_out,
	      int rtf_idx,
	      struct fpm_buff *fb)
{
	if (nethooks->fpm_tx(fb, ndev_out->dev) == 0) {
		rtf_inf_tbl[rtf_idx].packet_out_fwd_cnt++;
		return BCM_NETHOOK_CONSUMED;
	}
	return BCM_NETHOOK_PASS;
}

bool filter_match(struct dqnet_netdev *ndev, u8 *data, bool brcm_tag)
{
	__be16 eth_type;
	u8 ip_proto;
	__be16 dst_port;
	__be32 dst_ip;
	eth_type = rtf_ether_type(data, brcm_tag);

	if (eth_type == ETH_P_IP) {
		dst_ip = rtf_ipv4_dst_ip(data, brcm_tag);
		if (!is_filter_ipv4(dst_ip))
			return false;
		ip_proto = rtf_ipv4_proto(data, brcm_tag);
		dst_port = rtf_ipv4_dst_port(data, brcm_tag);
		if (ip_proto == IPPROTO_TCP) {
			if (is_filter_port(dst_port, IPPROTO_TCP))
				return true;
		} else if (ip_proto == IPPROTO_UDP) {
			if (is_filter_port(dst_port, IPPROTO_UDP))
				return true;
		}
	} else if (eth_type == ETH_P_ARP) {
		struct ethhdr *hdr = (struct ethhdr *)data;
		if (ether_addr_equal(hdr->h_dest, host_mac))
			return true;

	} else if ((ndev->if_id >= rdpa_if_lan0) &&
		   (ndev->if_id <= rdpa_if_ssid47)) {
		if (eth_type == ETH_P_IEEE1905) {
			return true;
		} else if (eth_type == ETH_P_BRCM) {
			return true;
		} else if (eth_type == ETH_P_802_1X) {
			return true;
		} else if (eth_type == ETH_P_802_1X_PREAUTH) {
			return true;
		}
	}
	return false;
}

bool igmp_match(struct dqnet_netdev *ndev, u8 *data, bool brcm_tag)
{
	__be16 eth_type;
	u8 ip_proto;
	eth_type = rtf_ether_type(data, brcm_tag);

	if (eth_type == ETH_P_IP) {
		ip_proto = rtf_ipv4_proto(data, brcm_tag);
		if (ip_proto == IPPROTO_IGMP)
			return true;
	}
	return false;
}

enum bcm_nethook_result rtf_rx_hook(
	struct net_device *dev, enum bcm_nethook_type type, void *buf)
{
	struct fpm_buff *fb = (struct fpm_buff *)buf;
	struct bcm_nethooks *nethooks = bcm_nethooks_priv();
	struct net_device *dev_out;
	struct dqnet_netdev *ndev_out;
	struct dqnet_netdev *ndev;
	int dev_group_type;
	int rtf_in, rtf_out;
	int i, ret;
	u32 cmim;

	if ((rtf_enable & RTF_ENABLE_MASK) == 0)
		return BCM_NETHOOK_PASS;

	rtf_in = rtf_interface(dev);
	if (rtf_in < 0)
		return BCM_NETHOOK_PASS;
	rtf_inf_tbl[rtf_in].packet_in_cnt++;

	ndev = netdev_priv(dev);

	if (filter_match(ndev, fb->data, fb->brcm_tag_len))
		goto pass;

	dev_group_type = BCM_NETDEVICE_GROUP_TYPE(dev->group);
	if (dev_group_type == BCM_NETDEVICE_GROUP_LAN) {
		rtf_out = 0;
		dev_out = rtf_inf_tbl[rtf_out].dev;
		ndev_out = netdev_priv(dev_out);
		ret = us_forwarding(nethooks, ndev, ndev_out, 0, fb);
		if (ret == BCM_NETHOOK_PASS)
			goto pass;
		else if (ret == BCM_NETHOOK_DROP)
			goto drop;
		else
			return ret;
	} else {
		cmim = fb->cmim;
		pr_debug("%s: rdpa-if %d\n", dev->name, cmim);

		if (igmp_match(ndev, fb->data, fb->brcm_tag_len))
			goto drop;

		if (!cmim)
			goto pass;

		for (i = 0; i < RTF_TABLE_MAX; i++) {
			if (!rtf_inf_tbl[i].dev)
				continue;

			if (cmim == rtf_inf_tbl[i].if_id) {
				rtf_out = i;
				break;
			}
		}

		if (i == RTF_TABLE_MAX)
			goto pass;

		dev_out = rtf_inf_tbl[rtf_out].dev;
		if (!netif_carrier_ok(dev_out))
			goto drop;

		pr_debug("%s: rdpa-if %d -> %s: table index %d\n",
			 dev->name, cmim, dev_out->name, i);

		if (rtf_inf_tbl[rtf_out].is_nethook) {
			ndev_out = netdev_priv(dev_out);
			ret = ds_forwarding(nethooks, ndev,
					    ndev_out, rtf_out, fb);
			if (ret == BCM_NETHOOK_PASS)
				goto pass;
			else if (ret == BCM_NETHOOK_DROP)
				goto drop;
			else
				return ret;
		} else {
			ret = ds_forwarding_no_nethook(nethooks, dev,
						       dev_out, rtf_out, fb);
			if (ret == BCM_NETHOOK_PASS)
				goto pass;
			else if (ret == BCM_NETHOOK_DROP)
				goto drop;
			else
				return ret;
		}
	}
drop:
	rtf_inf_tbl[rtf_in].packet_in_drop_cnt++;
	return BCM_NETHOOK_DROP;

pass:
	rtf_inf_tbl[rtf_in].packet_in_host_cnt++;
	return BCM_NETHOOK_PASS;
}

enum bcm_nethook_result rtf_rxskb_hook(
	struct net_device *dev, enum bcm_nethook_type type, void *buf)
{
	struct sk_buff *skb = buf;
	struct dqnet_netdev *ndev;
	int dev_group_type;

	if ((rtf_enable & RTF_ENABLE_MASK) == 0)
		return BCM_NETHOOK_PASS;

	if (!host_dev)
		return BCM_NETHOOK_DROP;

	ndev = netdev_priv(dev);

	dev_group_type = BCM_NETDEVICE_GROUP_TYPE(dev->group);
	if (dev_group_type == BCM_NETDEVICE_GROUP_WAN) {
#if (LINUX_VERSION_CODE <= KERNEL_VERSION(5,4,0))
		struct net_bridge *br;
		struct pcpu_sw_netstats *brstats;

		if (host_dev->priv_flags & IFF_EBRIDGE) {
			br = netdev_priv(host_dev);
			brstats = this_cpu_ptr(br->stats);
			u64_stats_update_begin(&brstats->syncp);
			brstats->rx_packets++;
			brstats->rx_bytes += skb->len;
			u64_stats_update_end(&brstats->syncp);
		}
#endif
		skb->pkt_type = PACKET_HOST;
		skb->dev = host_dev;
		netif_receive_skb(skb);
		return BCM_NETHOOK_CONSUMED;
	}
	return BCM_NETHOOK_PASS;
}

enum bcm_nethook_result rtf_tx_hook(
	struct net_device *dev, enum bcm_nethook_type type, void *buf)
{
	struct sk_buff *skb = buf;
	struct net_device *dev_in;
	int rtf_in, rtf_out;

	if ((rtf_enable & RTF_ENABLE_MASK) == 0)
		return BCM_NETHOOK_PASS;

	rtf_out = rtf_interface(dev);
	if (rtf_out == -1)
		return BCM_NETHOOK_PASS;
	rtf_inf_tbl[rtf_out].packet_out_cnt++;

	dev_in = __dev_get_by_index(&init_net, skb->skb_iif);
	pr_debug("%s: %d\n", dev->name, skb->skb_iif);
	if (!dev_in)
		return BCM_NETHOOK_PASS;

	rtf_in = rtf_interface(dev_in);
	if (rtf_in != -1) {
		rtf_inf_tbl[rtf_out].packet_out_drop_cnt++;
		return BCM_NETHOOK_DROP;
	}

	return BCM_NETHOOK_PASS;
}

static inline
#if LINUX_VERSION_CODE > KERNEL_VERSION(3, 17, 0)
unsigned int rtf_nf_pre_routing(void *priv,
				struct sk_buff *skb,
				const struct nf_hook_state *state)
#else
unsigned int rtf_nf_pre_routing(const struct nf_hook_ops *ops,
				struct sk_buff *skb,
				const struct net_device *in,
				const struct net_device *out,
				int (*okfn)(struct sk_buff *))
#endif
{
	int rtf_in, rtf_out = 0;
	struct net_device *dev;
	struct dqnet_netdev *ndev;
	struct bcm_nethooks *nethooks;
#if LINUX_VERSION_CODE > KERNEL_VERSION(3, 17, 0)
	struct net_device *in = state->in;
#endif
	if ((rtf_enable & RTF_ENABLE_MASK) == 0)
		return NF_ACCEPT;

	rtf_in = rtf_interface((struct net_device *)in);
	if (rtf_in == -1)
		return NF_ACCEPT;

	if (rtf_inf_tbl[rtf_in].is_nethook)
		return NF_ACCEPT;

	rtf_inf_tbl[rtf_in].packet_out_cnt++;

	dev = rtf_inf_tbl[rtf_out].dev;
	ndev = netdev_priv(dev);
	nethooks = bcm_nethooks_priv();

	skb_push(skb, ETH_HLEN);
	if (filter_match(ndev, skb->data, 0))
		goto pass;

	if (dev->netdev_ops->ndo_start_xmit(skb, dev) == 0) {
		rtf_inf_tbl[rtf_out].packet_out_fwd_cnt++;
		return NF_DROP;
	}
pass:
	skb_pull(skb, ETH_HLEN);
	rtf_inf_tbl[rtf_in].packet_in_host_cnt++;
	return NF_ACCEPT;
}

static inline
#if LINUX_VERSION_CODE > KERNEL_VERSION(3, 17, 0)
unsigned int rtf_nf_post_routing(void *priv,
				 struct sk_buff *skb,
				 const struct nf_hook_state *state)
#else
unsigned int rtf_nf_post_routing(const struct nf_hook_ops *ops,
				 struct sk_buff *skb,
				 const struct net_device *in,
				 const struct net_device *out,
				 int (*okfn)(struct sk_buff *))
#endif
{
	int ret;
#if LINUX_VERSION_CODE > KERNEL_VERSION(3, 17, 0)
	struct net_device *out = state->out;
#endif
	if ((rtf_enable & RTF_ENABLE_MASK) == 0)
		return NF_ACCEPT;

	ret = rtf_tx_hook((struct net_device *)out, 0, (void *) skb);
	if (ret == BCM_NETHOOK_DROP)
		return NF_DROP;
	return NF_ACCEPT;
}

#ifdef CONFIG_SYSCTL
static struct ctl_table rtf_sysctl_table[] = {
	{
		.procname	= "enable",
		.data		= &rtf_enable,
		.maxlen		= sizeof(unsigned int),
		.mode		= 0444,
		.proc_handler	= proc_dointvec,
	},
	{}
};
static struct ctl_table_header	*rtf_sysctl_header;
#endif

void rtf_show(void)
{
	int i;
	pr_info("RTF : %s\n", (rtf_enable&RTF_ENABLE_MASK)?"Enable":"Disable");
	pr_info("VLAN HDR REMOVAL: %s\n",
		(rtf_enable&RTF_VLAN_HDR_REMOVAL_MASK)?"Enable":"Disable");
	pr_info("Host Mac : %pM\n", host_mac);
	pr_info("Host IP  : %s\n", host_ip);
	pr_info("\n------------------------------------------------------------------------------------\n");
	pr_info("Interface Table:\n");
	pr_info("   Name: ID| hook|pktin     |pktin drop|pktin host|pktout fwd|pktout    |pktout drop\n");
	for (i = 0; i < RTF_TABLE_MAX; i++) {
		if (!rtf_inf_tbl[i].dev)
			continue;
		pr_info(" %6s: %02d|%5s|0x%08x|0x%08x|0x%08x|0x%08x|0x%08x|0x%08x\n",
			rtf_inf_tbl[i].dev->name,
			rtf_inf_tbl[i].if_id,
			rtf_inf_tbl[i].is_nethook ? "TRUE" : "FALSE",
			rtf_inf_tbl[i].packet_in_cnt,
			rtf_inf_tbl[i].packet_in_drop_cnt,
			rtf_inf_tbl[i].packet_in_host_cnt,
			rtf_inf_tbl[i].packet_out_fwd_cnt,
			rtf_inf_tbl[i].packet_out_cnt,
			rtf_inf_tbl[i].packet_out_drop_cnt);
	}
	pr_info("\n------------------------------------------------------------------------------------\n");
	filter_ipv4_show();
	filter_port_show();
}

void rtf_nethook_enable(unsigned int config)
{
	int i, rc;
	struct net_device *dev;
	bdmf_object_handle ip_class = NULL;
	rdpa_fc_bypass fc_bypass_mask;
	bdmf_object_handle filter_obj = NULL;
	rdpa_filter_key_t  filter_key = {};
	rdpa_filter_ctrl_t filter_entry = {1, rdpa_forward_action_host};
	unsigned int enable = (config & RTF_ENABLE_MASK);
	unsigned int vlan_hdr_removal = (config & RTF_VLAN_HDR_REMOVAL_MASK);

	if (rtf_enable == config)
		return;

	/* Configure SF2 ARL learning mode */
	rc = ethsw_arl_learning_enable(enable?false:true);
	if (rc)
		return;

	rtf_enable = config;

	rc = rdpa_filter_get(&filter_obj);
	if (!rc) {
		/* init the filter key to the begining of the list */
		*((bdmf_index *)&filter_key) = BDMF_INDEX_UNASSIGNED;
		while (!rdpa_filter_entry_get_next(filter_obj, &filter_key)) {
			if (filter_key.filter == RDPA_FILTER_ETYPE_ARP)
				continue;
			if (filter_key.filter == RDPA_FILTER_ETYPE_UDEF_0)
				continue;
			if (filter_key.filter == RDPA_FILTER_ETYPE_UDEF_1)
				continue;
			if (filter_key.filter == RDPA_FILTER_ETYPE_UDEF_2)
				continue;
			if (filter_key.filter == RDPA_FILTER_ETYPE_UDEF_3)
				continue;
			filter_entry.enabled = enable?false:true;
			rc = rdpa_filter_entry_set(filter_obj, &filter_key,
						   &filter_entry);
			if (rc)
				pr_err("Failed to rdpa_filter_entry_set for %d",
				       filter_key.filter);
		}
	}

	/* Configure Runner in Upstream transparent mode */
	rdpa_ip_class_get(&ip_class);
	rdpa_ip_class_fc_bypass_get(ip_class, &fc_bypass_mask);
	if (enable)
		fc_bypass_mask |= RDPA_IP_CLASS_MASK_BP_ALL;
	else
		fc_bypass_mask &= ~RDPA_IP_CLASS_MASK_BP_ALL;

	if (vlan_hdr_removal)
		fc_bypass_mask |= RDPA_IP_CLASS_MASK_VLAN_HDR_REMOVAL;
	else
		fc_bypass_mask &= ~RDPA_IP_CLASS_MASK_VLAN_HDR_REMOVAL;

	rdpa_ip_class_fc_bypass_set(ip_class, fc_bypass_mask);

	/* Configure Runner IP filter */
	rtf_ip_filter(enable, host_ip);
	// rtf_ip_filter(enable, "192.168.100.1");

	/* Enable/Disable nethooks */
	for (i = 0; i < RTF_TABLE_MAX; i++) {
		int dev_group_type;
		dev = rtf_inf_tbl[i].dev;
		if (!dev)
			continue;

		if (!rtf_inf_tbl[i].is_nethook)
			continue;

		dev_group_type = BCM_NETDEVICE_GROUP_TYPE(dev->group);
		if (dev_group_type == BCM_NETDEVICE_GROUP_LAN) {
			bcm_nethook_enable_hook(dev,
					BCM_NETHOOK_TX_SKB,
					rtf_tx_hook,
					enable);
		}

		if (dev_group_type == BCM_NETDEVICE_GROUP_WAN) {
			bcm_nethook_enable_hook(dev,
					BCM_NETHOOK_RX_SKB,
					rtf_rxskb_hook,
					enable);
		}

		if (dev->group) {
			bcm_nethook_enable_hook(dev,
					BCM_NETHOOK_RX_FPM,
					rtf_rx_hook,
					enable);
		}
	}
}

static inline int get_ip_and_netmask(struct net_device *dev,
				     __be32 *ip)
{
	struct net_device *pnet_device;
	int rc = -1;
	read_lock(&dev_base_lock);
	/* read all devices */
	for_each_netdev(&init_net, pnet_device) {
		if ((netif_running(pnet_device)) &&
		    (pnet_device->ip_ptr != NULL) &&
		    (!strncmp(pnet_device->name, dev->name, IFNAMSIZ))) {
			struct in_device *pin_dev;
			pin_dev = (struct in_device *)(pnet_device->ip_ptr);
			if (pin_dev && pin_dev->ifa_list) {
				*ip = htonl(pin_dev->ifa_list->ifa_address);
				rc = 0;
			}
			break;
		}
	}
	read_unlock(&dev_base_lock);
	return rc;
}

int rtf_host_interface(char *name)
{
	__be32 ip = 0;
	host_dev = __dev_get_by_name(&init_net, name);
	if (!host_dev) {
		memset(host_if, 0, sizeof(host_if));
		memset(host_mac, 0, sizeof(host_mac));
		return -1;
	}
	strncpy(host_if, host_dev->name, sizeof(host_if)-1);
	ether_addr_copy(host_mac, host_dev->dev_addr);
	if (get_ip_and_netmask(host_dev, &ip))
		pr_err(
		   "%s: IP address not initialized for device %s\n",
		   __func__, host_dev->name);
	ip = htonl(ip);
	snprintf(host_ip, INET_ADDRSTRLEN, "%pI4", &ip);
	return 0;
}

static struct nf_hook_ops rtf_nf_ops[] __read_mostly = {
	{
		.pf       = NFPROTO_BRIDGE,
		.priority = INT_MAX,
		.hooknum  = NF_BR_PRE_ROUTING,
		.hook     = rtf_nf_pre_routing,
#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 17, 0)
		.owner    = THIS_MODULE,
#endif
	},
	{
		.pf       = NFPROTO_BRIDGE,
		.priority = INT_MAX,
		.hooknum  = NF_BR_POST_ROUTING,
		.hook     = rtf_nf_post_routing,
#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 17, 0)
		.owner    = THIS_MODULE,
#endif
	},
};

static int registered_rtf_non_nethook;

/* !name, !dev, lookup for any */
/* !name, dev, lookup for dev->name */
/* name, update rtf_inf_tbl dev */
static int rtf_non_nethook_inf(const char *name, struct net_device *dev)
{
	int found = 0;
	int i;

	for (i = 0; i < RTF_TABLE_MAX; i++) {
		if (rtf_inf_tbl[i].is_nethook)
			continue;

		if (!rtf_inf_tbl[i].name)
			continue;

		if (name) {
			if (strncmp(rtf_inf_tbl[i].name, name, IFNAMSIZ))
				continue;

			rtf_inf_tbl[i].dev = dev;
			if (dev) {
				rtf_inf_tbl[i].packet_in_cnt = 0;
				rtf_inf_tbl[i].packet_in_drop_cnt = 0;
				rtf_inf_tbl[i].packet_in_host_cnt = 0;
				rtf_inf_tbl[i].packet_out_cnt = 0;
				rtf_inf_tbl[i].packet_out_fwd_cnt = 0;
				rtf_inf_tbl[i].packet_out_drop_cnt = 0;
			}

			found = 1;
		} else if (dev) {
			if (strncmp(rtf_inf_tbl[i].name, dev->name, IFNAMSIZ))
				continue;

			found = 1;
		} else if (rtf_inf_tbl[i].dev) {
			found = 1;
		}

		if (found)
			break;
	}

	return found;
}

static void rtf_register_non_nethook(struct net_device *dev)
{
	rtf_non_nethook_inf(dev->name, dev);
	if (!registered_rtf_non_nethook && rtf_non_nethook_inf(NULL, NULL)) {
#if LINUX_VERSION_CODE > KERNEL_VERSION(4, 9, 0)
		nf_register_net_hooks(&init_net, rtf_nf_ops,
				      ARRAY_SIZE(rtf_nf_ops));
#else
		nf_register_hooks(rtf_nf_ops, ARRAY_SIZE(rtf_nf_ops));
#endif
		registered_rtf_non_nethook = 1;
	}
}

static void rtf_unregister_non_nethook(struct net_device *dev)
{
	rtf_non_nethook_inf(dev->name, NULL);
	if (registered_rtf_non_nethook && !rtf_non_nethook_inf(NULL, NULL)) {
#if LINUX_VERSION_CODE > KERNEL_VERSION(4, 9, 0)
		nf_unregister_net_hooks(&init_net, rtf_nf_ops,
					ARRAY_SIZE(rtf_nf_ops));
#else
		nf_unregister_hooks(rtf_nf_ops, ARRAY_SIZE(rtf_nf_ops));
#endif
		registered_rtf_non_nethook = 0;
	}
}

static void rtf_register_nethook(struct net_device *dev)
{
	int rc, i;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	int dev_group_type = BCM_NETDEVICE_GROUP_TYPE(dev->group);

	/* Find index in RTF interface table */
	for (i = 0; i < RTF_TABLE_MAX; i++) {
		if (rtf_inf_tbl[i].if_id != ndev->if_id)
			continue;

		rtf_inf_tbl[i].dev = dev;
		rtf_inf_tbl[i].is_nethook = true;
		rtf_inf_tbl[i].packet_in_cnt = 0;
		rtf_inf_tbl[i].packet_in_drop_cnt = 0;
		rtf_inf_tbl[i].packet_in_host_cnt = 0;
		rtf_inf_tbl[i].packet_out_cnt = 0;
		rtf_inf_tbl[i].packet_out_fwd_cnt = 0;
		rtf_inf_tbl[i].packet_out_drop_cnt = 0;
		break;
	}

	if (i == RTF_TABLE_MAX)
		return;

	if (dev_group_type == BCM_NETDEVICE_GROUP_LAN) {
		rc = bcm_nethook_register_hook(dev,
				BCM_NETHOOK_TX_SKB,
				TX_SKB_PRIO_RTF,
				RTF_TX_NAME,
				rtf_tx_hook);
		if (rc) {
			pr_err("Failed to register nethook: %s for dev %s\n",
					RTF_TX_NAME, dev->name);
			return;
		}
		if (rtf_enable & RTF_ENABLE_MASK) {
			bcm_nethook_enable_hook(dev,
					BCM_NETHOOK_TX_SKB,
					rtf_tx_hook,
					true);
		}
	}

	if (dev_group_type == BCM_NETDEVICE_GROUP_WAN) {
		rc = bcm_nethook_register_hook(dev,
				BCM_NETHOOK_RX_SKB,
				RX_SKB_PRIO_RTF,
				RTF_RX_NAME,
				rtf_rxskb_hook);
		if (rc) {
			pr_err("Failed to register nethook: %s for dev %s\n",
					RTF_RX_NAME, dev->name);
			return;
		}
		if (rtf_enable & RTF_ENABLE_MASK) {
			bcm_nethook_enable_hook(dev,
					BCM_NETHOOK_RX_SKB,
					rtf_rxskb_hook,
					true);
		}
	}

	if (dev->group) {
		rc = bcm_nethook_register_hook(dev,
				BCM_NETHOOK_RX_FPM,
				RX_FPM_PRIO_RTF,
				RTF_RX_NAME,
				rtf_rx_hook);
		if (rc) {
			pr_err("Failed to register nethook: %s for dev %s\n",
					RTF_RX_NAME, dev->name);
			return;
		}
		if (rtf_enable & RTF_ENABLE_MASK) {
			bcm_nethook_enable_hook(dev,
					BCM_NETHOOK_RX_FPM,
					rtf_rx_hook,
					true);
		}
	}
}

static void rtf_unregister_nethook(struct net_device *dev)
{
	int rc, i;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	int dev_group_type = BCM_NETDEVICE_GROUP_TYPE(dev->group);
	rc = 0;
	/* Find index in RTF interface table */
	for (i = 0; i < RTF_TABLE_MAX; i++) {
		if (rtf_inf_tbl[i].if_id != ndev->if_id)
			continue;

		rtf_inf_tbl[i].dev = NULL;
		rtf_inf_tbl[i].is_nethook = false;
		break;
	}

	if (i == RTF_TABLE_MAX)
		return;

	if (dev->group) {
		rc = bcm_nethook_unregister_hook(dev, BCM_NETHOOK_RX_FPM,
				rtf_rx_hook);
		if (rc)
			pr_err("Failed to unregister nethook: %s for dev %s\n",
					RTF_RX_NAME, dev->name);
	}

	if (dev_group_type == BCM_NETDEVICE_GROUP_WAN) {
		rc = bcm_nethook_unregister_hook(dev, BCM_NETHOOK_RX_SKB,
				rtf_rxskb_hook);
		if (rc)
			pr_err("Failed to unregister nethook: %s for dev %s\n",
					RTF_RX_NAME, dev->name);
	}

	if (dev_group_type == BCM_NETDEVICE_GROUP_LAN) {
		bcm_nethook_unregister_hook(dev, BCM_NETHOOK_TX_SKB,
				rtf_tx_hook);
		if (rc)
			pr_err("Failed to unregister nethook: %s for dev %s\n",
					RTF_TX_NAME, dev->name);
	}
}

static int rtf_netdev_event(struct notifier_block *this,
		unsigned long event, void *ptr)
{
	struct net_device *dev = netdev_notifier_info_to_dev(ptr);
	struct dqnet_netdev *ndev;

	if (!dev)
		return NOTIFY_DONE;

	if (event == NETDEV_REGISTER) {
		if (bcm_nethook_dev(dev)) {
			ndev = netdev_priv(dev);
			if (ndev->chan->type == DQNET_CHAN_TYPE_FAP_EXCEPT) {
				rtf_register_nethook(dev);
			}
		}

		if (rtf_non_nethook_inf(NULL, dev)) {
			rtf_register_non_nethook(dev);
		}

		if (strncmp(host_if, dev->name, sizeof(host_if)) == 0) {
			host_dev = dev;
			ether_addr_copy(host_mac, dev->dev_addr);
		}
	} else if (event == NETDEV_UNREGISTER) {
		if (strncmp(host_if, dev->name, sizeof(host_if)) == 0) {
			host_dev = NULL;
			memset(host_mac, 0, sizeof(host_mac));
		}

		if (rtf_non_nethook_inf(NULL, dev)) {
			rtf_unregister_non_nethook(dev);
		}

		if (bcm_nethook_dev(dev)) {
			ndev = netdev_priv(dev);
			if (ndev->chan->type == DQNET_CHAN_TYPE_FAP_EXCEPT) {
				rtf_unregister_nethook(dev);
			}
		}
	}

	return NOTIFY_DONE;
}

static struct notifier_block rtf_netdev_notifier = {
	.notifier_call = rtf_netdev_event,
};

static int __init rtf_init(void)
{
	rtf_procfs_init();
#ifdef CONFIG_SYSCTL
	rtf_sysctl_header =
		register_sysctl("net/rtf",
				rtf_sysctl_table);
#endif
	register_netdevice_notifier(&rtf_netdev_notifier);
	pr_info("Runner Transparent Forwarding (RTF) Module Init: %s\n",
		VER_STR);
	return 0;
}

static void __exit rtf_exit(void)
{
	if (rtf_enable & RTF_ENABLE_MASK)
		rtf_nethook_enable(rtf_enable ^ RTF_ENABLE_MASK);

	unregister_netdevice_notifier(&rtf_netdev_notifier);
#ifdef CONFIG_SYSCTL
	unregister_sysctl_table(rtf_sysctl_header);
#endif
	rtf_procfs_exit();
	pr_info("Runner Transparent Forwarding (RTF) Module Exit:\n");
}

module_init(rtf_init);
module_exit(rtf_exit);
MODULE_LICENSE("GPL");
MODULE_ALIAS("rtf");
