 /****************************************************************************
 *
 * Copyright (c) 2016 Broadcom
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 ****************************************************************************/

#include <linux/capability.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/if_bridge.h>
#include <linux/rtnetlink.h>
#include <linux/spinlock.h>
#include <linux/times.h>
#include "netxl.h"

#ifdef CONFIG_NETXL_SYSFS
#define to_dev(obj)	container_of(obj, struct device, kobj)
#define to_netxl(cd)	((struct netxl *)NETXL_CTX(to_net_dev(cd)))

/*
 * Common code for storing netxl parameters.
 */
static ssize_t
store_netxl_parm(struct device *d,
		 const char *buf, size_t len,
		 int (*set)(struct netxl *, unsigned long))
{
	struct netxl *netxl = to_netxl(d);
	char *endp;
	unsigned long val;
	int err;

	if (!ns_capable(dev_net(netxl->dev)->user_ns, CAP_NET_ADMIN))
		return -EPERM;

	val = simple_strtoul(buf, &endp, 0);
	if (endp == buf)
		return -EINVAL;

	err = (*set)(netxl, val);
	return err ? err : len;
}

static ssize_t
rx_offload_show(struct device *d, struct device_attribute *attr, char *buf)
{
	struct netxl *netxl = to_netxl(d);
	return sprintf(buf, "%u\n", netxl->rx_offload);
}

static int
set_rx_offload(struct netxl *netxl, unsigned long val)
{
	netxl->rx_offload = val ? true : false;
	return 0;
}

static ssize_t
rx_offload_store(struct device *d, struct device_attribute *attr,
		 const char *buf, size_t len)
{
	return store_netxl_parm(d, buf, len, set_rx_offload);
}
static DEVICE_ATTR_RW(rx_offload);

static ssize_t
tx_bypass_show(struct device *d, struct device_attribute *attr, char *buf)
{
	struct netxl *netxl = to_netxl(d);
	return sprintf(buf, "%u\n", netxl->tx_bypass);
}

static int
set_tx_bypass(struct netxl *netxl, unsigned long val)
{
	netxl->tx_bypass = val ? true : false;
	return 0;
}

static ssize_t
tx_bypass_store(struct device *d, struct device_attribute *attr,
		 const char *buf, size_t len)
{
	return store_netxl_parm(d, buf, len, set_tx_bypass);
}
static DEVICE_ATTR_RW(tx_bypass);

static ssize_t
wlertx_skb_in_fpm_show(struct device *d, struct device_attribute *attr, char *buf)
{
	struct netxl *netxl = to_netxl(d);
	return sprintf(buf, "%u\n", netxl->wlertx_skb_in_fpm);
}

static int
set_wlertx_skb_in_fpm(struct netxl *netxl, unsigned long val)
{
	netxl->wlertx_skb_in_fpm = val ? true : false;
	return 0;
}

static ssize_t
wlertx_skb_in_fpm_store(struct device *d, struct device_attribute *attr,
		 const char *buf, size_t len)
{
	return store_netxl_parm(d, buf, len, set_wlertx_skb_in_fpm);
}
static DEVICE_ATTR_RW(wlertx_skb_in_fpm);

static ssize_t
mdqm_rx_id_show(struct device *d, struct device_attribute *attr,
		char *buf)
{
	struct netxl *netxl = to_netxl(d);
	return sprintf(buf, "%u\n", netxl->mdqm_rx_id);
}

static int set_mdqm_rx_id(struct netxl *netxl, unsigned long val)
{
	netxl->mdqm_rx_id = val;
	return 0;
}

static ssize_t
mdqm_rx_id_store(struct device *d, struct device_attribute *attr,
		 const char *buf, size_t len)
{
	return store_netxl_parm(d, buf, len, set_mdqm_rx_id);
}
static DEVICE_ATTR_RW(mdqm_rx_id);


static ssize_t
mdqm_tx_id_show(struct device *d, struct device_attribute *attr,
		char *buf)
{
	struct netxl *netxl = to_netxl(d);
	return sprintf(buf, "%u\n", netxl->mdqm_tx_id);
}

static int set_mdqm_tx_id(struct netxl *netxl, unsigned long val)
{
	netxl->mdqm_tx_id = val;
	return 0;
}

static ssize_t
mdqm_tx_id_store(struct device *d, struct device_attribute *attr,
		 const char *buf, size_t len)
{
	return store_netxl_parm(d, buf, len, set_mdqm_tx_id);
}
static DEVICE_ATTR_RW(mdqm_tx_id);

static struct attribute *netxl_attrs[] = {
	&dev_attr_rx_offload.attr,
	&dev_attr_tx_bypass.attr,
	&dev_attr_wlertx_skb_in_fpm.attr,
	&dev_attr_mdqm_rx_id.attr,
	&dev_attr_mdqm_tx_id.attr,
	NULL
};

static struct attribute_group netxl_group = {
	.name = "netxl",
	.attrs = netxl_attrs,
};

int netxl_sysfs_add(struct net_device *dev)
{
	struct kobject *kobj = &dev->dev.kobj;
	int err;

	err = sysfs_create_group(kobj, &netxl_group);
	if (err) {
		pr_info("%s: can't create group %s/%s\n",
			__func__, dev->name, netxl_group.name);
	}

	return err;
}

void netxl_sysfs_del(struct net_device *dev)
{
	struct kobject *kobj = &dev->dev.kobj;
	sysfs_remove_group(kobj, &netxl_group);
}
#else

int netxl_sysfs_add(struct net_device *dev)
{
	return 0;
}

void netxl_sysfs_del(struct net_device *dev)
{
}
#endif

#define NETXL_SYSCTL_PATH "net/netxl/"
#define NETXL_DEV_SYSCTL_ENTRY(var, name, mval, proc) \
	{ \
		.procname	= name, \
		.data		= (void *)offsetof(struct netxl, var), \
		.maxlen		= sizeof(int), \
		.mode		= mval, \
		.proc_handler	= proc, \
	}

static struct netxl_sysctl dev_sysctl = {
	.vars = {
		NETXL_DEV_SYSCTL_ENTRY(rx_offload,
			"rx_offload", 0644, proc_dointvec),
		NETXL_DEV_SYSCTL_ENTRY(tx_bypass,
			"tx_bypass", 0644, proc_dointvec),
		NETXL_DEV_SYSCTL_ENTRY(wlertx_skb_in_fpm,
			"wlertx_skb_in_fpm", 0644, proc_dointvec),
		NETXL_DEV_SYSCTL_ENTRY(mdqm_rx_id,
			"mdqm_rx_id", 0644, proc_dointvec),
		NETXL_DEV_SYSCTL_ENTRY(mdqm_tx_id,
			"mdqm_tx_id", 0644, proc_dointvec),
		{}
	},
};

int netxl_sysctl_add(struct net_device *dev)
{
	char path[sizeof(NETXL_SYSCTL_PATH) + IFNAMSIZ];
	struct netxl *netxl;
	int i;

	netxl = NETXL_CTX(dev);

	memcpy(&netxl->sysctl, &dev_sysctl, sizeof(struct netxl_sysctl));

#if defined (CONFIG_ARM64)
	for (i = 0; i < ARRAY_SIZE(dev_sysctl.vars) - 1; i++)
		netxl->sysctl.vars[i].data += (u64) netxl;
#else
	for (i = 0; i < ARRAY_SIZE(dev_sysctl.vars) - 1; i++)
		netxl->sysctl.vars[i].data += (u32) netxl;
#endif

	snprintf(path, sizeof(path), NETXL_SYSCTL_PATH"%s",
		 dev->name);
	netxl->sysctl.ctl_hdr = register_sysctl(path,
						netxl->sysctl.vars);
	if (!netxl->sysctl.ctl_hdr)
		pr_err("NEXL: Error create sysctl table");

	return 0;
}

void netxl_sysctl_del(struct net_device *dev)
{
	struct netxl *netxl;

	netxl = NETXL_CTX(dev);
	if (netxl->sysctl.ctl_hdr)
		unregister_sysctl_table(netxl->sysctl.ctl_hdr);
}

int netxl_sys_add(struct net_device *dev)
{
	netxl_sysctl_add(dev);
	netxl_sysfs_add(dev);
	return 0;
}

void netxl_sys_del(struct net_device *dev)
{
	netxl_sysctl_del(dev);
	netxl_sysfs_del(dev);
}
