 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2021 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayesh.patel@broadcom.com>
 ****************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/kthread.h>
#include <linux/rtnetlink.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/if_vlan.h>
#include <net/netevent.h>
#include "netxl.h"
#include "dqnet_priv.h"
#include "dqnet_dbg.h"
#include "mdqm.h"
#include "fpm.h"
#include "interface_defs_rg.h"
#include "host_api_defs.h"
#include "gfap_host_api.h"

#define SLAVE_NETOPS netxl->slave->netdev_ops

#define MAX_DEVS   (16*7)
#define SSID_MASK(id) (id)
#define WL_IF(id) (GFAP_RG_IF_PCIE0+id)
#define APP_IF(id) (GFAP_RG_IF_APP0+id)
#define NETIF_F_DIS_OFF  (NETIF_F_TSO | NETIF_F_TSO6 | NETIF_F_GSO | NETIF_F_GRO)


/* dqm device name */
#define RXQ0 "mdqmrx0"
#define RXQ1 "mdqmrx1"
#define RXQ2 "mdqmrx2"
#define RXQ3 "mdqmrx3"
#define TXQ0 "mdqmtx0"
#define TXQ1 "mdqmtx1"
#define TXQ2 "mdqmtx2"
#define TXQ3 "mdqmtx3"

#define RXQ0WL "mdqmrx0wifi"
#define RXQ1WL "mdqmrx1wifi"
#define RXQ2WL "mdqmrx2wifi"
#define TXQ0WL "mdqmtx0wifi"
#define TXQ1WL "mdqmtx1wifi"
#define TXQ2WL "mdqmtx2wifi"

/* Netxl interface Mapping table: May be we can move it to device tree */
struct netxl_if_map ifmapping[MAX_DEVS] = {
	{.id =   0+16*0, .fap_if = WL_IF(0),  .r = WL_IF(0),  .s = SSID_MASK( 0), .mrx = RXQ0WL, .mtx = TXQ0WL},
	{.id =   1+16*0, .fap_if = WL_IF(0),  .r = WL_IF(0),  .s = SSID_MASK( 1), .mrx = RXQ0WL, .mtx = TXQ0WL},
	{.id =   2+16*0, .fap_if = WL_IF(0),  .r = WL_IF(0),  .s = SSID_MASK( 2), .mrx = RXQ0WL, .mtx = TXQ0WL},
	{.id =   3+16*0, .fap_if = WL_IF(0),  .r = WL_IF(0),  .s = SSID_MASK( 3), .mrx = RXQ0WL, .mtx = TXQ0WL},
	{.id =   4+16*0, .fap_if = WL_IF(0),  .r = WL_IF(0),  .s = SSID_MASK( 4), .mrx = RXQ0WL, .mtx = TXQ0WL},
	{.id =   5+16*0, .fap_if = WL_IF(0),  .r = WL_IF(0),  .s = SSID_MASK( 5), .mrx = RXQ0WL, .mtx = TXQ0WL},
	{.id =   6+16*0, .fap_if = WL_IF(0),  .r = WL_IF(0),  .s = SSID_MASK( 6), .mrx = RXQ0WL, .mtx = TXQ0WL},
	{.id =   7+16*0, .fap_if = WL_IF(0),  .r = WL_IF(0),  .s = SSID_MASK( 7), .mrx = RXQ0WL, .mtx = TXQ0WL},
	{.id =   8+16*0, .fap_if = WL_IF(0),  .r = WL_IF(0),  .s = SSID_MASK( 8), .mrx = RXQ0WL, .mtx = TXQ0WL},
	{.id =   9+16*0, .fap_if = WL_IF(0),  .r = WL_IF(0),  .s = SSID_MASK( 9), .mrx = RXQ0WL, .mtx = TXQ0WL},
	{.id =  10+16*0, .fap_if = WL_IF(0),  .r = WL_IF(0),  .s = SSID_MASK(10), .mrx = RXQ0WL, .mtx = TXQ0WL},
	{.id =  11+16*0, .fap_if = WL_IF(0),  .r = WL_IF(0),  .s = SSID_MASK(11), .mrx = RXQ0WL, .mtx = TXQ0WL},
	{.id =  12+16*0, .fap_if = WL_IF(0),  .r = WL_IF(0),  .s = SSID_MASK(12), .mrx = RXQ0WL, .mtx = TXQ0WL},
	{.id =  13+16*0, .fap_if = WL_IF(0),  .r = WL_IF(0),  .s = SSID_MASK(13), .mrx = RXQ0WL, .mtx = TXQ0WL},
	{.id =  14+16*0, .fap_if = WL_IF(0),  .r = WL_IF(0),  .s = SSID_MASK(14), .mrx = RXQ0WL, .mtx = TXQ0WL},
	{.id =  15+16*0, .fap_if = WL_IF(0),  .r = WL_IF(0),  .s = SSID_MASK(15), .mrx = RXQ0WL, .mtx = TXQ0WL},

	{.id =   0+16*1, .fap_if = WL_IF(1),  .r = WL_IF(1),  .s = SSID_MASK( 0), .mrx = RXQ1WL, .mtx = TXQ1WL},
	{.id =   1+16*1, .fap_if = WL_IF(1),  .r = WL_IF(1),  .s = SSID_MASK( 1), .mrx = RXQ1WL, .mtx = TXQ1WL},
	{.id =   2+16*1, .fap_if = WL_IF(1),  .r = WL_IF(1),  .s = SSID_MASK( 2), .mrx = RXQ1WL, .mtx = TXQ1WL},
	{.id =   3+16*1, .fap_if = WL_IF(1),  .r = WL_IF(1),  .s = SSID_MASK( 3), .mrx = RXQ1WL, .mtx = TXQ1WL},
	{.id =   4+16*1, .fap_if = WL_IF(1),  .r = WL_IF(1),  .s = SSID_MASK( 4), .mrx = RXQ1WL, .mtx = TXQ1WL},
	{.id =   5+16*1, .fap_if = WL_IF(1),  .r = WL_IF(1),  .s = SSID_MASK( 5), .mrx = RXQ1WL, .mtx = TXQ1WL},
	{.id =   6+16*1, .fap_if = WL_IF(1),  .r = WL_IF(1),  .s = SSID_MASK( 6), .mrx = RXQ1WL, .mtx = TXQ1WL},
	{.id =   7+16*1, .fap_if = WL_IF(1),  .r = WL_IF(1),  .s = SSID_MASK( 7), .mrx = RXQ1WL, .mtx = TXQ1WL},
	{.id =   8+16*1, .fap_if = WL_IF(1),  .r = WL_IF(1),  .s = SSID_MASK( 8), .mrx = RXQ1WL, .mtx = TXQ1WL},
	{.id =   9+16*1, .fap_if = WL_IF(1),  .r = WL_IF(1),  .s = SSID_MASK( 9), .mrx = RXQ1WL, .mtx = TXQ1WL},
	{.id =  10+16*1, .fap_if = WL_IF(1),  .r = WL_IF(1),  .s = SSID_MASK(10), .mrx = RXQ1WL, .mtx = TXQ1WL},
	{.id =  11+16*1, .fap_if = WL_IF(1),  .r = WL_IF(1),  .s = SSID_MASK(11), .mrx = RXQ1WL, .mtx = TXQ1WL},
	{.id =  12+16*1, .fap_if = WL_IF(1),  .r = WL_IF(1),  .s = SSID_MASK(12), .mrx = RXQ1WL, .mtx = TXQ1WL},
	{.id =  13+16*1, .fap_if = WL_IF(1),  .r = WL_IF(1),  .s = SSID_MASK(13), .mrx = RXQ1WL, .mtx = TXQ1WL},
	{.id =  14+16*1, .fap_if = WL_IF(1),  .r = WL_IF(1),  .s = SSID_MASK(14), .mrx = RXQ1WL, .mtx = TXQ1WL},
	{.id =  15+16*1, .fap_if = WL_IF(1),  .r = WL_IF(1),  .s = SSID_MASK(15), .mrx = RXQ1WL, .mtx = TXQ1WL},

	{.id =   0+16*2, .fap_if = WL_IF(2),  .r = WL_IF(2),  .s = SSID_MASK( 0), .mrx = RXQ2WL, .mtx = TXQ2WL},
	{.id =   1+16*2, .fap_if = WL_IF(2),  .r = WL_IF(2),  .s = SSID_MASK( 1), .mrx = RXQ2WL, .mtx = TXQ2WL},
	{.id =   2+16*2, .fap_if = WL_IF(2),  .r = WL_IF(2),  .s = SSID_MASK( 2), .mrx = RXQ2WL, .mtx = TXQ2WL},
	{.id =   3+16*2, .fap_if = WL_IF(2),  .r = WL_IF(2),  .s = SSID_MASK( 3), .mrx = RXQ2WL, .mtx = TXQ2WL},
	{.id =   4+16*2, .fap_if = WL_IF(2),  .r = WL_IF(2),  .s = SSID_MASK( 4), .mrx = RXQ2WL, .mtx = TXQ2WL},
	{.id =   5+16*2, .fap_if = WL_IF(2),  .r = WL_IF(2),  .s = SSID_MASK( 5), .mrx = RXQ2WL, .mtx = TXQ2WL},
	{.id =   6+16*2, .fap_if = WL_IF(2),  .r = WL_IF(2),  .s = SSID_MASK( 6), .mrx = RXQ2WL, .mtx = TXQ2WL},
	{.id =   7+16*2, .fap_if = WL_IF(2),  .r = WL_IF(2),  .s = SSID_MASK( 7), .mrx = RXQ2WL, .mtx = TXQ2WL},
	{.id =   8+16*2, .fap_if = WL_IF(2),  .r = WL_IF(2),  .s = SSID_MASK( 8), .mrx = RXQ2WL, .mtx = TXQ2WL},
	{.id =   9+16*2, .fap_if = WL_IF(2),  .r = WL_IF(2),  .s = SSID_MASK( 9), .mrx = RXQ2WL, .mtx = TXQ2WL},
	{.id =  10+16*2, .fap_if = WL_IF(2),  .r = WL_IF(2),  .s = SSID_MASK(10), .mrx = RXQ2WL, .mtx = TXQ2WL},
	{.id =  11+16*2, .fap_if = WL_IF(2),  .r = WL_IF(2),  .s = SSID_MASK(11), .mrx = RXQ2WL, .mtx = TXQ2WL},
	{.id =  12+16*2, .fap_if = WL_IF(2),  .r = WL_IF(2),  .s = SSID_MASK(12), .mrx = RXQ2WL, .mtx = TXQ2WL},
	{.id =  13+16*2, .fap_if = WL_IF(2),  .r = WL_IF(2),  .s = SSID_MASK(13), .mrx = RXQ2WL, .mtx = TXQ2WL},
	{.id =  14+16*2, .fap_if = WL_IF(2),  .r = WL_IF(2),  .s = SSID_MASK(14), .mrx = RXQ2WL, .mtx = TXQ2WL},
	{.id =  15+16*2, .fap_if = WL_IF(2),  .r = WL_IF(2),  .s = SSID_MASK(15), .mrx = RXQ2WL, .mtx = TXQ2WL},

	{.id =   0+16*3, .fap_if = APP_IF(0), .r = APP_IF(0), .s = SSID_MASK( 0), .mrx = RXQ0, .mtx = TXQ0},
	{.id =   1+16*3, .fap_if = APP_IF(0), .r = APP_IF(0), .s = SSID_MASK( 1), .mrx = RXQ0, .mtx = TXQ0},
	{.id =   2+16*3, .fap_if = APP_IF(0), .r = APP_IF(0), .s = SSID_MASK( 2), .mrx = RXQ0, .mtx = TXQ0},
	{.id =   3+16*3, .fap_if = APP_IF(0), .r = APP_IF(0), .s = SSID_MASK( 3), .mrx = RXQ0, .mtx = TXQ0},
	{.id =   4+16*3, .fap_if = APP_IF(0), .r = APP_IF(0), .s = SSID_MASK( 4), .mrx = RXQ0, .mtx = TXQ0},
	{.id =   5+16*3, .fap_if = APP_IF(0), .r = APP_IF(0), .s = SSID_MASK( 5), .mrx = RXQ0, .mtx = TXQ0},
	{.id =   6+16*3, .fap_if = APP_IF(0), .r = APP_IF(0), .s = SSID_MASK( 6), .mrx = RXQ0, .mtx = TXQ0},
	{.id =   7+16*3, .fap_if = APP_IF(0), .r = APP_IF(0), .s = SSID_MASK( 7), .mrx = RXQ0, .mtx = TXQ0},
	{.id =   8+16*3, .fap_if = APP_IF(0), .r = APP_IF(0), .s = SSID_MASK( 8), .mrx = RXQ0, .mtx = TXQ0},
	{.id =   9+16*3, .fap_if = APP_IF(0), .r = APP_IF(0), .s = SSID_MASK( 9), .mrx = RXQ0, .mtx = TXQ0},
	{.id =  10+16*3, .fap_if = APP_IF(0), .r = APP_IF(0), .s = SSID_MASK(10), .mrx = RXQ0, .mtx = TXQ0},
	{.id =  11+16*3, .fap_if = APP_IF(0), .r = APP_IF(0), .s = SSID_MASK(11), .mrx = RXQ0, .mtx = TXQ0},
	{.id =  12+16*3, .fap_if = APP_IF(0), .r = APP_IF(0), .s = SSID_MASK(12), .mrx = RXQ0, .mtx = TXQ0},
	{.id =  13+16*3, .fap_if = APP_IF(0), .r = APP_IF(0), .s = SSID_MASK(13), .mrx = RXQ0, .mtx = TXQ0},
	{.id =  14+16*3, .fap_if = APP_IF(0), .r = APP_IF(0), .s = SSID_MASK(14), .mrx = RXQ0, .mtx = TXQ0},
	{.id =  15+16*3, .fap_if = APP_IF(0), .r = APP_IF(0), .s = SSID_MASK(15), .mrx = RXQ0, .mtx = TXQ0},

	{.id =   0+16*4, .fap_if = APP_IF(1), .r = APP_IF(1), .s = SSID_MASK( 0), .mrx = RXQ1, .mtx = TXQ1},
	{.id =   1+16*4, .fap_if = APP_IF(1), .r = APP_IF(1), .s = SSID_MASK( 1), .mrx = RXQ1, .mtx = TXQ1},
	{.id =   2+16*4, .fap_if = APP_IF(1), .r = APP_IF(1), .s = SSID_MASK( 2), .mrx = RXQ1, .mtx = TXQ1},
	{.id =   3+16*4, .fap_if = APP_IF(1), .r = APP_IF(1), .s = SSID_MASK( 3), .mrx = RXQ1, .mtx = TXQ1},
	{.id =   4+16*4, .fap_if = APP_IF(1), .r = APP_IF(1), .s = SSID_MASK( 4), .mrx = RXQ1, .mtx = TXQ1},
	{.id =   5+16*4, .fap_if = APP_IF(1), .r = APP_IF(1), .s = SSID_MASK( 5), .mrx = RXQ1, .mtx = TXQ1},
	{.id =   6+16*4, .fap_if = APP_IF(1), .r = APP_IF(1), .s = SSID_MASK( 6), .mrx = RXQ1, .mtx = TXQ1},
	{.id =   7+16*4, .fap_if = APP_IF(1), .r = APP_IF(1), .s = SSID_MASK( 7), .mrx = RXQ1, .mtx = TXQ1},
	{.id =   8+16*4, .fap_if = APP_IF(1), .r = APP_IF(1), .s = SSID_MASK( 8), .mrx = RXQ1, .mtx = TXQ1},
	{.id =   9+16*4, .fap_if = APP_IF(1), .r = APP_IF(1), .s = SSID_MASK( 9), .mrx = RXQ1, .mtx = TXQ1},
	{.id =  10+16*4, .fap_if = APP_IF(1), .r = APP_IF(1), .s = SSID_MASK(10), .mrx = RXQ1, .mtx = TXQ1},
	{.id =  11+16*4, .fap_if = APP_IF(1), .r = APP_IF(1), .s = SSID_MASK(11), .mrx = RXQ1, .mtx = TXQ1},
	{.id =  12+16*4, .fap_if = APP_IF(1), .r = APP_IF(1), .s = SSID_MASK(12), .mrx = RXQ1, .mtx = TXQ1},
	{.id =  13+16*4, .fap_if = APP_IF(1), .r = APP_IF(1), .s = SSID_MASK(13), .mrx = RXQ1, .mtx = TXQ1},
	{.id =  14+16*4, .fap_if = APP_IF(1), .r = APP_IF(1), .s = SSID_MASK(14), .mrx = RXQ1, .mtx = TXQ1},
	{.id =  15+16*4, .fap_if = APP_IF(1), .r = APP_IF(1), .s = SSID_MASK(15), .mrx = RXQ1, .mtx = TXQ1},

	{.id =   0+16*5, .fap_if = APP_IF(2), .r = APP_IF(2), .s = SSID_MASK( 0), .mrx = RXQ2, .mtx = TXQ2},
	{.id =   1+16*5, .fap_if = APP_IF(2), .r = APP_IF(2), .s = SSID_MASK( 1), .mrx = RXQ2, .mtx = TXQ2},
	{.id =   2+16*5, .fap_if = APP_IF(2), .r = APP_IF(2), .s = SSID_MASK( 2), .mrx = RXQ2, .mtx = TXQ2},
	{.id =   3+16*5, .fap_if = APP_IF(2), .r = APP_IF(2), .s = SSID_MASK( 3), .mrx = RXQ2, .mtx = TXQ2},
	{.id =   4+16*5, .fap_if = APP_IF(2), .r = APP_IF(2), .s = SSID_MASK( 4), .mrx = RXQ2, .mtx = TXQ2},
	{.id =   5+16*5, .fap_if = APP_IF(2), .r = APP_IF(2), .s = SSID_MASK( 5), .mrx = RXQ2, .mtx = TXQ2},
	{.id =   6+16*5, .fap_if = APP_IF(2), .r = APP_IF(2), .s = SSID_MASK( 6), .mrx = RXQ2, .mtx = TXQ2},
	{.id =   7+16*5, .fap_if = APP_IF(2), .r = APP_IF(2), .s = SSID_MASK( 7), .mrx = RXQ2, .mtx = TXQ2},
	{.id =   8+16*5, .fap_if = APP_IF(2), .r = APP_IF(2), .s = SSID_MASK( 8), .mrx = RXQ2, .mtx = TXQ2},
	{.id =   9+16*5, .fap_if = APP_IF(2), .r = APP_IF(2), .s = SSID_MASK( 9), .mrx = RXQ2, .mtx = TXQ2},
	{.id =  10+16*5, .fap_if = APP_IF(2), .r = APP_IF(2), .s = SSID_MASK(10), .mrx = RXQ2, .mtx = TXQ2},
	{.id =  11+16*5, .fap_if = APP_IF(2), .r = APP_IF(2), .s = SSID_MASK(11), .mrx = RXQ2, .mtx = TXQ2},
	{.id =  12+16*5, .fap_if = APP_IF(2), .r = APP_IF(2), .s = SSID_MASK(12), .mrx = RXQ2, .mtx = TXQ2},
	{.id =  13+16*5, .fap_if = APP_IF(2), .r = APP_IF(2), .s = SSID_MASK(13), .mrx = RXQ2, .mtx = TXQ2},
	{.id =  14+16*5, .fap_if = APP_IF(2), .r = APP_IF(2), .s = SSID_MASK(14), .mrx = RXQ2, .mtx = TXQ2},
	{.id =  15+16*5, .fap_if = APP_IF(2), .r = APP_IF(2), .s = SSID_MASK(15), .mrx = RXQ2, .mtx = TXQ2},

	{.id =   0+16*6, .fap_if = APP_IF(3), .r = APP_IF(3), .s = SSID_MASK( 0), .mrx = RXQ3, .mtx = TXQ3},
	{.id =   1+16*6, .fap_if = APP_IF(3), .r = APP_IF(3), .s = SSID_MASK( 1), .mrx = RXQ3, .mtx = TXQ3},
	{.id =   2+16*6, .fap_if = APP_IF(3), .r = APP_IF(3), .s = SSID_MASK( 2), .mrx = RXQ3, .mtx = TXQ3},
	{.id =   3+16*6, .fap_if = APP_IF(3), .r = APP_IF(3), .s = SSID_MASK( 3), .mrx = RXQ3, .mtx = TXQ3},
	{.id =   4+16*6, .fap_if = APP_IF(3), .r = APP_IF(3), .s = SSID_MASK( 4), .mrx = RXQ3, .mtx = TXQ3},
	{.id =   5+16*6, .fap_if = APP_IF(3), .r = APP_IF(3), .s = SSID_MASK( 5), .mrx = RXQ3, .mtx = TXQ3},
	{.id =   6+16*6, .fap_if = APP_IF(3), .r = APP_IF(3), .s = SSID_MASK( 6), .mrx = RXQ3, .mtx = TXQ3},
	{.id =   7+16*6, .fap_if = APP_IF(3), .r = APP_IF(3), .s = SSID_MASK( 7), .mrx = RXQ3, .mtx = TXQ3},
	{.id =   8+16*6, .fap_if = APP_IF(3), .r = APP_IF(3), .s = SSID_MASK( 8), .mrx = RXQ3, .mtx = TXQ3},
	{.id =   9+16*6, .fap_if = APP_IF(3), .r = APP_IF(3), .s = SSID_MASK( 9), .mrx = RXQ3, .mtx = TXQ3},
	{.id =  10+16*6, .fap_if = APP_IF(3), .r = APP_IF(3), .s = SSID_MASK(10), .mrx = RXQ3, .mtx = TXQ3},
	{.id =  11+16*6, .fap_if = APP_IF(3), .r = APP_IF(3), .s = SSID_MASK(11), .mrx = RXQ3, .mtx = TXQ3},
	{.id =  12+16*6, .fap_if = APP_IF(3), .r = APP_IF(3), .s = SSID_MASK(12), .mrx = RXQ3, .mtx = TXQ3},
	{.id =  13+16*6, .fap_if = APP_IF(3), .r = APP_IF(3), .s = SSID_MASK(13), .mrx = RXQ3, .mtx = TXQ3},
	{.id =  14+16*6, .fap_if = APP_IF(3), .r = APP_IF(3), .s = SSID_MASK(14), .mrx = RXQ3, .mtx = TXQ3},
	{.id =  15+16*6, .fap_if = APP_IF(3), .r = APP_IF(3), .s = SSID_MASK(15), .mrx = RXQ3, .mtx = TXQ3},
};

static int tx_headroom = 0;
module_param(tx_headroom, int, S_IRUGO);
MODULE_PARM_DESC(tx_headroom, "SKB TX headroom");

/* netxl handler for rx packets from slave device for offloading to FAP */
static rx_handler_result_t netxl_handle_frame_fpm(struct sk_buff **pskb)
{
	struct sk_buff *skb = *pskb;
	struct net_device *slave;
	struct net_device *master;
	struct netxl *netxl;
	int status = 0;

	slave = skb->dev;
	master = (struct net_device *)rcu_dereference(slave->rx_handler_data);
	netxl = NETXL_CTX(master);

	if (!netxl->rx_offload) {
		/* Offload is disabled */
		skb->dev = master;
		master->stats.rx_packets++;
		master->stats.rx_bytes += skb->len;

		return RX_HANDLER_ANOTHER;
	} else if (!netxl->rx_type) {
		gfap_host_drvr_rx_msg_t drvr_rx;
		u32 msgdata[DQM_MAX_MSGSZ];
		u32 token;	/* FPM token */
		u32 len;	/* FPM data length */
		u32 offset;	/* FPM data offset */
		u8  *buf;
		int runt_pad_len;
		struct dqnet_netdev *ndev = netdev_priv(master);

		skb_push(skb, ETH_HLEN);

#if LINUX_VERSION_CODE > KERNEL_VERSION(3, 18, 0)
		if (skb_vlan_tag_present(skb)) {
			skb = vlan_insert_tag_set_proto(skb, skb->vlan_proto,
					     skb_vlan_tag_get(skb));
		}
#else
		if (vlan_tx_tag_present(skb)) {
			skb = __vlan_put_tag(skb, skb->vlan_proto,
					     vlan_tx_tag_get(skb));
		}
#endif

		len = skb->len;
		runt_pad_len = len < ETH_ZLEN ? ETH_ZLEN - len : 0;
		token = fpm_alloc_token(len + netxl->head_pad +
					netxl->tail_pad + runt_pad_len);
		if (!fpm_is_valid_token(token))
			goto done;

		buf = fpm_token_to_buffer(token);
		fpm_track_token_tx(token);
		fpm_set_token_size(&token, len+runt_pad_len);

		offset = netxl->head_pad;

		/* FAP specific */
		memset(msgdata, 0, sizeof(msgdata));
		drvr_rx.rx_if = ifmapping[netxl->id].r;
		drvr_rx.rx_pri = 0;
		drvr_rx.rx_sub_if = ifmapping[netxl->id].s;
		drvr_rx.token = token;
		drvr_rx.sop_offset = offset;
		gfap_host_drvr_rx_msg(&drvr_rx, (gfap_pc_dqm_msg_t *)msgdata);
		//msgdata[0] = 0x80000000 | offset;
		//msgdata[1] = token;
		//msgdata[3] = ifmapping[netxl->id].r;
		pr_debug("dqm msgdata: %08x %08x %08x %08x\n",
			 msgdata[0],
			 msgdata[1],
			 msgdata[2],
			 msgdata[3]);
		pr_debug("gfap rx_if    : %d\n"
			 "gfap rx_sub_if: %d\n"
			 "gfap token    : 0x%x\n"
			 "gfap offset   : %d\n",
			 drvr_rx.rx_if,
			 drvr_rx.rx_sub_if,
			 drvr_rx.token,
			 drvr_rx.sop_offset);

		memcpy(buf+offset, skb->data, len);
		memset(buf+offset+len, 0, runt_pad_len);

		fpm_flush_invalidate_token(token, offset, 0, 0);

		if (ndev->msg_enable & NETIF_MSG_RX_STATUS) {
			netif_err(ndev, rx_status, master,
				  "===============================================\n");
			netif_err(ndev, rx_status, master, "NETXL: RX packet -> Offload\n");
			show_rx_pkt(ndev, skb->data, skb->len);
			netif_err(ndev, rx_status, master,
				  "===============================================\n");
			netif_err(ndev, rx_status, master, "NETXL: RX packet -> FPM Offload\n");
			show_rx_pkt(ndev, buf+offset, len);
			netif_err(ndev, rx_status, master,
				  "===============================================\n");
		}

		status = dqm_tx(netxl->dqm_tx_h, 1, netxl->msg_size, msgdata);
		if (status == 0) {
			master->stats.rx_packets++;
			master->stats.rx_bytes += skb->len;
			consume_skb(skb);
			return RX_HANDLER_CONSUMED;
		}

	}
done:
	master->stats.rx_dropped++;
	pr_err(" Error (%d) sending packet to Offlod Q\n", status);

	consume_skb(skb);
	return RX_HANDLER_CONSUMED;
}

static int offload_tx_handler_fpm(int id, u32 *msgdata, void *ctx)
{
	return 0;
}

/* netxl handler for rx packets from offload queue for xmit to slave device */
static int offload_rx_handler_fpm(int id, u32 *msgdata, void *ctx)
{
	struct net_device *dev = (struct net_device *) ctx;
	struct netxl *netxl;
	struct sk_buff *skb;
	int status;
	u32 token;	/* FPM token */
	u32 len;	/* FPM data length */
	u32 offset;	/* FPM data offset */
	u8  *buf;
	gfap_host_drvr_tx_msg_t drvr_tx;
	gfap_pc_dqm_msg_t *msg;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	netxl = NETXL_CTX(dev);

	if (!msgdata)
		return 0;

	msg = (gfap_pc_dqm_msg_t *)msgdata;
	gfap_host_drvr_tx_msg(&drvr_tx, msg);
	pr_debug("dqm msgdata: %08x %08x %08x %08x\n",
		 msgdata[0],
		 msgdata[1],
		 msgdata[2],
		 msgdata[3]);
	pr_debug("gfap tx_if  : %d\n"
		 "gfap tx_meta: 0x%x\n"
		 "gfap token  : 0x%x\n"
		 "gfap offset : %d\n",
		 drvr_tx.tx_if,
		 drvr_tx.tx_meta,
		 drvr_tx.token,
		 drvr_tx.sop_offset);

	if (ifmapping[netxl->id].r != (drvr_tx.tx_meta & 0xFF)) {
		pr_debug("Id invalid %d != %d\n",
			 ifmapping[netxl->id].r,
			 (drvr_tx.tx_meta & 0xFF));
		/* return -1; */
	}

	if (drvr_tx.tx_meta &&
	    (ifmapping[netxl->id].s != ((drvr_tx.tx_meta >> 8) & 0xFF))) {
		pr_debug("Sub id invalid %d != %d\n",
			 ifmapping[netxl->id].s,
			 ((drvr_tx.tx_meta >> 8) & 0xFF));
		return -1;
	}

	token = drvr_tx.token;
	fpm_track_token_rx(token);
	buf = fpm_token_to_buffer(token);
	offset = drvr_tx.sop_offset;
	len = fpm_get_token_size(token);

	if (!fpm_is_fpm_buf(buf)) {
		pr_err("%s: Invalid FPM token buffer %p len=%d",
		      __func__, buf + offset, len);
		dev->stats.tx_errors++;
		return -1;
	}

	fpm_invalidate_token(token, offset, 0,
			     FPM_SYNC_HEAD);

	skb = alloc_skb(netxl->tx_headroom + len, GFP_ATOMIC);
	skb_reserve(skb, netxl->tx_headroom);
	skb->dev = netxl->slave;
	memcpy(skb->data, buf+offset, len);
	skb_put(skb, len);
	fpm_free_token(token);

	if (ndev->msg_enable & NETIF_MSG_TX_DONE) {
		netif_err(ndev, tx_done, dev,
			  "===============================================\n");
		netif_err(ndev, tx_done, dev, "NETXL: TX packet\n");
		show_tx_pkt(ndev, skb->data, skb->len);
		netif_err(ndev, tx_done, dev,
			  "===============================================\n");
	}
	len = skb->len;
	status = dev_queue_xmit(skb);

	dev->stats.tx_packets++;
	dev->stats.tx_bytes += len;

	return status;
}

struct netxl_if_map *netxl_fap_ifmapping(unsigned id)
{
	if (id >= MAX_DEVS)
		return 0;
	return &ifmapping[id];
}


int netxl_rx_handler_register(struct net_device *dev,
			       rx_handler_func_t *rx_handler,
			       void *rx_handler_data)
{
	if (dev->priv_flags & IFF_NO_RX_HANDLER)
		return -EINVAL;

	/* Note: rx_handler_data must be set before rx_handler */
	rcu_assign_pointer(dev->rx_handler_data, rx_handler_data);
	rcu_assign_pointer(dev->rx_handler, rx_handler);

	return 0;
}

void netxl_rx_handler_unregister(struct net_device *dev)
{

	ASSERT_RTNL();
	RCU_INIT_POINTER(dev->rx_handler, NULL);
	/* a reader seeing a non NULL rx_handler in a rcu_read_lock()
	 * section has a guarantee to see a non NULL rx_handler_data
	 * as well.
	 */
	synchronize_net();
	RCU_INIT_POINTER(dev->rx_handler_data, NULL);
}

bool netxl_netdevice_is_slave(struct net_device *dev)
{
	rx_handler_func_t *rx_handler;
	if (!(dev->flags & IFF_SLAVE))
		return false;
	rx_handler = rcu_dereference(dev->rx_handler);
	if (rx_handler == netxl_handle_frame_fpm)
		return true;
	return false;
}

int netxl_fap_netdev_init(struct netxl *netxl, char *pkt_type)
{
	int status = 0;
	struct mdqm_cb cb;
	mdqm_handler_t	mdqm_rx_handler;
	mdqm_handler_t	mdqm_tx_handler;
	rx_handler_func_t *netdev_rx_handler;
	struct netxl_if_map *ifmapping;
	gfap_if_t ifdesc;
	u32 dqm[1];

	ifmapping = netxl_fap_ifmapping(netxl->id);
	gfap_host_interface_desc(ifmapping->fap_if, &ifdesc);
	pr_info("NETXL: if id %d hostdrv queue rx %d tx %d\n",
		ifmapping->fap_if,
		ifdesc.host_drvr_rx_dqm, ifdesc.host_drvr_tx_dqm);

	{
		/* FAP FPM Interface */
		struct fpm_hw_info fpm_info;
		netxl->rx_type = 0;
		netxl->tx_type = 0;
		fpm_get_hw_info(&fpm_info);
		netxl->head_pad = fpm_info.net_buf_head_pad;
		netxl->tail_pad = fpm_info.net_buf_tail_pad;
		netxl->tx_headroom = tx_headroom;
		mdqm_rx_handler = offload_rx_handler_fpm;
		mdqm_tx_handler = offload_tx_handler_fpm;
		netdev_rx_handler = netxl_handle_frame_fpm;
	}

	rtnl_lock();
	status = netxl_rx_handler_register(netxl->slave,
					    netdev_rx_handler,
					    netxl->dev);
	rtnl_unlock();
	if (status) {
		pr_err("Error, Device %s was already enslaved\n",
		       netxl->slave->name);
		return -1;
	}

	dqm[0] = ifdesc.host_drvr_rx_dqm;
	mdqm_create(ifmapping->mrx, "gfap", "rx", 1, dqm);
	dqm[0] = ifdesc.host_drvr_tx_dqm;
	mdqm_create(ifmapping->mtx, "gfap", "tx", 1, dqm);

	cb.handler = mdqm_rx_handler;
	cb.name = netxl->dev->name;
	cb.dev_id = 0;
	cb.ctx = netxl->dev;
	netxl->mdqm_rx_id = mdqm_register(ifmapping->mrx, &cb);
	cb.handler = mdqm_tx_handler;
	netxl->mdqm_tx_id = mdqm_register(ifmapping->mtx, &cb);
	netxl->dqm_tx_h = mdqm_to_dqm_h(ifmapping->mtx, 0);
	netxl->msg_size = dqm_msg_size(netxl->dqm_tx_h);
	netxl->rx_offload = true;

	if (netxl->mdqm_rx_id < 0) {
		pr_err("Error, Invalid RX MDQM %s\n",
		       ifmapping->mrx);
		return -1;
	}

	if (netxl->mdqm_tx_id < 0) {
		pr_err("Error, Invalid TX MDQM %s\n",
		       netxl->ifmapping->mtx);
		mdqm_unregister(ifmapping->mrx, netxl->mdqm_rx_id);
		return -1;
	}

	if ((ifmapping->fap_if >= WL_IF(0)) && (ifmapping->fap_if <= WL_IF(2)))
		dqnet_wifi_open(netxl->dev, ifmapping->fap_if, ifmapping->s);
	else
		dqnet_hostdrv_open(netxl->dev, ifmapping->fap_if, ifmapping->s, 0);

	ifmapping->dev = netxl->dev;
	netxl->ifmapping = ifmapping;

	return 0;
}

int netxl_fap_netdev_exit(struct netxl *netxl)
{
	int sleepcnt = 0;
	struct netxl_if_map *ifmapping;
	while (!rtnl_trylock()) {
		msleep(10);
		if (sleepcnt++ > 100)
			return restart_syscall();
	}

	ifmapping = netxl_fap_ifmapping(netxl->id);

	if ((ifmapping->fap_if >= WL_IF(0)) && (ifmapping->fap_if <= WL_IF(2)))
		dqnet_hostdrv_close(netxl->dev);
	else
		dqnet_wifi_close(netxl->dev);

	netxl_sys_del(netxl->dev);

	unregister_netdevice(netxl->dev);
	netxl_rx_handler_unregister(netxl->slave);
	rtnl_unlock();

	if (netxl->mdqm_rx_id >= 0)
		mdqm_unregister(ifmapping->mrx, netxl->mdqm_rx_id);
	if (netxl->mdqm_tx_id >= 0)
		mdqm_unregister(ifmapping->mtx, netxl->mdqm_tx_id);


	mdqm_delete(ifmapping->mrx);
	mdqm_delete(ifmapping->mtx);

	ifmapping->dev = NULL;
	return 0;
}

int netxl_fap_show(struct netxl *netxl)
{
	pr_info("Mapping Group      : %d\n", netxl->ifmapping->r);
	pr_info("Mapping Sub-Group  : 0x%x\n", netxl->ifmapping->s);
	pr_info("     MDQM RX       : %s\n", netxl->ifmapping->mrx);
	pr_info("     MDQM TX       : %s\n", netxl->ifmapping->mtx);
	pr_info("     MDQM RX  ID   : %d\n", netxl->mdqm_rx_id);
	pr_info("     MDQM TX  ID   : %d\n", netxl->mdqm_tx_id);
	return 0;
}

int netxl_fap_init(void)
{
	return 0;
}

void netxl_fap_exit(void)
{
}
