 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2016 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayesh.patel@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/kthread.h>
#include <linux/rtnetlink.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/if_vlan.h>
#include <linux/pci.h>
#include <linux/pci_ids.h>
#include <net/netevent.h>
#include "netxl.h"
#include "dqnet_priv.h"
#include "dqnet_dbg.h"
#include "mdqm.h"
#if defined(CONFIG_BCM_DQSKB) || defined(CONFIG_BCM_DQSKB_MODULE)
#include "dqskb.h"
#endif
#include "fpm.h"

#include <rdpa_api.h>
#include <rdpa_types.h>
#include <rdpa_cpu_helper.h>

#define SLAVE_NETOPS netxl->slave->netdev_ops

#define MAX_DEVS 48
#define RADIO0   0
#define RADIO1   1
#define RADIO2   2
#define SSID_MASK(id) (1<<id)
#define NETIF_F_DIS_OFF  (NETIF_F_TSO | NETIF_F_TSO6 | NETIF_F_GSO | NETIF_F_GRO)


/* mdqm device name */
#define RXQ0 "mdqmrx0"
#define RXQ1 "mdqmrx1"
#define RXQ2 "mdqmrx2"
#define TXQ0 "mdqmtx"
#define TXQ1 "mdqmtx"
#define TXQ2 "mdqmtx"
#define DQSKB_DEV "dqskb-rfap"

spinlock_t netxl_lock;

/* netxl interface mapping structure */
struct if_id_mapping {
	int id;			/* Mapping ID: Dev ID  */
	int r;			/* Radio match: 0, 1 or 2 */
	int s;			/* SSID match: Bit mask */
	char *mrx;		/* RX MDQM Name */
	char *mtx;		/* TX MDQM Name */
	struct net_device  *dev;/* Netxl device */
};

/* Netxl interface Mapping table: May be we can move it to device tree */
struct if_id_mapping ifmapping[MAX_DEVS] = {
	{.id =  0, .r = RADIO0, .s = SSID_MASK( 0), .mrx = RXQ0, .mtx = TXQ0},
	{.id =  1, .r = RADIO0, .s = SSID_MASK( 1), .mrx = RXQ0, .mtx = TXQ0},
	{.id =  2, .r = RADIO0, .s = SSID_MASK( 2), .mrx = RXQ0, .mtx = TXQ0},
	{.id =  3, .r = RADIO0, .s = SSID_MASK( 3), .mrx = RXQ0, .mtx = TXQ0},
	{.id =  4, .r = RADIO0, .s = SSID_MASK( 4), .mrx = RXQ0, .mtx = TXQ0},
	{.id =  5, .r = RADIO0, .s = SSID_MASK( 5), .mrx = RXQ0, .mtx = TXQ0},
	{.id =  6, .r = RADIO0, .s = SSID_MASK( 6), .mrx = RXQ0, .mtx = TXQ0},
	{.id =  7, .r = RADIO0, .s = SSID_MASK( 7), .mrx = RXQ0, .mtx = TXQ0},
	{.id =  8, .r = RADIO0, .s = SSID_MASK( 8), .mrx = RXQ0, .mtx = TXQ0},
	{.id =  9, .r = RADIO0, .s = SSID_MASK( 9), .mrx = RXQ0, .mtx = TXQ0},
	{.id = 10, .r = RADIO0, .s = SSID_MASK(10), .mrx = RXQ0, .mtx = TXQ0},
	{.id = 11, .r = RADIO0, .s = SSID_MASK(11), .mrx = RXQ0, .mtx = TXQ0},
	{.id = 12, .r = RADIO0, .s = SSID_MASK(12), .mrx = RXQ0, .mtx = TXQ0},
	{.id = 13, .r = RADIO0, .s = SSID_MASK(13), .mrx = RXQ0, .mtx = TXQ0},
	{.id = 14, .r = RADIO0, .s = SSID_MASK(14), .mrx = RXQ0, .mtx = TXQ0},
	{.id = 15, .r = RADIO0, .s = SSID_MASK(15), .mrx = RXQ0, .mtx = TXQ0},

	{.id = 16, .r = RADIO1, .s = SSID_MASK( 0), .mrx = RXQ1, .mtx = TXQ1},
	{.id = 17, .r = RADIO1, .s = SSID_MASK( 1), .mrx = RXQ1, .mtx = TXQ1},
	{.id = 18, .r = RADIO1, .s = SSID_MASK( 2), .mrx = RXQ1, .mtx = TXQ1},
	{.id = 19, .r = RADIO1, .s = SSID_MASK( 3), .mrx = RXQ1, .mtx = TXQ1},
	{.id = 20, .r = RADIO1, .s = SSID_MASK( 4), .mrx = RXQ1, .mtx = TXQ1},
	{.id = 21, .r = RADIO1, .s = SSID_MASK( 5), .mrx = RXQ1, .mtx = TXQ1},
	{.id = 22, .r = RADIO1, .s = SSID_MASK( 6), .mrx = RXQ1, .mtx = TXQ1},
	{.id = 23, .r = RADIO1, .s = SSID_MASK( 7), .mrx = RXQ1, .mtx = TXQ1},
	{.id = 24, .r = RADIO1, .s = SSID_MASK( 8), .mrx = RXQ1, .mtx = TXQ1},
	{.id = 25, .r = RADIO1, .s = SSID_MASK( 9), .mrx = RXQ1, .mtx = TXQ1},
	{.id = 26, .r = RADIO1, .s = SSID_MASK(10), .mrx = RXQ1, .mtx = TXQ1},
	{.id = 27, .r = RADIO1, .s = SSID_MASK(11), .mrx = RXQ1, .mtx = TXQ1},
	{.id = 28, .r = RADIO1, .s = SSID_MASK(12), .mrx = RXQ1, .mtx = TXQ1},
	{.id = 29, .r = RADIO1, .s = SSID_MASK(13), .mrx = RXQ1, .mtx = TXQ1},
	{.id = 30, .r = RADIO1, .s = SSID_MASK(14), .mrx = RXQ1, .mtx = TXQ1},
	{.id = 31, .r = RADIO1, .s = SSID_MASK(15), .mrx = RXQ1, .mtx = TXQ1},

	{.id = 32, .r = RADIO2, .s = SSID_MASK( 0), .mrx = RXQ2, .mtx = TXQ2},
	{.id = 33, .r = RADIO2, .s = SSID_MASK( 1), .mrx = RXQ2, .mtx = TXQ2},
	{.id = 34, .r = RADIO2, .s = SSID_MASK( 2), .mrx = RXQ2, .mtx = TXQ2},
	{.id = 35, .r = RADIO2, .s = SSID_MASK( 3), .mrx = RXQ2, .mtx = TXQ2},
	{.id = 36, .r = RADIO2, .s = SSID_MASK( 4), .mrx = RXQ2, .mtx = TXQ2},
	{.id = 37, .r = RADIO2, .s = SSID_MASK( 5), .mrx = RXQ2, .mtx = TXQ2},
	{.id = 38, .r = RADIO2, .s = SSID_MASK( 6), .mrx = RXQ2, .mtx = TXQ2},
	{.id = 39, .r = RADIO2, .s = SSID_MASK( 7), .mrx = RXQ2, .mtx = TXQ2},
	{.id = 40, .r = RADIO2, .s = SSID_MASK( 8), .mrx = RXQ2, .mtx = TXQ2},
	{.id = 41, .r = RADIO2, .s = SSID_MASK( 9), .mrx = RXQ2, .mtx = TXQ2},
	{.id = 42, .r = RADIO2, .s = SSID_MASK(10), .mrx = RXQ2, .mtx = TXQ2},
	{.id = 43, .r = RADIO2, .s = SSID_MASK(11), .mrx = RXQ2, .mtx = TXQ2},
	{.id = 44, .r = RADIO2, .s = SSID_MASK(12), .mrx = RXQ2, .mtx = TXQ2},
	{.id = 45, .r = RADIO2, .s = SSID_MASK(13), .mrx = RXQ2, .mtx = TXQ2},
	{.id = 46, .r = RADIO2, .s = SSID_MASK(14), .mrx = RXQ2, .mtx = TXQ2},
	{.id = 47, .r = RADIO2, .s = SSID_MASK(15), .mrx = RXQ2, .mtx = TXQ2},
};

static int tx_headroom = 0;
module_param(tx_headroom, int, S_IRUGO);
MODULE_PARM_DESC(tx_headroom, "SKB TX headroom");

#if defined(CONFIG_BCM_DQSKB) || defined(CONFIG_BCM_DQSKB_MODULE)
struct net_device *dqskbdev;

/* netxl handler for rx packets from slave device for offloading to FAP */
static rx_handler_result_t netxl_handle_frame_skb(struct sk_buff **pskb)
{
	struct sk_buff *skb = *pskb;
	struct net_device *slave;
	struct net_device *master;
	struct netxl *netxl;
	int status = 0;

	slave = skb->dev;
	master = (struct net_device *)rcu_dereference(slave->rx_handler_data);
	netxl = NETXL_CTX(master);

	if (!netxl->rx_offload) {
		/* Offload is disabled */
		skb->dev = master;
		master->stats.rx_packets++;
		master->stats.rx_bytes += skb->len;

		return RX_HANDLER_ANOTHER;
	} else if (netxl->rx_type) {
		phys_addr_t paddr;
		dma_addr_t dma;
		u32 msgdata[netxl->msg_size];
		rdpa_cpu_tx_info_t tx_info;
		struct dqnet_netdev *ndev = netdev_priv(master);

		skb_push(skb, ETH_HLEN);

#if LINUX_VERSION_CODE > KERNEL_VERSION(3, 18, 0)
		if (skb_vlan_tag_present(skb)) {
			skb = vlan_insert_tag_set_proto(skb, skb->vlan_proto,
					     skb_vlan_tag_get(skb));
		}
#else
		if (vlan_tx_tag_present(skb)) {
			skb = __vlan_put_tag(skb, skb->vlan_proto,
					     vlan_tx_tag_get(skb));
		}
#endif

		dma = dma_map_single(&master->dev,
				     skb->data,
				     skb->len, DMA_TO_DEVICE);
		paddr = dma;

		/* FAP specific */
		memset(&tx_info, 0, sizeof(rdpa_cpu_tx_info_t));
		tx_info.method = rdpa_cpu_tx_bridge;
		tx_info.port = netxl->id + rdpa_if_ssid0;
		memcpy(&tx_info.data, &paddr,
		       sizeof(tx_info.data));
		tx_info.context = (uint32_t) skb;
		tx_info.data_size = skb->len;
		tx_info.buff_type = rdpa_buf_type_host;

		dqskb_unprepare_skb(dqskbdev, skb);

		if (ndev->msg_enable & NETIF_MSG_RX_STATUS) {
			struct dqnet_netdev *ndev = netdev_priv(master);
			netif_err(ndev, rx_status, master,
				  "===============================================\n");
			netif_err(ndev, rx_status, master, "NETXL: RX packet -> Offload\n");
			show_rx_pkt(ndev, skb->data, skb->len);
			netif_err(ndev, rx_status, master,
				  "===============================================\n");
		}

		status = rdpa_cpu_tx_pd_set(&tx_info, msgdata);
		if (status == 0)
			status = dqm_tx(netxl->dqm_tx_h, 1, netxl->msg_size, msgdata);
		else
			pr_err("%s: Failure calling rdpa_cpu_tx_pd_set",
			       __func__);

		if (status == 0) {
			master->stats.rx_packets++;
			master->stats.rx_bytes += skb->len;
			return RX_HANDLER_CONSUMED;
		}
	}
	master->stats.rx_dropped++;
	pr_err(" Error (%d) sending packet to Offlod Q\n", status);

	consume_skb(skb);
	return RX_HANDLER_CONSUMED;
}
#endif

/* netxl handler for rx packets from slave device for offloading to FAP */
static rx_handler_result_t netxl_handle_frame_fpm(struct sk_buff **pskb)
{
	struct sk_buff *skb = *pskb;
	struct net_device *slave;
	struct net_device *master;
	struct netxl *netxl;
	int status = 0;

	slave = skb->dev;
	master = (struct net_device *)rcu_dereference(slave->rx_handler_data);
	netxl = NETXL_CTX(master);

	if (!netxl->rx_offload) {
		/* Offload is disabled */
		skb->dev = master;
		master->stats.rx_packets++;
		master->stats.rx_bytes += skb->len;

		return RX_HANDLER_ANOTHER;
	} else if (!netxl->rx_type) {
		u32 msgdata[netxl->msg_size];
		rdpa_cpu_tx_info_t tx_info;
		u32 token;	/* FPM token */
		u32 len;	/* FPM data length */
		u32 vlan_len = 0;
		u32 offset;	/* FPM data offset */
		u8  *buf;
		int runt_pad_len;
		struct dqnet_netdev *ndev = netdev_priv(master);

		skb_push(skb, ETH_HLEN);

#if LINUX_VERSION_CODE > KERNEL_VERSION(3, 18, 0)
		if (skb_vlan_tag_present(skb)) {
#else
		if (vlan_tx_tag_present(skb)) {
#endif
			vlan_len = VLAN_HLEN;
			skb_push(skb, VLAN_HLEN);
		}

		len = skb->len + vlan_len;
		runt_pad_len = len < ETH_ZLEN ? ETH_ZLEN - len : 0;
		token = fpm_alloc_token(len + netxl->head_pad +
					netxl->tail_pad + runt_pad_len);
		if (!fpm_is_valid_token(token))
			goto done;

		buf = fpm_token_to_buffer(token);
		fpm_track_token_tx(token);

		offset = netxl->head_pad;

		/* FAP specific */
		memset(&tx_info, 0, sizeof(rdpa_cpu_tx_info_t));
		tx_info.method = rdpa_cpu_tx_bridge;
		tx_info.port = netxl->id + rdpa_if_ssid0;
		tx_info.data_offset = offset;
		tx_info.data_size = len + runt_pad_len;
		tx_info.data = token;

		if (vlan_len) {
			u16 vlan_tci;
			memcpy(buf+offset, skb->data, 2 * ETH_ALEN);
			buf += 2 * ETH_ALEN;
			len -= 2 * ETH_ALEN;
#if LINUX_VERSION_CODE > KERNEL_VERSION(3, 18, 0)
			vlan_tci = htons(skb_vlan_tag_get(skb));
#else
			vlan_tci = htons(vlan_tx_tag_get(skb));
#endif
			memcpy(buf+offset, &skb->vlan_proto, 2);
			memcpy(buf+offset+2, &vlan_tci, 2);
			buf += vlan_len;
			len -= vlan_len;
			skb->data += 2 * ETH_ALEN + vlan_len;
		}

		memcpy(buf+offset, skb->data, len);
		memset(buf+offset+len, 0, runt_pad_len);
		fpm_set_token_size(&token, tx_info.data_size);

		fpm_flush_invalidate_token(token, offset, 0, 0);

		if (ndev->msg_enable & NETIF_MSG_RX_STATUS) {
			netif_err(ndev, rx_status, master,
				  "===============================================\n");
			netif_err(ndev, rx_status, master, "NETXL: RX packet -> Offload\n");
			show_rx_pkt(ndev, skb->data, skb->len);
			netif_err(ndev, rx_status, master,
				  "===============================================\n");
			netif_err(ndev, rx_status, master, "NETXL: RX packet -> FPM Offload\n");
			show_rx_pkt(ndev, buf+offset, tx_info.data_size);
			netif_err(ndev, rx_status, master,
				  "===============================================\n");
		}

		status = rdpa_cpu_tx_pd_set(&tx_info, msgdata);
		if (status == 0)
			status = dqm_tx(netxl->dqm_tx_h, 1, netxl->msg_size, msgdata);
		else
			pr_err("%s: Failure calling rdpa_cpu_tx_pd_set",
			       __func__);

		if (status == 0) {
			master->stats.rx_packets++;
			master->stats.rx_bytes += skb->len;
			consume_skb(skb);
			return RX_HANDLER_CONSUMED;
		}

	}
done:
	master->stats.rx_dropped++;
	pr_err(" Error (%d) sending packet to Offlod Q\n", status);

	consume_skb(skb);
	return RX_HANDLER_CONSUMED;
}

int netxl_open(struct net_device *dev)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	mdqm_enable_id(ifmapping[netxl->id].mrx, netxl->mdqm_rx_id);
	mdqm_enable_id(ifmapping[netxl->id].mtx, netxl->mdqm_tx_id);

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5,4,0))
	return dev_open(netxl->slave, NULL);
#else
	return dev_open(netxl->slave);
#endif
}

int netxl_stop(struct net_device *dev)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	mdqm_disable_id(ifmapping[netxl->id].mrx, netxl->mdqm_rx_id);
	mdqm_disable_id(ifmapping[netxl->id].mtx, netxl->mdqm_tx_id);

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5,4,0))
	dev_close(netxl->slave);
	return 0;
#else
	return dev_close(netxl->slave);
#endif
}

static int netxl_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	dev->stats.tx_packets++;
	dev->stats.tx_bytes += skb->len;

	pr_err("%s: Netxl Xmit\n", dev->name);

	return SLAVE_NETOPS->ndo_start_xmit(skb, netxl->slave);
}

static int netxl_set_mac_address(struct net_device *dev, void *p)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	if (SLAVE_NETOPS->ndo_set_mac_address)
		return SLAVE_NETOPS->ndo_set_mac_address(netxl->slave, p);
	return 0;
}

static int netxl_do_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);
	if (SLAVE_NETOPS->ndo_do_ioctl)
		return SLAVE_NETOPS->ndo_do_ioctl(netxl->slave, rq, cmd);
	return 0;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5,15,0))
static void netxl_tx_timeout(struct net_device *dev, unsigned int txqueue)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	if (SLAVE_NETOPS->ndo_tx_timeout)
		SLAVE_NETOPS->ndo_tx_timeout(netxl->slave, txqueue);
}
#else
static void netxl_tx_timeout(struct net_device *dev)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	if (SLAVE_NETOPS->ndo_tx_timeout)
		SLAVE_NETOPS->ndo_tx_timeout(netxl->slave);
}
#endif

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5,4,0))
static void
netxl_get_stats64(struct net_device *dev, struct rtnl_link_stats64 *stats)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);
	if (SLAVE_NETOPS->ndo_get_stats64)
		SLAVE_NETOPS->ndo_get_stats64(netxl->slave, stats);
}
#else
struct net_device_stats *
netxl_get_stats(struct net_device *dev)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	if (SLAVE_NETOPS->ndo_get_stats)
		return SLAVE_NETOPS->ndo_get_stats(netxl->slave);
	return &dev->stats;
}

static struct rtnl_link_stats64 *
netxl_get_stats64(struct net_device *dev, struct rtnl_link_stats64 *stats)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);
	if (SLAVE_NETOPS->ndo_get_stats64)
		return SLAVE_NETOPS->ndo_get_stats64(netxl->slave, stats);
	return NULL;
}
#endif

/*
static int netxl_set_mac_addr(struct net_device *dev, void *p)
{
	struct sockaddr *addr = (struct sockaddr *)p;

	if (netif_running(dev)) {
		netdev_err(dev, "Device busy.\n");
		return -EBUSY;
	}

	memcpy(dev->dev_addr, addr->sa_data, dev->addr_len);

	return 0;
}
*/

static void netxl_change_rx_flags(struct net_device *dev, int flags)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	if (SLAVE_NETOPS->ndo_change_rx_flags)
		return SLAVE_NETOPS->ndo_change_rx_flags(netxl->slave, flags);
}

static void netxl_set_rx_mode(struct net_device *dev)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	if (SLAVE_NETOPS->ndo_set_rx_mode) {
		if ((dev->flags & IFF_PROMISC) && !(netxl->slave->flags & IFF_PROMISC)) {
			netxl->slave->flags = (dev->flags & ~IFF_MASTER) | IFF_SLAVE;
			dev_set_promiscuity(netxl->slave, 1);
		} else {
			netxl->slave->flags = (dev->flags & ~IFF_MASTER) | IFF_SLAVE;
			SLAVE_NETOPS->ndo_set_rx_mode(netxl->slave);
		}
	}
}

static int netxl_change_mtu(struct net_device *dev, int new_mtu)
{
	struct netxl *netxl;
	int ret = 0;
	netxl = NETXL_CTX(dev);

	if (SLAVE_NETOPS->ndo_change_mtu) {
		ret = SLAVE_NETOPS->ndo_change_mtu(netxl->slave, new_mtu);
		if (ret == 0) {
			dev->mtu = new_mtu;
		}
	}
	return ret;
}

static int netxl_set_features(struct net_device *dev,
			      netdev_features_t features)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	/* Avoid calling this in netdev register call */
	if (dev->state == __LINK_STATE_START)
		return 0;

	/* Clear offload feature */
	features = features & ~NETIF_F_DIS_OFF ;
	if (SLAVE_NETOPS->ndo_set_features)
		return SLAVE_NETOPS->ndo_set_features(netxl->slave, features);
	return 0;
}

static netdev_features_t netxl_fix_features(struct net_device *dev,
					    netdev_features_t features)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	/* Avoid calling this in netdev register call */
	if (dev->state == __LINK_STATE_START)
		return 0;

	if (SLAVE_NETOPS->ndo_fix_features)
		features = SLAVE_NETOPS->ndo_fix_features(netxl->slave,
							  features);
	/* Clear offload feature */
	return features & ~NETIF_F_DIS_OFF ;
}

static int netxl_vlan_rx_add_vid(struct net_device *dev,
				 __be16 proto, u16 vid)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);
	if (SLAVE_NETOPS->ndo_vlan_rx_add_vid)
		return SLAVE_NETOPS->ndo_vlan_rx_add_vid(netxl->slave, proto, vid);
	return 0;
}

static int netxl_vlan_rx_kill_vid(struct net_device *dev,
				  __be16 proto, u16 vid)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);
	if (SLAVE_NETOPS->ndo_vlan_rx_add_vid)
		return SLAVE_NETOPS->ndo_vlan_rx_kill_vid(netxl->slave, proto, vid);
	return 0;
}

#if defined(CONFIG_BCM_DQSKB) || defined(CONFIG_BCM_DQSKB_MODULE)
int offload_tx_handler_skb(int id, u32 *msgdata, void *ctx)
{
	return 0;
}
#endif

int offload_tx_handler_fpm(int id, u32 *msgdata, void *ctx)
{
	return 0;
}

#if defined(CONFIG_BCM_DQSKB) || defined(CONFIG_BCM_DQSKB_MODULE)
/* netxl handler for rx packets from offload queue for xmit to slave device */
int offload_rx_handler_skb(int id, u32 *msgdata, void *ctx)
{
	struct net_device *dev = (struct net_device *) ctx;
	struct netxl *netxl;
	struct sk_buff *skb;
	int status;
	rdpa_cpu_rx_info_t rx_info;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	netxl = NETXL_CTX(dev);

	if (!msgdata)
		return 0;

	memset(&rx_info, 0, sizeof(rdpa_cpu_rx_info_t));
	status = rdpa_cpu_rx_pd_get(msgdata, &rx_info);

	if (status) {
		pr_err("%s: Failure calling rdpa_cpu_rx_pd_get.", __func__);
	} else if (rx_info.buff_type == rdpa_buf_type_host) {

		if (ifmapping[netxl->id].r != rx_info.radio_idx)
			return -1;

		if (ifmapping[netxl->id].s != rx_info.dest_ssid)
			return -1;

		skb = (struct sk_buff *)rx_info.data;
		skb_put(skb, rx_info.size);
		if (skb->dev) {
			dqskbdev = skb->dev;
			dqskb_prepare_skb(dqskbdev, skb);
		}
		dqskb_invalidate_buffer(skb->data, skb->len);

		dev->stats.tx_packets++;
		dev->stats.tx_bytes += skb->len;

		if (ndev->msg_enable & NETIF_MSG_TX_DONE) {
			netif_err(ndev, tx_done, dev,
				  "===============================================\n");
			netif_err(ndev, tx_done, dev, "NETXL: TX packet\n");
			show_tx_pkt(ndev, skb->data, skb->len);
			netif_err(ndev, tx_done, dev,
				  "===============================================\n");
		}

		skb->dev = netxl->slave;
		return dev_queue_xmit(skb);
	} else {
		pr_err("offload_rx_handler_skb: Invalid buffer\n");
	}

	dev->stats.tx_errors++;
	return 0;
}
#endif

#define GET_WLER_CNTXT(skb)	(skb + sizeof(struct sk_buff));
#define WLER_CNTXT_TAG			0xddaaee11
#define CONTEXT_MASK			0x00ffffff
#define DIR_NIBBLE_SHIFT		28
#define IF_NIBBLE_SHIFT			24
#define WL_DIR_RX			0
#define WL_DIR_TX			1

struct sk_buff *wlertx_netxl_build_skb(u8 *buf, u32 len, u32 offset,
				       u32 pad_size, u32 token, u32 idx)
{
	u32 *wler_cntxt;
	struct skb_shared_info *shinfo;
	struct sk_buff *skb;
	u32 skb_sharedinfo_size = (pad_size - sizeof(struct sk_buff));

	skb = (struct sk_buff *)((u32)buf + offset + len + skb_sharedinfo_size);
	skb = PTR_ALIGN(skb, sizeof(u32));
	memset(skb, 0, offsetof(struct sk_buff, tail));


	memset((void *)&skb->cb[0], 0, 48);
	skb->truesize = (offset + len + pad_size);
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5,4,0))
	refcount_set(&skb->users, 1);
#else
	atomic_set(&skb->users, 1);
#endif
	skb->head = &buf[0];
	skb->data = skb->head;
	skb->tail = skb->data;
	skb->len = 0;
	skb->end = skb->tail + len + offset;
	skb->end = PTR_ALIGN(skb->end, sizeof(u32));
	skb->mac_header = (typeof(skb->mac_header))~0U;
	skb->transport_header = (typeof(skb->transport_header))~0U;
	skb->cloned = 0;
	skb->next = NULL;
	skb_reserve(skb, offset);

	shinfo = skb_shinfo(skb);
	memset(shinfo, 0, offsetof(struct skb_shared_info, dataref));
	atomic_set(&shinfo->dataref, 1);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(5,4,0))
	kmemcheck_annotate_variable(shinfo->destructor_arg);
#endif
	wler_cntxt = (u32 *)GET_WLER_CNTXT((u32)skb);
	wler_cntxt[0] = token;
	wler_cntxt[1] = (WLER_CNTXT_TAG & CONTEXT_MASK);

	wler_cntxt[1] |= (((WL_DIR_TX << DIR_NIBBLE_SHIFT) & 0xf0000000) |
			  ((idx & 0x0f) << IF_NIBBLE_SHIFT));

	return skb;
}

/* netxl handler for rx packets from offload queue for xmit to slave device */
int offload_rx_handler_fpm(int id, u32 *msgdata, void *ctx)
{
	struct net_device *dev = (struct net_device *) ctx;
	struct netxl *netxl;
	struct sk_buff *skb;
	int status;
	rdpa_cpu_rx_info_t rx_info;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	netxl = NETXL_CTX(dev);

	if (!msgdata)
		return 0;

	memset(&rx_info, 0, sizeof(rdpa_cpu_rx_info_t));
	status = rdpa_cpu_rx_pd_get(msgdata, &rx_info);

	if (status) {
		pr_err("%s: Failure calling rdpa_cpu_rx_pd_get.", __func__);
	} else if (rx_info.buff_type == rdpa_buf_type_rdp) {
		u32 token;	/* FPM token */
		u32 len;	/* FPM data length */
		u32 offset;	/* FPM data offset */
		u8  *buf;

		if (ifmapping[netxl->id].r != rx_info.radio_idx)
			return -1;

		if (ifmapping[netxl->id].s != rx_info.dest_ssid)
			return -1;

		token = rx_info.data;
		fpm_track_token_rx(token);
		buf = fpm_token_to_buffer(token);
		offset = rx_info.data_offset;
		len = rx_info.size;

		if (!fpm_is_fpm_buf_wler(buf)) {
			pr_err("%s: Invalid FPM token buffer %p len=%d",
			      __func__, buf, len);
			return -1;
		}

		if (len != rx_info.size) {
			pr_err("%s: Token size does not match pkt desc size len=%d rxinfo_sz=%d",
			       __func__, len, rx_info.size);
			goto err;
		}

		if (netxl->wlertx_skb_in_fpm && !is_multicast_ether_addr(buf+offset)) {
			int buf_size;

			buf_size = fpm_get_min_buffer_size(offset + len + netxl->tail_pad);
			fpm_set_token_size(&token, buf_size - 1);
			fpm_invalidate_token(token, 0, 1, FPM_SYNC_TAIL);
			/*fpm_invalidate_buffer_wler(buf+offset, len);*/

			skb = wlertx_netxl_build_skb(buf, len, offset, netxl->tail_pad,
						     token, rx_info.radio_idx);
			skb->dev = netxl->slave;
			skb_put(skb, len);

			if (ndev->msg_enable & NETIF_MSG_TX_DONE) {
				netif_err(ndev, tx_done, dev,
					  "===============================================\n");
				netif_err(ndev, tx_done, dev, "NETXL: TX packet\n");
				show_tx_pkt(ndev, skb->data, skb->len);
				netif_err(ndev, tx_done, dev,
					  "===============================================\n");
			}
			status = SLAVE_NETOPS->ndo_start_xmit(skb, netxl->slave);
			if (status)
				status = dev_queue_xmit(skb);
		} else {
			fpm_invalidate_buffer_wler(buf, len+offset);
			skb = alloc_skb(netxl->tx_headroom + len, GFP_ATOMIC);
			skb_reserve(skb, netxl->tx_headroom);
			skb->dev = netxl->slave;
			memcpy(skb->data, buf+offset, len);
			skb_put(skb, len);
			fpm_free_token(token);

			if (ndev->msg_enable & NETIF_MSG_TX_DONE) {
				netif_err(ndev, tx_done, dev,
					  "===============================================\n");
				netif_err(ndev, tx_done, dev, "NETXL: TX packet\n");
				show_tx_pkt(ndev, skb->data, skb->len);
				netif_err(ndev, tx_done, dev,
					  "===============================================\n");
			}
			status = dev_queue_xmit(skb);
		}

		dev->stats.tx_packets++;
		dev->stats.tx_bytes += skb->len;


		return status;

err:
		fpm_free_token(token);

	} else {
		pr_err("offload_rx_handler_fpm: Invalid buffer\n");
	}

	dev->stats.tx_errors++;
	return 0;
}

 static struct net_device_ops netxl_netdev_ops = {
	.ndo_open		= netxl_open,
	.ndo_stop		= netxl_stop,
	.ndo_start_xmit		= netxl_start_xmit,
	.ndo_change_rx_flags    = netxl_change_rx_flags,
	.ndo_set_rx_mode	= netxl_set_rx_mode,
	.ndo_set_mac_address	= netxl_set_mac_address,
	.ndo_validate_addr      = eth_validate_addr,
	.ndo_do_ioctl		= netxl_do_ioctl,
	.ndo_change_mtu         = netxl_change_mtu,
	.ndo_tx_timeout		= netxl_tx_timeout,
#if (LINUX_VERSION_CODE < KERNEL_VERSION(5,4,0))
	.ndo_get_stats		= netxl_get_stats,
#endif
	.ndo_get_stats64	= netxl_get_stats64,
	.ndo_fix_features	= netxl_fix_features,
	.ndo_set_features	= netxl_set_features,
	.ndo_vlan_rx_add_vid    = netxl_vlan_rx_add_vid,
	.ndo_vlan_rx_kill_vid   = netxl_vlan_rx_kill_vid,
};

/* Update netxl netdev ops */
void netxl_copy_netdev_ops(struct net_device *dst,
			   struct net_device *src)
{
	int i;
	funcptr *p1;
	funcptr *p2;
	dst->netdev_ops = kmemdup(&netxl_netdev_ops,
				  sizeof(struct net_device_ops),
				  GFP_KERNEL);
	p1 = (funcptr *) dst->netdev_ops;
	p2 = (funcptr *) src->netdev_ops;
	for (i = 0; i < (sizeof(struct net_device_ops)/sizeof(funcptr)); i++) {
		/* If src function is null then make dst function null */
		if (!*p2)
			*p1 = 0;
		p2++;
		p1++;
	}
}

/* netxl device internal counters */
void netxl_netdevice_show(char *master_name)
{
	struct net_device *master;
	struct netxl *netxl;

	master = __dev_get_by_name(&init_net, master_name);

	if (!master)
		return;

	netxl = NETXL_CTX(master);

	if (!netxl)
		return;

	pr_info("Master        : %s\n", master->name);
	pr_info(" Flags        : %X\n", master->flags);
	pr_info(" Features         : %llX\n", master->features);
	pr_info(" HW Features      : %llX\n", master->hw_features);
	pr_info(" Wanted Features  : %llX\n", master->wanted_features);
	pr_info(" VLAN Features    : %llX\n", master->vlan_features);
	pr_info("Slave         : %s\n", netxl->slave->name);
	pr_info(" Flags        : %X\n", netxl->slave->flags);
	pr_info(" Features         : %llX\n", netxl->slave->features);
	pr_info(" HW Features      : %llX\n", netxl->slave->hw_features);
	pr_info(" Wanted Features  : %llX\n", netxl->slave->wanted_features);
	pr_info(" VLAN Features    : %llX\n", netxl->slave->vlan_features);
	pr_info("Dev ID        : %d\n", netxl->id);
	pr_info("Rx_offload    : %d\n", netxl->rx_offload);
	pr_info("Tx_bypass     : %d\n", netxl->tx_bypass);
	pr_info("FPM in SKB    : %d\n", netxl->wlertx_skb_in_fpm);
	pr_info("Rx type       : %s\n", netxl->rx_type ? "skb":"fpm");
	pr_info("Tx type       : %s\n", netxl->tx_type ? "skb":"fpm");
	pr_info("Head_pad      : %d\n", netxl->head_pad);
	pr_info("Tail_pad      : %d\n", netxl->tail_pad);
	pr_info("TX Headroom   : %d\n", netxl->tx_headroom);
	pr_info("Mapping Group      : %d\n", ifmapping[netxl->id].r);
	pr_info("Mapping Sub-Group  : 0x%x\n", ifmapping[netxl->id].s);
	pr_info("     MDQM RX       : %s\n", ifmapping[netxl->id].mrx);
	pr_info("     MDQM TX       : %s\n", ifmapping[netxl->id].mtx);
	pr_info("     MDQM RX  ID   : %d\n", netxl->mdqm_rx_id);
	pr_info("     MDQM TX  ID   : %d\n", netxl->mdqm_tx_id);
#if defined(CONFIG_BCM_DQSKB) || defined(CONFIG_BCM_DQSKB_MODULE)
	pr_info("DQSKB Dev     : %s\n", dqskbdev ? dqskbdev->name : "Unknown");
#endif
	pr_info("Stats:\n");
	pr_info("RX packets: %lu, bytes: %lu errors: %lu dropped: %lu\n",
		master->stats.rx_packets, master->stats.rx_bytes,
		master->stats.rx_errors, master->stats.rx_dropped);
	pr_info("TX packets: %lu, bytes: %lu errors: %lu dropped: %lu\n",
		master->stats.tx_packets, master->stats.tx_bytes,
		master->stats.tx_errors, master->stats.tx_dropped);
	return;
}

/* */
static int rfap_init_wlan_queue(struct net_device *dev, int queue_id, int type)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	rdpa_cpu_rxq_cfg_t rxq_cfg;
	bdmf_object_handle rdpa_cpu_obj;
	int rc;

	rc = rdpa_cpu_get(rdpa_cpu_wlan0, &rdpa_cpu_obj);
	if (rc < 0) {

		BDMF_MATTR(cpu_attrs, rdpa_cpu_drv());

		rdpa_cpu_index_set(cpu_attrs, rdpa_cpu_wlan0);
		rc = bdmf_new_and_set(rdpa_cpu_drv(), NULL, cpu_attrs,
				      &rdpa_cpu_obj);
		if (rc < 0) {
			pr_err("%s: bdmf_new_and_set rdpa_cpu obj failed\n",
			       __func__);
			return rc;
		}
	}

	rc = rdpa_cpu_rxq_cfg_get(rdpa_cpu_obj, queue_id, &rxq_cfg);
	if (rc < 0) {
		pr_err(" rdpa_cpu_rxq_cfg_get error %d:  %s.\n",
		       rc, ndev->name);
		return rc;
	}
	if (rxq_cfg.buff_type != type) {
		rxq_cfg.buff_type = type;
		rc = rdpa_cpu_rxq_cfg_set(rdpa_cpu_obj,
					  queue_id, &rxq_cfg);
	}
	if (rc < 0) {
		pr_err(" rdpa_cpu_rxq_cfg_set error %d:  %s.\n",
		       rc, ndev->name);
		return rc;
	}
	return rc;
}

/* TX nethook to bypass TX path directly to slave device instead of
   passing through FAP */
static enum bcm_nethook_result bcm_nethook_tx(
	struct net_device *out, enum bcm_nethook_type type, void *buf)
{
	struct sk_buff *skb = (struct sk_buff *)buf;
	struct netxl *netxl;
	netxl = NETXL_CTX(out);

	if (netxl->tx_bypass) {
		SLAVE_NETOPS->ndo_start_xmit(skb, netxl->slave);
		return BCM_NETHOOK_CONSUMED;
	}
	return BCM_NETHOOK_PASS;
}

/* Configure netxl device parameters */
int netxl_netdevice_cfg(char *master_name, char *param,
			int val)
{
	struct net_device *master;
	struct netxl *netxl;
	int status = 0;
	unsigned long flags;

	spin_lock_irqsave(&netxl_lock, flags);

	master = __dev_get_by_name(&init_net, master_name);

	if (!master) {
		pr_err("Error, Master %s not found\n", master_name);
		status = -EINVAL;
		goto done;
	}

	if (!(master->flags & IFF_MASTER)) {
		pr_err("Error, Device %s is not a master\n", master_name);
		status = -EINVAL;
		goto done;
	}

	netxl = NETXL_CTX(master);

	if (!netxl) {
		pr_err("Error, Netxl not found\n");
		status = -EINVAL;
		goto done;
	}

	if (!strcmp(param, "tx_headroom")) {
		netxl->tx_headroom = val;
	}

done:
	spin_unlock_irqrestore(&netxl_lock, flags);
	return status;
}

int netxl_netdev_update_features(struct net_device *dev)
{
	netdev_features_t features;
	int err = 0;

	ASSERT_RTNL();

	features = netdev_get_wanted_features(dev);

	if (dev->netdev_ops->ndo_fix_features)
		features = dev->netdev_ops->ndo_fix_features(dev, features);

	if (dev->features == features)
		return 0;

	netdev_dbg(dev, "Features changed: %pNF -> %pNF\n",
		&dev->features, &features);

	if (dev->netdev_ops->ndo_set_features)
		err = dev->netdev_ops->ndo_set_features(dev, features);

	if (unlikely(err < 0)) {
		netdev_err(dev,
			"set_features() failed (%d); wanted %pNF, left %pNF\n",
			err, &features, &dev->features);
		return -1;
	}

	if (!err)
		dev->features = features;

	return 1;
}

bool netxl_netdevice_is_wler(struct net_device *net)
{
	struct ethtool_drvinfo info;
	if (!net->ethtool_ops)
		return false;
	if (!net->ethtool_ops->get_drvinfo)
		return false;
	memset(&info, 0, sizeof(info));
	net->ethtool_ops->get_drvinfo(net, &info);
	if (strncmp(info.reserved2, "wlerouter", sizeof("wlerouter")) == 0)
		return true;
	return false;
}

#define PCI_VENDOR_ID_QUANTENNA 0x1bb5
#define PCI_VENDOR_ID_CELENO1   0x1d59
#define PCI_VENDOR_ID_CELENO2   0x1d69

const struct pci_device_id netxl_allowed_pci_ids[] = {
	{ PCI_DEVICE(PCI_VENDOR_ID_BROADCOM,  PCI_ANY_ID) },
	{ PCI_DEVICE(PCI_VENDOR_ID_QUANTENNA, PCI_ANY_ID) },
	{ PCI_DEVICE(PCI_VENDOR_ID_CELENO1,   PCI_ANY_ID) },
	{ PCI_DEVICE(PCI_VENDOR_ID_CELENO2,   PCI_ANY_ID) },
	{}
};

const struct pci_device_id netxl_any_pci_ids[] = {
	{ PCI_DEVICE(PCI_ANY_ID,    PCI_ANY_ID) },
	{}
};

int netxl_is_pci_allowed(struct pci_dev *dev)
{
	int i;

	for (i = 0; i < sizeof(netxl_allowed_pci_ids)/sizeof(struct pci_device_id); i++) {
		if (dev->vendor == netxl_allowed_pci_ids[i].vendor)
			return true;
	}
	return false;
}

int netxl_is_allowed(struct net_device *slave)
{
	struct ethtool_drvinfo info;
	struct pci_dev *dev = NULL;

	if (!pci_dev_present(netxl_any_pci_ids)) {
		/* If no PCI devices are present then allow netxl
		   to be used on non PCI network devices like USB */
		if (slave->ethtool_ops && slave->ethtool_ops->get_drvinfo) {
			slave->ethtool_ops->get_drvinfo(slave, &info);
			if (strstr(info.bus_info, "usb"))
				return true;
		}
	} else {
		for_each_pci_dev(dev) {
			if (!netxl_is_pci_allowed(dev)) {
				pci_dev_put(dev);
				return false;
			}
		}
	}
	return true;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5,4,0))
static u64 netxl_dmamask = DMA_BIT_MASK(32);
#endif

/* Add netxl device */
int netxl_netdevice_add(char *master_name, char *slave_name,
			int id, char *pkt_type)
{
	struct net_device *slave;
	struct net_device *master;
	struct netxl *netxl;
	struct mdqm_cb cb;
	int status = 0;
	unsigned long flags;
	mdqm_handler_t	mdqm_rx_handler;
	mdqm_handler_t	mdqm_tx_handler;
	rx_handler_func_t *netdev_rx_handler;
	netdev_features_t mask;

	spin_lock_irqsave(&netxl_lock, flags);
	/* Check if ID is available */
	if ((id < 0) || (id >= MAX_DEVS)) {
		pr_err("Error, Invalid Mapping ID %d (should be between 0 and %d)\n",
		       id, MAX_DEVS);
		status = -EINVAL;
		goto done;
	}

	if (!strcmp(master_name, slave_name)) {
		pr_err("Error, Master name should not be same as slave name\n");
		status = -EINVAL;
		goto done;
	}

	if (ifmapping[id].dev) {
		pr_err("Error, Mapping ID %d already in use by %s\n", id,
		       ifmapping[id].dev->name);
		status = -EINVAL;
		goto done;
	}

	slave = __dev_get_by_name(&init_net, slave_name);

	if (!slave) {
		pr_err("Error, Slave device %s not found\n", slave_name);
		status = -EINVAL;
		goto done;
	}

	if (!netxl_is_allowed(slave)) {
		pr_err("Error, Unsupported PCI device is present\n");
		status = -EPERM;
		goto done;
	}

	if (slave->flags & IFF_UP) {
		pr_err("Error, Up Device %s cannot be enslaved\n", slave_name);
		status = -EINVAL;
		goto done;
	}

	if (slave->flags & IFF_MASTER) {
		pr_err("Error, Device %s is master device. It cannot be enslaved\n",
		       slave_name);
		status = -EINVAL;
		goto done;
	}

	if (slave->flags & IFF_SLAVE) {
		pr_err("Error, Device %s was already enslaved\n", slave_name);
		status = -EINVAL;
		goto done;
	}

	if (slave->rx_handler) {
		pr_err("Error, Device %s was already enslaved\n", slave_name);
		status = -EINVAL;
		goto done;
	}

	netxl = kzalloc(sizeof(struct netxl), GFP_KERNEL);
	if (!netxl) {
		pr_err("Error, Allocating Netxl Priv Data\n");
		status = -EINVAL;
		goto done;
	}

	master = alloc_etherdev(sizeof(struct dqnet_netdev));
	if (!master) {
		pr_err("Error, Allocating Master Device\n");
		status = -EINVAL;
		kfree(netxl);
		goto done;
	}

	slave->flags |= IFF_SLAVE;
	master->flags |= IFF_MASTER;

	/* Disable offload features */
	mask = NETIF_F_DIS_OFF ;
	if (mask) {
		slave->wanted_features = slave->hw_features;
		slave->wanted_features &= ~mask;
		rtnl_lock();
		netxl_netdev_update_features(slave);
		rtnl_unlock();
		slave->wanted_features = 0;
	}

	NETXL_CTX(master) = netxl;
	netxl_copy_netdev_ops(master, slave);
	netxl->dev = master;
	netxl->slave = slave;

	snprintf(master->name, IFNAMSIZ, "%s", master_name);

	memcpy(master->dev_addr, slave->dev_addr, master->addr_len);

	/*
	if (!(master->dev_addr[0] & 0x2)) {
		master->dev_addr[0] |= 0x2;
	}
	*/

	netxl_copy_ethtool_ops(master, slave);
	if (register_netdev(master))
		pr_err("Failed to register master network device.\n");

	netif_carrier_off(master);

	dev_hold(slave);
	netxl_sys_add(master);

#if defined(CONFIG_BCM_DQSKB) || defined(CONFIG_BCM_DQSKB_MODULE)
	if (strncmp(pkt_type, "skb", 3) == 0) {
		/* FAP SKB Interface */
		netxl->rx_type = 1;
		netxl->tx_type = 1;
		mdqm_rx_handler = offload_rx_handler_skb;
		mdqm_tx_handler = offload_tx_handler_skb;
		netdev_rx_handler = netxl_handle_frame_skb;
	} else {
#else
	{
#endif
		/* FAP FPM Interface */
		struct fpm_hw_info fpm_info;
		netxl->rx_type = 1;
		netxl->tx_type = 0;
		fpm_get_hw_info(&fpm_info);
		netxl->head_pad = fpm_info.net_buf_head_pad;
		netxl->tail_pad = fpm_info.net_buf_tail_pad;
		netxl->tx_headroom = tx_headroom;
		mdqm_rx_handler = offload_rx_handler_fpm;
		mdqm_tx_handler = offload_tx_handler_fpm;
		if (netxl->rx_type)
			netdev_rx_handler = netxl_handle_frame_skb;
		else
			netdev_rx_handler = netxl_handle_frame_fpm;
	}

	rtnl_lock();
	status = netdev_rx_handler_register(slave,
					    netdev_rx_handler,
					    master);
	rtnl_unlock();
	if (status) {
		pr_err("Error, Device %s was already enslaved\n",
		       slave_name);
		goto device_cleanup;
	}

	cb.handler = mdqm_rx_handler;
	cb.name = master->name;
	cb.dev_id = 0;
	cb.ctx = master;
	netxl->mdqm_rx_id = mdqm_register(ifmapping[id].mrx, &cb);
	cb.handler = mdqm_tx_handler;
	netxl->mdqm_tx_id = mdqm_register(ifmapping[id].mtx, &cb);
	netxl->dqm_tx_h = mdqm_to_dqm_h(ifmapping[id].mtx, 0);
	netxl->msg_size = dqm_msg_size(netxl->dqm_tx_h);
	netxl->rx_offload = true;

	if (netxl->mdqm_rx_id < 0) {
		pr_err("Error, Invalid RX MDQM %s\n",
		       ifmapping[id].mrx);
		goto device_cleanup;
	}

	if (netxl->mdqm_tx_id < 0) {
		pr_err("Error, Invalid TX MDQM %s\n",
		       ifmapping[id].mtx);
		mdqm_unregister(ifmapping[netxl->id].mrx, netxl->mdqm_rx_id);
		goto device_cleanup;
	}

	if ((netxl->mdqm_rx_id >= 0) && (netxl->mdqm_tx_id >= 0))
		dqnet_wifi_open(master, id + rdpa_if_ssid0, 0);

	/* Register tx nethook after flowmgr */
	bcm_nethook_register_hook(master, BCM_NETHOOK_TX_SKB,
				  TX_SKB_PRIO_FLOWMGR+1, "NetXL",
				  bcm_nethook_tx);
	bcm_nethook_enable_hook(master, BCM_NETHOOK_TX_SKB,
				bcm_nethook_tx, true);

	ifmapping[id].dev = master;
	netxl->id  = id;

	if (id < 8) {
		rfap_init_wlan_queue(master, 8, netxl->tx_type);
		rfap_init_wlan_queue(master, 9, netxl->tx_type);
	} else if (id < 16) {
		rfap_init_wlan_queue(master, 10, netxl->tx_type);
		rfap_init_wlan_queue(master, 11, netxl->tx_type);
	} else if (id < 24) {
		rfap_init_wlan_queue(master, 12, netxl->tx_type);
		rfap_init_wlan_queue(master, 13, netxl->tx_type);
	}

	master->features = slave->features;
	master->hw_features = slave->hw_features;
	master->vlan_features = slave->vlan_features;

	if (netxl_netdevice_is_wler(slave))
		netxl->wlertx_skb_in_fpm = 1;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5,4,0))
	master->dev.dma_mask = &netxl_dmamask;
	if (dma_set_mask(&master->dev, DMA_BIT_MASK(32)))
		pr_err("Failed to set dma mask for master network device %s.\n",
		       master->name);

#endif

	goto done;

device_cleanup:
	dev_put(slave);
	netxl_sys_del(master);
	unregister_netdev(master);
	kfree(master->ethtool_ops);
	kfree(master->netdev_ops);
	free_netdev(master);
	kfree(netxl);
	status = -EINVAL;

done:
	spin_unlock_irqrestore(&netxl_lock, flags);
	return status;
}

/* Delete netxl device */
int netxl_netdevice_del(char *master_name)
{
	struct net_device *slave;
	struct net_device *master;
	struct netxl *netxl;
	int status = 0;
	int sleepcnt = 0;

	master = __dev_get_by_name(&init_net, master_name);

	if (!master) {
		pr_err("Error, Master %s not found\n", master_name);
		status = -EINVAL;
		goto done;
	}

	if (!(master->flags & IFF_MASTER)) {
		pr_err("Error, Device %s is not a master\n", master_name);
		status = -EINVAL;
		goto done;
	}

	netxl = NETXL_CTX(master);

	if (!netxl) {
		pr_err("Error, Netxl not found\n");
		status = -EINVAL;
		goto done;
	}

	slave = netxl->slave;

	if (slave->flags & IFF_UP) {
		pr_err("Error, Up Slave Device %s cannot be removed\n",
		       slave->name);
		status = -EINVAL;
		goto done;
	}

	while (!rtnl_trylock()) {
		msleep(10);
		if (sleepcnt++ > 100)
			return restart_syscall();
	}

	if ((netxl->mdqm_rx_id >= 0) && (netxl->mdqm_tx_id >= 0))
		dqnet_wifi_close(master);

	netxl_sys_del(master);

	unregister_netdevice(master);
	netdev_rx_handler_unregister(slave);
	rtnl_unlock();

	kfree(master->ethtool_ops);
	kfree(master->netdev_ops);

	slave->flags &= ~IFF_SLAVE;
	dev_put(slave);

	if (netxl->mdqm_rx_id >= 0)
		mdqm_unregister(ifmapping[netxl->id].mrx, netxl->mdqm_rx_id);
	if (netxl->mdqm_tx_id >= 0)
		mdqm_unregister(ifmapping[netxl->id].mtx, netxl->mdqm_tx_id);

	free_netdev(master);

	ifmapping[netxl->id].dev = NULL;
	kfree(netxl);
done:
	pr_info("Netxl device %s removed\n", master_name);
	return status;
}

bool netxl_netdevice_is_slave(struct net_device *dev)
{
	rx_handler_func_t *rx_handler;
	if (!(dev->flags & IFF_SLAVE))
		return false;
	rx_handler = rcu_dereference(dev->rx_handler);
	if (rx_handler == netxl_handle_frame_fpm || rx_handler == netxl_handle_frame_skb)
		return true;
	return false;
}

static inline
int knetxl_del(void *ptr)
{
	char *name = (char *) ptr;
	netxl_netdevice_del(name);
	return 0;
}

struct task_struct *knetxl_task;


/* Update Master State based on Slave event */
static int netxl_slave_netdev_event(unsigned long event,
				    struct net_device *dev)
{
	struct net_device *master;
	master = (struct net_device *)rcu_dereference(dev->rx_handler_data);

	if (!master)
		return NOTIFY_DONE;

	if (!(master->flags & IFF_MASTER))
		return NOTIFY_DONE;
	switch (event) {
	case NETDEV_DOWN:
	case NETDEV_UP:
	case NETDEV_CHANGE:
		if (netif_carrier_ok(dev))
			netif_carrier_on(master);
		else
			netif_carrier_off(master);
		break;
	case NETDEV_UNREGISTER:
		netif_carrier_off(master);
		knetxl_task = kthread_create(knetxl_del, master->name, "%s", "knetxl_del");
		if (knetxl_task)
			wake_up_process(knetxl_task);
		break;
	}
	return NOTIFY_DONE;
}

static int netxl_netdev_event(struct notifier_block *this,
			     unsigned long event, void *ptr)
{
	struct net_device *event_dev = netdev_notifier_info_to_dev(ptr);

	pr_debug("event_dev: %s, event: %lx\n",
		 event_dev ? event_dev->name : "None",
		 event);

	if (!event_dev)
		return NOTIFY_DONE;

	if (netxl_netdevice_is_slave(event_dev))
		return netxl_slave_netdev_event(event, event_dev);

	return NOTIFY_DONE;
}

static struct notifier_block netdev_notifier = {
	.notifier_call = netxl_netdev_event,
};

static int __init netxl_init(void)
{
#if defined(CONFIG_BCM_DQSKB) || defined(CONFIG_BCM_DQSKB_MODULE)
	dqskbdev = dqskb_get_dev_by_name(DQSKB_DEV);
#endif
	netxl_procfs_init();
	register_netdevice_notifier(&netdev_notifier);
	spin_lock_init(&netxl_lock);
	pr_info("NETXL Core Init\n");
	return 0;
}

static void __exit netxl_exit(void)
{
	unregister_netdevice_notifier(&netdev_notifier);
	netxl_procfs_exit();
	pr_info("NETXL Core Exit\n");
}

module_init(netxl_init);
module_exit(netxl_exit);
MODULE_LICENSE("GPL");
MODULE_ALIAS("netxl");
