 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2021 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayesh.patel@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/kthread.h>
#include <linux/rtnetlink.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/if_vlan.h>
#include <linux/pci.h>
#include <linux/pci_ids.h>
#include <net/netevent.h>
#include "netxl.h"
#include "dqnet_priv.h"
#include "dqnet_dbg.h"
#include "mdqm.h"
#include "fpm.h"

#define SLAVE_NETOPS netxl->slave->netdev_ops

#define NETIF_F_DIS_OFF  (NETIF_F_TSO | NETIF_F_TSO6 | NETIF_F_GSO | NETIF_F_GRO)

spinlock_t netxl_lock;

int netxl_open(struct net_device *dev)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);
	pr_debug("--> Device %s\n", dev->name);
	mdqm_enable_id(netxl->ifmapping->mrx, netxl->mdqm_rx_id);
	mdqm_enable_id(netxl->ifmapping->mtx, netxl->mdqm_tx_id);

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5,4,0))
        return dev_open(netxl->slave, NULL);
#else
	return dev_open(netxl->slave);
#endif
}

int netxl_stop(struct net_device *dev)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);
	pr_debug("--> Device %s\n", dev->name);
	mdqm_disable_id(netxl->ifmapping->mrx, netxl->mdqm_rx_id);
	mdqm_disable_id(netxl->ifmapping->mtx, netxl->mdqm_tx_id);

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5,4,0))
        dev_close(netxl->slave);
        return 0;
#else
	return dev_close(netxl->slave);
#endif
}

static int netxl_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	dev->stats.tx_packets++;
	dev->stats.tx_bytes += skb->len;

	pr_err("%s: Netxl Xmit\n", dev->name);

	return SLAVE_NETOPS->ndo_start_xmit(skb, netxl->slave);
}

static int netxl_set_mac_address(struct net_device *dev, void *p)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	if (SLAVE_NETOPS->ndo_set_mac_address)
		return SLAVE_NETOPS->ndo_set_mac_address(netxl->slave, p);
	return 0;
}

static int netxl_do_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);
	if (SLAVE_NETOPS->ndo_do_ioctl)
		return SLAVE_NETOPS->ndo_do_ioctl(netxl->slave, rq, cmd);
	return 0;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5,15,0))
static void netxl_tx_timeout(struct net_device *dev, unsigned int txq)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	if (SLAVE_NETOPS->ndo_tx_timeout)
		SLAVE_NETOPS->ndo_tx_timeout(netxl->slave, txq);
}
#else
static void netxl_tx_timeout(struct net_device *dev)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	if (SLAVE_NETOPS->ndo_tx_timeout)
		SLAVE_NETOPS->ndo_tx_timeout(netxl->slave);
}
#endif

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5,4,0))
static void
netxl_get_stats64(struct net_device *dev, struct rtnl_link_stats64 *stats)
{
        struct netxl *netxl;
        netxl = NETXL_CTX(dev);
        if (SLAVE_NETOPS->ndo_get_stats64)
                SLAVE_NETOPS->ndo_get_stats64(netxl->slave, stats);
}
#else
struct net_device_stats * netxl_get_stats(struct net_device *dev)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	if (SLAVE_NETOPS->ndo_get_stats)
		return SLAVE_NETOPS->ndo_get_stats(netxl->slave);
	return &dev->stats;
}

static struct rtnl_link_stats64 *
netxl_get_stats64(struct net_device *dev, struct rtnl_link_stats64 *stats)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);
	if (SLAVE_NETOPS->ndo_get_stats64)
		return SLAVE_NETOPS->ndo_get_stats64(netxl->slave, stats);
	return NULL;
}
#endif
/*
static int netxl_set_mac_addr(struct net_device *dev, void *p)
{
	struct sockaddr *addr = (struct sockaddr *)p;

	if (netif_running(dev)) {
		netdev_err(dev, "Device busy.\n");
		return -EBUSY;
	}

	memcpy(dev->dev_addr, addr->sa_data, dev->addr_len);

	return 0;
}
*/

static void netxl_change_rx_flags(struct net_device *dev, int flags)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	if (SLAVE_NETOPS->ndo_change_rx_flags)
		return SLAVE_NETOPS->ndo_change_rx_flags(netxl->slave, flags);
}

static void netxl_set_rx_mode(struct net_device *dev)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	if (SLAVE_NETOPS->ndo_set_rx_mode) {
		if ((dev->flags & IFF_PROMISC) && !(netxl->slave->flags & IFF_PROMISC)) {
			netxl->slave->flags = (dev->flags & ~IFF_MASTER) | IFF_SLAVE;
			dev_set_promiscuity(netxl->slave, 1);
		} else {
			netxl->slave->flags = (dev->flags & ~IFF_MASTER) | IFF_SLAVE;
			SLAVE_NETOPS->ndo_set_rx_mode(netxl->slave);
		}
	}
}

static int netxl_change_mtu(struct net_device *dev, int new_mtu)
{
	struct netxl *netxl;
	int ret = 0;
	netxl = NETXL_CTX(dev);

	if (SLAVE_NETOPS->ndo_change_mtu) {
		ret = SLAVE_NETOPS->ndo_change_mtu(netxl->slave, new_mtu);
		if (ret == 0) {
			dev->mtu = new_mtu;
		}
	}
	return ret;
}

static int netxl_set_features(struct net_device *dev,
			      netdev_features_t features)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	/* Avoid calling this in netdev register call */
	if (dev->state == __LINK_STATE_START)
		return 0;

	/* Clear offload feature */
	features = features & ~NETIF_F_DIS_OFF ;
	if (SLAVE_NETOPS->ndo_set_features)
		return SLAVE_NETOPS->ndo_set_features(netxl->slave, features);
	return 0;
}

static netdev_features_t netxl_fix_features(struct net_device *dev,
					    netdev_features_t features)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);

	/* Avoid calling this in netdev register call */
	if (dev->state == __LINK_STATE_START)
		return 0;

	if (SLAVE_NETOPS->ndo_fix_features)
		features = SLAVE_NETOPS->ndo_fix_features(netxl->slave,
							  features);
	/* Clear offload feature */
	return features & ~NETIF_F_DIS_OFF ;
}

static int netxl_vlan_rx_add_vid(struct net_device *dev,
				 __be16 proto, u16 vid)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);
	if (SLAVE_NETOPS->ndo_vlan_rx_add_vid)
		return SLAVE_NETOPS->ndo_vlan_rx_add_vid(netxl->slave, proto, vid);
	return 0;
}

static int netxl_vlan_rx_kill_vid(struct net_device *dev,
				  __be16 proto, u16 vid)
{
	struct netxl *netxl;
	netxl = NETXL_CTX(dev);
	if (SLAVE_NETOPS->ndo_vlan_rx_add_vid)
		return SLAVE_NETOPS->ndo_vlan_rx_kill_vid(netxl->slave, proto, vid);
	return 0;
}

static struct net_device_ops netxl_netdev_ops = {
	.ndo_open		= netxl_open,
	.ndo_stop		= netxl_stop,
	.ndo_start_xmit		= netxl_start_xmit,
	.ndo_change_rx_flags    = netxl_change_rx_flags,
	.ndo_set_rx_mode	= netxl_set_rx_mode,
	.ndo_set_mac_address	= netxl_set_mac_address,
	.ndo_validate_addr      = eth_validate_addr,
	.ndo_do_ioctl		= netxl_do_ioctl,
	.ndo_change_mtu         = netxl_change_mtu,
	.ndo_tx_timeout		= netxl_tx_timeout,
#if (LINUX_VERSION_CODE < KERNEL_VERSION(5,4,0))
        .ndo_get_stats          = netxl_get_stats,
#endif
	.ndo_get_stats64	= netxl_get_stats64,
	.ndo_fix_features	= netxl_fix_features,
	.ndo_set_features	= netxl_set_features,
	.ndo_vlan_rx_add_vid    = netxl_vlan_rx_add_vid,
	.ndo_vlan_rx_kill_vid   = netxl_vlan_rx_kill_vid,
};

/* Update netxl netdev ops */
void netxl_copy_netdev_ops(struct net_device *dst,
			   struct net_device *src)
{
	int i;
	funcptr *p1;
	funcptr *p2;
	dst->netdev_ops = kmemdup(&netxl_netdev_ops,
				  sizeof(struct net_device_ops),
				  GFP_KERNEL);
	p1 = (funcptr *) dst->netdev_ops;
	p2 = (funcptr *) src->netdev_ops;
	for (i = 0; i < (sizeof(struct net_device_ops)/sizeof(funcptr)); i++) {
		/* If src function is null then make dst function null */
		if (!*p2)
			*p1 = 0;
		p2++;
		p1++;
	}
}

/* netxl device internal counters */
void netxl_netdevice_show(char *master_name)
{
	struct net_device *master;
	struct netxl *netxl;

	master = __dev_get_by_name(&init_net, master_name);

	if (!master)
		return;

	netxl = NETXL_CTX(master);

	if (!netxl)
		return;

	pr_info("Master        : %s\n", master->name);
	pr_info(" Flags        : %X\n", master->flags);
	pr_info(" Features         : %llX\n", master->features);
	pr_info(" HW Features      : %llX\n", master->hw_features);
	pr_info(" Wanted Features  : %llX\n", master->wanted_features);
	pr_info(" VLAN Features    : %llX\n", master->vlan_features);
	pr_info("Slave         : %s\n", netxl->slave->name);
	pr_info(" Flags        : %X\n", netxl->slave->flags);
	pr_info(" Features         : %llX\n", netxl->slave->features);
	pr_info(" HW Features      : %llX\n", netxl->slave->hw_features);
	pr_info(" Wanted Features  : %llX\n", netxl->slave->wanted_features);
	pr_info(" VLAN Features    : %llX\n", netxl->slave->vlan_features);
	pr_info("Dev ID        : %d\n", netxl->id);
	pr_info("Rx_offload    : %d\n", netxl->rx_offload);
	pr_info("Tx_bypass     : %d\n", netxl->tx_bypass);
	pr_info("FPM in SKB    : %d\n", netxl->wlertx_skb_in_fpm);
	pr_info("Rx type       : %s\n", netxl->rx_type ? "skb":"fpm");
	pr_info("Tx type       : %s\n", netxl->tx_type ? "skb":"fpm");
	pr_info("Head_pad      : %d\n", netxl->head_pad);
	pr_info("Tail_pad      : %d\n", netxl->tail_pad);
	pr_info("TX Headroom   : %d\n", netxl->tx_headroom);
	netxl_fap_show(netxl);
	pr_info("Stats:\n");
	pr_info("RX packets: %lu, bytes: %lu errors: %lu dropped: %lu\n",
		master->stats.rx_packets, master->stats.rx_bytes,
		master->stats.rx_errors, master->stats.rx_dropped);
	pr_info("TX packets: %lu, bytes: %lu errors: %lu dropped: %lu\n",
		master->stats.tx_packets, master->stats.tx_bytes,
		master->stats.tx_errors, master->stats.tx_dropped);
	return;
}

/* TX nethook to bypass TX path directly to slave device instead of
   passing through FAP */
static enum bcm_nethook_result bcm_nethook_tx(
	struct net_device *out, enum bcm_nethook_type type, void *buf)
{
	struct sk_buff *skb = (struct sk_buff *)buf;
	struct netxl *netxl;
	netxl = NETXL_CTX(out);

	if (netxl->tx_bypass) {
		SLAVE_NETOPS->ndo_start_xmit(skb, netxl->slave);
		return BCM_NETHOOK_CONSUMED;
	}
	return BCM_NETHOOK_PASS;
}

/* Configure netxl device parameters */
int netxl_netdevice_cfg(char *master_name, char *param,
			int val)
{
	struct net_device *master;
	struct netxl *netxl;
	int status = 0;
	unsigned long flags;

	spin_lock_irqsave(&netxl_lock, flags);

	master = __dev_get_by_name(&init_net, master_name);

	if (!master) {
		pr_err("Error, Master %s not found\n", master_name);
		status = -EINVAL;
		goto done;
	}

	if (!(master->flags & IFF_MASTER)) {
		pr_err("Error, Device %s is not a master\n", master_name);
		status = -EINVAL;
		goto done;
	}

	netxl = NETXL_CTX(master);

	if (!netxl) {
		pr_err("Error, Netxl not found\n");
		status = -EINVAL;
		goto done;
	}

	if (!strcmp(param, "tx_headroom")) {
		netxl->tx_headroom = val;
	}

done:
	spin_unlock_irqrestore(&netxl_lock, flags);
	return status;
}

int netxl_netdev_update_features(struct net_device *dev)
{
	netdev_features_t features;
	int err = 0;

	ASSERT_RTNL();

	features = netdev_get_wanted_features(dev);

	if (dev->netdev_ops->ndo_fix_features)
		features = dev->netdev_ops->ndo_fix_features(dev, features);

	if (dev->features == features)
		return 0;

	netdev_dbg(dev, "Features changed: %pNF -> %pNF\n",
		&dev->features, &features);

	if (dev->netdev_ops->ndo_set_features)
		err = dev->netdev_ops->ndo_set_features(dev, features);

	if (unlikely(err < 0)) {
		netdev_err(dev,
			"set_features() failed (%d); wanted %pNF, left %pNF\n",
			err, &features, &dev->features);
		return -1;
	}

	if (!err)
		dev->features = features;

	return 1;
}

bool netxl_netdevice_is_wler(struct net_device *net)
{
	struct ethtool_drvinfo info;
	if (!net->ethtool_ops)
		return false;
	if (!net->ethtool_ops->get_drvinfo)
		return false;
	memset(&info, 0, sizeof(info));
	net->ethtool_ops->get_drvinfo(net, &info);
	if (strncmp(info.reserved2, "wlerouter", sizeof("wlerouter")) == 0)
		return true;
	return false;
}

#define PCI_VENDOR_ID_QUANTENNA 0x1bb5
#define PCI_VENDOR_ID_CELENO1   0x1d59
#define PCI_VENDOR_ID_CELENO2   0x1d69

const struct pci_device_id netxl_allowed_pci_ids[] = {
	{ PCI_DEVICE(PCI_VENDOR_ID_BROADCOM,  PCI_ANY_ID) },
	{ PCI_DEVICE(PCI_VENDOR_ID_QUANTENNA, PCI_ANY_ID) },
	{ PCI_DEVICE(PCI_VENDOR_ID_CELENO1,   PCI_ANY_ID) },
	{ PCI_DEVICE(PCI_VENDOR_ID_CELENO2,   PCI_ANY_ID) },
	{}
};

const struct pci_device_id netxl_any_pci_ids[] = {
	{ PCI_DEVICE(PCI_ANY_ID,    PCI_ANY_ID) },
	{}
};

int netxl_is_pci_allowed(struct pci_dev *dev)
{
	int i;

	for (i = 0; i < sizeof(netxl_allowed_pci_ids)/sizeof(struct pci_device_id); i++) {
		if (dev->vendor == netxl_allowed_pci_ids[i].vendor)
			return true;
	}
	return false;
}

int netxl_is_allowed(struct net_device *slave)
{
	struct ethtool_drvinfo info;
	struct pci_dev *dev = NULL;

	if (!pci_dev_present(netxl_any_pci_ids)) {
		/* If no PCI devices are present then allow netxl
		   to be used on non PCI network devices like USB */
		if (slave->ethtool_ops && slave->ethtool_ops->get_drvinfo) {
			slave->ethtool_ops->get_drvinfo(slave, &info);
			if (strstr(info.bus_info, "usb"))
				return true;
		}
	} else {
		for_each_pci_dev(dev) {
			if (!netxl_is_pci_allowed(dev)) {
				pci_dev_put(dev);
				return false;
			}
		}
	}
	return true;
}

/* Add netxl device */
int netxl_netdevice_add(char *master_name, char *slave_name,
			int id, char *pkt_type)
{
	struct net_device *slave;
	struct net_device *master;
	struct netxl *netxl;
	int status = 0;
	unsigned long flags;
	netdev_features_t mask;
	struct netxl_if_map *ifmapping;

	spin_lock_irqsave(&netxl_lock, flags);
	if (!strcmp(master_name, slave_name)) {
		pr_err("Error, Master name should not be same as slave name\n");
		status = -EINVAL;
		goto done;
	}

	ifmapping = netxl_fap_ifmapping(id);
	/* Check if ID is available */
	if (!ifmapping) {
		pr_err("Error, Invalid Mapping ID %d\n",
		       id);
		status = -EINVAL;
		goto done;
	}


	if (ifmapping->dev) {
		pr_err("Error, Mapping ID %d already in use by %s\n", id,
		       ifmapping->dev->name);
		status = -EINVAL;
		goto done;
	}

	slave = __dev_get_by_name(&init_net, slave_name);

	if (!slave) {
		pr_err("Error, Slave device %s not found\n", slave_name);
		status = -EINVAL;
		goto done;
	}

	if (!netxl_is_allowed(slave)) {
		pr_err("Error, Unsupported PCI device is present\n");
		status = -EPERM;
		goto done;
	}

	if (slave->flags & IFF_UP) {
		pr_err("Error, Up Device %s cannot be enslaved\n", slave_name);
		status = -EINVAL;
		goto done;
	}

	if (slave->flags & IFF_MASTER) {
		pr_err("Error, Device %s is master device. It cannot be enslaved\n",
		       slave_name);
		status = -EINVAL;
		goto done;
	}

	if (slave->flags & IFF_SLAVE) {
		pr_err("Error, Device %s was already enslaved\n", slave_name);
		status = -EINVAL;
		goto done;
	}

	if (slave->rx_handler) {
		pr_err("Error, Device %s was already enslaved\n", slave_name);
		status = -EINVAL;
		goto done;
	}

	netxl = kzalloc(sizeof(struct netxl), GFP_KERNEL);
	if (!netxl) {
		pr_err("Error, Allocating Netxl Priv Data\n");
		status = -EINVAL;
		goto done;
	}

	master = alloc_etherdev(sizeof(struct dqnet_netdev));
	if (!master) {
		pr_err("Error, Allocating Master Device\n");
		status = -EINVAL;
		kfree(netxl);
		goto done;
	}

	slave->flags |= IFF_SLAVE;
	master->flags |= IFF_MASTER;

	/* Disable offload features */
	mask = NETIF_F_DIS_OFF ;
	if (mask) {
		slave->wanted_features = slave->hw_features;
		slave->wanted_features &= ~mask;
		rtnl_lock();
		netxl_netdev_update_features(slave);
		rtnl_unlock();
		slave->wanted_features = 0;
	}

	NETXL_CTX(master) = netxl;
	netxl_copy_netdev_ops(master, slave);
	netxl->dev = master;
	netxl->slave = slave;

	snprintf(master->name, IFNAMSIZ, "%s", master_name);

	memcpy(master->dev_addr, slave->dev_addr, master->addr_len);

	/*
	if (!(master->dev_addr[0] & 0x2)) {
		master->dev_addr[0] |= 0x2;
	}
	*/

	netxl_copy_ethtool_ops(master, slave);
	if (register_netdev(master))
		pr_err("Failed to register master network device.\n");

	netif_carrier_off(master);

	dev_hold(slave);
	netxl_sys_add(master);

	netxl->id  = id;

       if (netxl_fap_netdev_init(netxl, pkt_type))
	       goto device_cleanup;

	/* Register tx nethook after flowmgr */
	bcm_nethook_register_hook(master, BCM_NETHOOK_TX_SKB,
				  TX_SKB_PRIO_FLOWMGR+1, "NetXL",
				  bcm_nethook_tx);
	bcm_nethook_enable_hook(master, BCM_NETHOOK_TX_SKB,
				bcm_nethook_tx, true);

	master->features = slave->features;
	master->hw_features = slave->hw_features;
	master->vlan_features = slave->vlan_features;

	if (netxl_netdevice_is_wler(slave))
		netxl->wlertx_skb_in_fpm = 1;

	goto done;

device_cleanup:
	dev_put(slave);
	netxl_sys_del(master);
	unregister_netdev(master);
	kfree(master->ethtool_ops);
	kfree(master->netdev_ops);
	free_netdev(master);
	kfree(netxl);
	status = -EINVAL;

done:
	spin_unlock_irqrestore(&netxl_lock, flags);
	return status;
}

/* Delete netxl device */
int netxl_netdevice_del(char *master_name)
{
	struct net_device *slave;
	struct net_device *master;
	struct netxl *netxl;
	int status = 0;
	struct netxl_if_map *ifmapping;

	master = __dev_get_by_name(&init_net, master_name);

	if (!master) {
		pr_err("Error, Master %s not found\n", master_name);
		status = -EINVAL;
		goto done;
	}

	if (!(master->flags & IFF_MASTER)) {
		pr_err("Error, Device %s is not a master\n", master_name);
		status = -EINVAL;
		goto done;
	}

	netxl = NETXL_CTX(master);

	if (!netxl) {
		pr_err("Error, Netxl not found\n");
		status = -EINVAL;
		goto done;
	}

	ifmapping = netxl->ifmapping;
	slave = netxl->slave;

	if (slave->flags & IFF_UP) {
		pr_err("Error, Up Slave Device %s cannot be removed\n",
		       slave->name);
		status = -EINVAL;
		goto done;
	}

	netxl_fap_netdev_exit(netxl);

	kfree(master->ethtool_ops);
	kfree(master->netdev_ops);

	slave->flags &= ~IFF_SLAVE;
	dev_put(slave);

	free_netdev(master);

	kfree(netxl);
done:
	pr_info("Netxl device %s removed\n", master_name);
	return status;
}

static inline
int knetxl_del(void *ptr)
{
	char *name = (char *) ptr;
	netxl_netdevice_del(name);
	return 0;
}

struct task_struct *knetxl_task;


/* Update Master State based on Slave event */
static int netxl_slave_netdev_event(unsigned long event,
				    struct net_device *dev)
{
	struct net_device *master;
	master = (struct net_device *)rcu_dereference(dev->rx_handler_data);

	if (!master)
		return NOTIFY_DONE;

	if (!(master->flags & IFF_MASTER))
		return NOTIFY_DONE;
	switch (event) {
	case NETDEV_DOWN:
	case NETDEV_UP:
	case NETDEV_CHANGE:
		if (netif_carrier_ok(dev))
			netif_carrier_on(master);
		else
			netif_carrier_off(master);
		break;
	case NETDEV_UNREGISTER:
		netif_carrier_off(master);
		knetxl_task = kthread_create(knetxl_del, master->name, "%s", "knetxl_del");
		if (knetxl_task)
			wake_up_process(knetxl_task);
		break;
	}
	return NOTIFY_DONE;
}

static int netxl_netdev_event(struct notifier_block *this,
			     unsigned long event, void *ptr)
{
	struct net_device *event_dev = netdev_notifier_info_to_dev(ptr);

	pr_debug("event_dev: %s, event: %lx\n",
		 event_dev ? event_dev->name : "None",
		 event);

	if (!event_dev)
		return NOTIFY_DONE;

	if (netxl_netdevice_is_slave(event_dev))
		return netxl_slave_netdev_event(event, event_dev);

	return NOTIFY_DONE;
}

static struct notifier_block netdev_notifier = {
	.notifier_call = netxl_netdev_event,
};

static int __init netxl_init(void)
{
	netxl_fap_init();
	netxl_procfs_init();
	register_netdevice_notifier(&netdev_notifier);
	spin_lock_init(&netxl_lock);
	pr_info("NETXL Core Init\n");
	return 0;
}

static void __exit netxl_exit(void)
{
	netxl_fap_exit();
	unregister_netdevice_notifier(&netdev_notifier);
	netxl_procfs_exit();
	pr_info("NETXL Core Exit\n");
}

module_init(netxl_init);
module_exit(netxl_exit);
MODULE_LICENSE("GPL");
MODULE_ALIAS("netxl");
