/*
 * Copyright (C) 2017 Broadcom Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/ctype.h>
#include <linux/file.h>
#include <linux/jiffies.h>
#include <linux/string.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include "bspeed.h"

static struct bspeed_stats stats;
static struct bspeed_udp_stats udp_stats;

void bspeed_record_stats(struct bspeed_stats *sts)
{
	if (sts->packets) {
		stats.packets += sts->packets;
		stats.acks += sts->acks;
		stats.sacks += sts->sacks;
		stats.ooo += sts->ooo;
		stats.bytes += sts->bytes;
		stats.end_jiffies += sts->end_jiffies - sts->start_jiffies;
		memset(sts, 0, sizeof(*sts));
	}
}

void bspeed_print_stream_stats(struct bspeed_stats *sts)
{
	u32 mb = (u32)(sts->bytes >> 20);
	pr_info("bspeed acceleration stats:\n");
	pr_info("packets:           %u\n", sts->packets);
	pr_info("acks:              %u\n", sts->acks);
	pr_info("sacks:             %u\n", sts->sacks);
	pr_info("out of order:      %u\n", sts->ooo);
	pr_info("MBytes:            %u\n", mb);
	pr_info("Rate (MBitS):      %u\n",
		(mb / (jiffies_to_msecs(sts->end_jiffies - sts->start_jiffies) / 1000)) * 8);
}

void bspeed_print_stats(void)
{
	u32 mb = (u32)(stats.bytes >> 20);
	pr_info("bspeed accumulated stats:\n");
	pr_info("packets:           %u\n", stats.packets);
	pr_info("acks:              %u\n", stats.acks);
	pr_info("sacks:             %u\n", stats.sacks);
	pr_info("out of order:      %u\n", stats.ooo);
	pr_info("MBytes:            %u\n", mb);
	pr_info("Rate (MBitS):      %u\n",
		(mb / (jiffies_to_msecs(stats.end_jiffies) / 1000)) * 8);
}

void bspeed_record_udp_stats(struct bspeed_udp_stats *sts)
{
	if (sts->packets) {
		udp_stats.packets += sts->packets;
		udp_stats.rd_calls += sts->rd_calls;
		udp_stats.ooo += sts->ooo;
		udp_stats.lost += sts->lost;
		udp_stats.jitter += sts->jitter;
		udp_stats.bytes += sts->bytes;
		udp_stats.end_jiffies += sts->end_jiffies - sts->start_jiffies;
		memset(sts, 0, sizeof(*sts));
	}
}

void bspeed_print_stream_udp_stats(struct bspeed_udp_stats *sts)
{
	u32 mb = (u32)(sts->bytes >> 20);
	pr_info("bspeed acceleration udp stats:\n");
	pr_info("packets:           %llu\n", sts->packets);
	pr_info("rd/wr calls:       %u\n", sts->rd_calls);
	pr_info("out of order:      %llu\n", sts->ooo);
	pr_info("lost:              %llu\n", sts->lost);
	pr_info("jitter:            %d usecs\n", sts->jitter);
	pr_info("MBytes:            %u\n", mb);
	pr_info("Rate (MBitS):      %u\n",
		(mb / (jiffies_to_msecs(sts->end_jiffies - sts->start_jiffies) / 1000)) * 8);
}

void bspeed_print_udp_stats(void)
{
	u32 mb = (u32)(udp_stats.bytes >> 20);
	pr_info("bspeed accumulated udp stats:\n");
	pr_info("packets:           %llu\n", udp_stats.packets);
	pr_info("rd/wr calls:       %u\n", udp_stats.rd_calls);
	pr_info("out of order:      %llu\n", udp_stats.ooo);
	pr_info("lost:              %llu\n", udp_stats.lost);
	pr_info("jitter:            %d usecs\n", udp_stats.jitter);
	pr_info("MBytes:            %u\n", mb);
	pr_info("Rate (MBitS):      %u\n",
		(mb / (jiffies_to_msecs(udp_stats.end_jiffies) / 1000)) * 8);
}

void bspeed_print_stats_proc(struct seq_file *seq)
{
	u32 mb = (u32)(stats.bytes >> 20);
	seq_printf(seq, "bspeed accumulated stats:\n");
	seq_printf(seq, "packets:           %u\n", stats.packets);
	seq_printf(seq, "acks:              %u\n", stats.acks);
	seq_printf(seq, "sacks:             %u\n", stats.sacks);
	seq_printf(seq, "out of order:      %u\n", stats.ooo);
	seq_printf(seq, "MBytes:            %u\n", mb);
	seq_printf(seq, "Rate (MBitS):      %u\n",
		(mb / (jiffies_to_msecs(stats.end_jiffies) / 1000)) * 8);
	mb = (u32)(udp_stats.bytes >> 20);
	seq_printf(seq, "bspeed accumulated udp stats:\n");
	seq_printf(seq, "packets:           %llu\n", udp_stats.packets);
	seq_printf(seq, "rd/wr calls:       %u\n", udp_stats.rd_calls);
	seq_printf(seq, "out of order:      %llu\n", udp_stats.ooo);
	seq_printf(seq, "lost:              %llu\n", udp_stats.lost);
	seq_printf(seq, "jitter:            %d usecs\n", udp_stats.jitter);
	seq_printf(seq, "MBytes:            %u\n", mb);
	seq_printf(seq, "Rate (MBitS):      %u\n",
		(mb / (jiffies_to_msecs(udp_stats.end_jiffies) / 1000)) * 8);
}

void bspeed_reset_stats(void)
{
	memset(&stats, 0, sizeof(stats));
	memset(&udp_stats, 0, sizeof(stats));
}
