/*
 * Copyright (C) 2018 Broadcom Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *
 *
 * Author: Peter Sulc <peter.sulc@broadcom.com>
 *
 */

#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/ctype.h>
#include <linux/file.h>
#include <linux/skbuff.h>
#include <linux/net.h>
#include <linux/etherdevice.h>
#include <linux/socket.h>
#include <linux/ktime.h>
#include <linux/printk.h>
#include <linux/dynamic_debug.h>
#include <net/neighbour.h>
#include <net/netevent.h>
#include <linux/if_arp.h>
#include <net/ipv6.h>
#include <net/tcp.h>
#include <net/udp.h>
#include <net/sock.h>
#include <net/protocol.h>
#include <net/inet_sock.h>
#include <linux/netfilter.h>
#include <linux/netfilter_bridge.h>
#include <net/ip6_checksum.h>

#include "fpm.h"
#include "dqnet.h"
#include "bcmnethooks.h"
#include "bspeed.h"
#include "dqnet_priv.h"


struct sk_udp_priv {
	struct completion compl;
	struct proto_ops *new_ops;
	const struct proto_ops *old_ops;
};

DEFINE_RWLOCK(bspeed_conn_list_lock);

void bspeed_conn_list_write_lock(void)
{
	write_lock_bh(&bspeed_conn_list_lock);
}

void bspeed_conn_list_write_unlock(void)
{
	write_unlock_bh(&bspeed_conn_list_lock);
}

void bspeed_conn_list_read_lock(void)
{
	read_lock_bh(&bspeed_conn_list_lock);
}

void bspeed_conn_list_read_unlock(void)
{
	read_unlock_bh(&bspeed_conn_list_lock);
}

void *bspeed_spin_lock_create(void)
{
	spinlock_t *lock = kmalloc(sizeof(spinlock_t), GFP_KERNEL);
	spin_lock_init(lock);
	return lock;
}

void bspeed_spin_lock_destroy(void *lock)
{
	kfree(lock);
}

void bspeed_spin_lock(void *lock, unsigned long *flags)
{
	spin_lock_irqsave(lock, *flags);
}

int bspeed_spin_trylock(void *lock, unsigned long *flags)
{
	return spin_trylock_irqsave(lock, *flags);
}

void bspeed_spin_unlock(void *lock, unsigned long flags)
{
	spin_unlock_irqrestore(lock, flags);
}

/* older kernel needs this */
#ifndef LOGLEVEL_DEFAULT
#define LOGLEVEL_DEFAULT	-1
#endif
#ifndef LOGLEVEL_ERR
#define LOGLEVEL_ERR		3
#endif
#ifndef LOGLEVEL_INFO
#define LOGLEVEL_INFO		6
#endif

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5,15,0))
#define vprintk_emit(a,b,c,d,e,f) vprintk_emit(a,b,c,e,f)
#endif

int bspeed_printk(const char *fmt, ...)
{
	va_list args;
	int r;

	va_start(args, fmt);
	r = vprintk_emit(0, LOGLEVEL_DEFAULT, NULL, 0, fmt, args);
	va_end(args);

	return r;
}

int bspeed_pr_info(const char *fmt, ...)
{
	va_list args;
	int r;

	va_start(args, fmt);
	r = vprintk_emit(0, LOGLEVEL_INFO, NULL, 0, fmt, args);
	va_end(args);

	return r;
}

int bspeed_pr_err(const char *fmt, ...)
{
	va_list args;
	int r;

	va_start(args, fmt);
	r = vprintk_emit(0, LOGLEVEL_ERR, NULL, 0, fmt, args);
	va_end(args);

	return r;
}

int bspeed_sprintf(char *buf, const char *fmt, ...)
{
	va_list args;
	int r;

	va_start(args, fmt);
	r = vsprintf(buf, fmt, args);
	va_end(args);

	return r;
}

void *bspeed_kmalloc(int size, bool atomic)
{
	if (atomic)
		return kmalloc(size, GFP_ATOMIC);
	else
		return kmalloc(size, GFP_KERNEL);
}

void bspeed_kfree(void *p)
{
	kfree(p);
}

u32 bspeed_get_jiffies(void)
{
	return jiffies;
}

u32 bspeed_get_tcp_timestamp(void *sk)
{
	struct tcp_sock *tp = tcp_sk((struct sock *)sk);
	return tp->tsoffset + tcp_time_stamp_raw();
}

u32 bspeed_get_tcp_timestamp_diff(void *sk, u32 ts)
{
	struct tcp_sock *tp = tcp_sk((struct sock *)sk);
	return tp->tsoffset + tcp_time_stamp_raw() - ts;
}

u32 bspeed_msecs_since(u32 prior_jiffies)
{
	return jiffies_to_msecs(jiffies - prior_jiffies);
}

u64 bspeed_get_ns(void)
{
	return ktime_to_ns(ktime_get());
}

u64 bspeed_ns_since(u64 ns)
{
	return ktime_to_ns(ktime_sub_ns(ktime_get(), ns));
}

void bspeed_update_us_tcp_send_seq(void *skp, u32 seq)
{
	struct tcp_sock *tp = tcp_sk((struct sock *)skp);
	tp->snd_up = tp->snd_sml = tp->snd_nxt = seq;
	tp->pushed_seq = tp->write_seq = seq;
	tp->lsndtime = tcp_jiffies32;
	tp->packets_out = 0;
	tp->max_packets_seq = seq;
}

void bspeed_update_ds_tcp_send_seq(void *skp, u32 seq)
{
	struct tcp_sock *tp = tcp_sk((struct sock *)skp);
	u32 delta = seq - tp->snd_nxt;
	tp->snd_sml = tp->snd_nxt = tp->snd_una = seq;
	tp->bytes_sent += delta;
}

void bspeed_update_ds_tcp_recv_seq(void *sk, u32 rx_next_seq)
{
	struct tcp_sock *tp = tcp_sk((struct sock *)sk);
	u32 delta = rx_next_seq - tp->rcv_nxt;
	tp->rcv_nxt = tp->rcv_wup = rx_next_seq;
	tp->bytes_received += delta;
	tcp_data_ready(sk);
}

void bspeed_update_ds_tcp_copied_seq(void *sk, int copied)
{
	struct tcp_sock *tp = tcp_sk((struct sock *)sk);
	tp->copied_seq += copied;
}

__be16 bspeed_tcp_v4_csum(int len, __be32 saddr, __be32 daddr, const void *tcph)
{
	return tcp_v4_check(len, saddr, daddr, csum_partial(tcph, len, 0));
}

void bspeed_ip_send_check(struct iphdr *iph)
{
	ip_send_check(iph);
}

__be16 bspeed_tcp_v6_csum(int len,
			 const struct in6_addr *saddr,
			 const struct in6_addr *daddr,
			 const void *tcph)
{
	return tcp_v6_check(len, saddr, daddr, csum_partial(tcph, len, 0));
}

static void *bspeed_build_skb(unsigned int size, void *data)
{
	struct sk_buff *skb = NULL;

	skb = dev_alloc_skb(0);
	if(skb)
		skb->data = (unsigned char *)data;

	return skb;
}

static void bspeed_release_skb(void *skb)
{
	dev_kfree_skb_any((struct sk_buff *)skb);
}

u64 bspeed_tx_fpm_buff(struct bspeed_fpm_buff *ifb)
{
	struct fpm_buff *fb = (struct fpm_buff *)ifb;
	struct dqnet_netdev *ndev = netdev_priv(fb->dev_out);
	int ret;

	if (ndev->dhdol_get_flow && !fb->skb)
		fb->skb = bspeed_build_skb(0, fb->data);
	ret = bcm_nethook_tx_fpm(fb, fb->dev_out);
	if (ndev->dhdol_get_flow && fb->skb)
		bspeed_release_skb(fb->skb);

	if (ret)
		return 0;
	return ktime_to_ns(ktime_get());
}

u32 bspeed_get_tcpsock_mss(void *sk)
{
	struct tcp_sock *tp = tcp_sk((struct sock *)sk);
	return tp->mss_cache;
}

void bspeed_update_send_tcp_window(void *sk, u32 seq, u16 window)
{
	struct tcp_sock *tp = tcp_sk((struct sock *)sk);
	tp->snd_wnd = window << tp->rx_opt.snd_wscale;
	tp->snd_wl1 = seq;
}

u32 bspeed_get_snd_una(void *sk)
{
	struct tcp_sock *tp = tcp_sk((struct sock *)sk);
	return tp->snd_una;
}

int bspeed_get_avail(void *sk)
{
	struct tcp_sock *tp = tcp_sk((struct sock *)sk);
	int avail = tp->rcv_nxt - tp->copied_seq;
	if (avail < 0) {
		pr_debug("%s ERROR: socket math error\n", __func__);
		avail = 0;
	}
	return avail;
}

void bspeed_set_snd_una(void *sk, u32 seq)
{
	struct tcp_sock *tp = tcp_sk((struct sock *)sk);
	tp->snd_una = seq;
	tp->rcv_tstamp = tcp_jiffies32;
}

static void bspeed_tcp_clear_retrans(struct tcp_sock *tp)
{
	tp->retrans_out = 0;
	tp->lost_out = 0;
	tp->undo_marker = 0;
	tp->undo_retrans = -1;
	tp->sacked_out = 0;
}

void bspeed_clear_xmit_timers(void *sck)
{
	struct sock *sk = sck;
	struct tcp_sock *tp = tcp_sk(sk);

	inet_csk_clear_xmit_timers(sk);
	__skb_queue_purge(&sk->sk_receive_queue);
	tcp_write_queue_purge(sk);
	bspeed_tcp_clear_retrans(tp);
}

void bspeed_set_rcvlowat(void *sk, int lowat)
{
	tcp_set_rcvlowat(sk, lowat);
}

void *bspeed_register_napi_complete(void *dev, void (*napi_complete_hook)(void *))
{
	return dqnet_register_napi_complete(dev, napi_complete_hook);
}

void bspeed_unregister_napi_complete(void *chan)
{
	dqnet_unregister_napi_complete(chan);
}

static enum bcm_nethook_result
nethook_tx(struct net_device *dev, enum bcm_nethook_type type, void *buf)
{
	struct sk_buff *skb = (struct sk_buff *)buf;
	struct ethhdr *eh = eth_hdr(skb);

	if (eh->h_proto == htons(ETH_P_IP)) {
		struct iphdr *iph = (struct iphdr *)((void *)eh + ETH_HLEN);
		if (iph->protocol == IPPROTO_TCP)
			return bspeed_nethook_tx(eh);
		else if (iph->protocol == IPPROTO_UDP)
			return bspeed_nethook_tx_udp4(dev, eh, skb->len);
	}
	else if (eh->h_proto == htons(ETH_P_IPV6)) {
		struct ipv6hdr *ipv6h = (struct ipv6hdr *)((void *)eh + ETH_HLEN);
		if (ipv6h->nexthdr == IPPROTO_TCP)
			return bspeed_nethook_tx(eh);
		else if (ipv6h->nexthdr == IPPROTO_UDP)
			return bspeed_nethook_tx_udp6(dev, eh, skb->len);
	}
	return BCM_NETHOOK_PASS;
}

static enum bcm_nethook_result
nethook_rx(struct net_device *dev, enum bcm_nethook_type type, void *buf)
{
	struct fpm_buff *fb = buf;
	struct ethhdr *eh = (struct ethhdr *)fb->data;

	if (is_multicast_ether_addr(eh->h_dest))
		return BCM_NETHOOK_PASS;

	if (eh->h_proto == htons(ETH_P_IP)) {
		struct iphdr *iph = (struct iphdr *)(fb->data + ETH_HLEN);
		if (ip_is_fragment(iph))
			return BCM_NETHOOK_PASS;
		if (iph->protocol == IPPROTO_TCP)
			return bspeed_nethook_rx(fb);
		else if (iph->protocol == IPPROTO_UDP)
			return bspeed_nethook_rx_udp4(fb);
	}
	else if (eh->h_proto == htons(ETH_P_IPV6)) {
		struct ipv6hdr *ipv6h = (struct ipv6hdr *)(fb->data + ETH_HLEN);
		if (ipv6h->nexthdr == NEXTHDR_FRAGMENT)
			return BCM_NETHOOK_PASS;
		if (ipv6h->nexthdr == IPPROTO_TCP)
			return bspeed_nethook_rx(fb);
		else if (ipv6h->nexthdr == IPPROTO_UDP)
			return bspeed_nethook_rx_udp6(fb);
	}
	return BCM_NETHOOK_PASS;
}

static int register_hooks(struct sock *sk)
{
	int devices = 0;
	int n;
	n = bcm_nethook_register_hook_devs(BCM_NETHOOK_RX_FPM,
					   RX_FPM_PRIO_RTF, "bSpeedRx",
					   nethook_rx);
	devices = n;
	n = bcm_nethook_register_hook_devs(BCM_NETHOOK_TX_SKB,
					   TX_SKB_PRIO_RTF, "bSpeedTx",
					   nethook_tx);
	if (n != devices)
		pr_err("%s error register BCM_NETHOOK_TX_SKB\n", __func__);
	n = bcm_nethook_enable_hook_devs(BCM_NETHOOK_RX_FPM,
				     nethook_rx, true);
	if (n != devices)
		pr_err("%s error enable BCM_NETHOOK_RX_FPM\n", __func__);
	n = bcm_nethook_enable_hook_devs(BCM_NETHOOK_TX_SKB,
				     nethook_tx, true);
	if (n != devices)
		pr_err("%s error enable BCM_NETHOOK_TX_SKB\n", __func__);

	return devices;
}

static int unregister_hooks(struct sock *sk)
{
	int devices = 0;
	int n;

	n = bcm_nethook_enable_hook_devs(BCM_NETHOOK_RX_FPM,
					 nethook_rx, false);
	devices = n;
	n = bcm_nethook_enable_hook_devs(BCM_NETHOOK_TX_SKB,
					 nethook_tx, false);
	if (n != devices)
		pr_err("%s error disable BCM_NETHOOK_TX_SKB\n", __func__);
	n = bcm_nethook_unregister_hook_devs(BCM_NETHOOK_RX_FPM,
					     nethook_rx);
	if (n != devices)
		pr_err("%s error unregister BCM_NETHOOK_RX_FPM\n", __func__);
	n = bcm_nethook_unregister_hook_devs(BCM_NETHOOK_TX_SKB,
					     nethook_tx);
	if (n != devices)
		pr_err("%s error unregister BCM_NETHOOK_TX_SKB\n", __func__);
	return devices;
}

static __be32 get_ports(struct sock *sk)
{
        u32     sport = ntohs(sk->sk_dport);
        u32     dport = sk->sk_num;
        __be32  ports = htonl((sport << 16) | dport);
        return ports;
}

static int add_nethook_client(struct sock *sk, u32 rate, bool sender,
			      int udp, u64 blocks)
{
	if (bspeed_connection_count())
		bspeed_clean_old();
	else
		register_hooks(sk);

	if (sk->sk_family == AF_INET) {
		__be32	saddr = sk->sk_daddr;
		__be32	daddr = sk->sk_rcv_saddr;
		__be32	ports = get_ports(sk);
		bspeed_new_connection_v4(sk, saddr, daddr, ports,
					 rate, sender, udp, blocks);

	}
	else if (sk->sk_family == AF_INET6) {
		__be32	*saddr = sk->sk_v6_daddr.s6_addr32;
		__be32	*daddr = sk->sk_v6_rcv_saddr.s6_addr32;
		__be32	ports = get_ports(sk);
		bspeed_new_connection_v6(sk, saddr, daddr, ports,
					 rate, sender, udp, blocks);
	}
	else
		pr_err("%s: unsupported sock family %d\n", __func__,
		       sk->sk_family);
	pr_info("%s %s rate %d mbs udp %d blocks %lld\n", __func__,
		sender ? "sender" : "receiver", rate, udp, blocks);
	return 0;
}

static int del_nethook_client(struct sock *sk)
{
	int left = bspeed_del_connection(sk);
	if (left == 0)
		unregister_hooks(sk);
	return left;
}

static int tcp_data_recv(read_descriptor_t *rd_desc, struct sk_buff *skb,
			 unsigned int offset, size_t len)
{
	return len;
}


static int bspeed_read_sock(struct sock *sk, int buflen)
{
	int r, accelerated;
	read_descriptor_t rd_desc = {
		.count	  = 1,
	};
	lock_sock(sk);
	if (sk->sk_state == TCP_LISTEN ||
	    sk->sk_state == TCP_CLOSE) {
		release_sock(sk);
		return -ENOTCONN;
	}
	r = bspeed_read_connection(sk, buflen, &accelerated);
	if (!accelerated) {
		r = tcp_read_sock(sk, &rd_desc, tcp_data_recv);
		if (r < 0)
			pr_err("%s ERROR: tcp_read_sock failed\n", __func__);
	}
	release_sock(sk);
	return r;
}

bool bspeed_socket_tx_acks_pending(void *sock)
{
	struct sock *sk;
	struct tcp_sock *tp;
	unsigned pkts = 0;
	sk = ((struct socket *)sock)->sk;
	if (!sk)
		return false;
	tp = tcp_sk(sk);
	if (!tp)
		return false;
	pkts = tcp_packets_in_flight(tp);
	return pkts || (tp->snd_una < tp->write_seq);
}

void bspeed_get_timeval32(struct timeval32 *value)
{
        value->tv_usec = (jiffies % HZ) * (1000000L / HZ);
        value->tv_sec = jiffies / HZ;
}

u32 bspeed_usecs_since(u32 prior_jiffies)
{
	return jiffies_to_usecs(jiffies - prior_jiffies);
}

void bspeed_update_ds_udp_recv(void *sk)
{
	struct sock *udp_sk = (struct sock *)sk;
	if (!sock_flag(udp_sk, SOCK_DEAD)) {
		struct sk_udp_priv *priv;

		priv = (struct sk_udp_priv *) udp_sk->sk_user_data;
		if (priv)
			complete(&priv->compl);
	}
}

__be16 bspeed_udp_v4_csum(int len, __be32 saddr, __be32 daddr, const void *udph)
{
	return udp_v4_check(len, saddr, daddr, csum_partial(udph, len, 0));
}

__be16 bspeed_udp_v6_csum(int len,
			 const struct in6_addr *saddr,
			 const struct in6_addr *daddr,
			 const void *udph)
{
	return udp_v6_check(len, saddr, daddr, csum_partial(udph, len, 0));
}

void bspeed_csum_replace4(__be16 *sum, __be32 from, __be32 to)
{
	csum_replace4(sum, from, to);
}

static int bspeed_read_udp_sock(struct sock *sk, struct iperf_driver_message __user *msg)
{
	int r, accelerated;

	lock_sock(sk);
	r = bspeed_read_udp_connection(sk, msg->data, &accelerated);
	release_sock(sk);
	if (!bspeed_noaccel && !accelerated) {
		r = 0;
	}
	return r;
}

int bspeed_sock_sendmsg_udp(void *sock, void *buff, int len)
{
	struct msghdr msg;
	struct iovec iov;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,0,0))
	int err;

	err = import_single_range(WRITE, buff, len, &iov, &msg.msg_iter);
	if (unlikely(err))
		return err;
#else
	iov.iov_base = buff;
	iov.iov_len = len;
	msg.msg_iov = &iov;
	msg.msg_iovlen = 1;
#endif
	msg.msg_name = NULL;
	msg.msg_control = NULL;
	msg.msg_controllen = 0;
	msg.msg_namelen = 0;
	msg.msg_flags = MSG_DONTWAIT | MSG_NOSIGNAL;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,0,0))
	return sock_sendmsg(sock, &msg);
#else
	return sock_sendmsg(sock, &msg, len);
#endif
}

__poll_t bspeed_udp_poll(struct file *file, struct socket *sock, poll_table *wait)
{
	__poll_t mask = 0;
	struct sk_udp_priv *priv;

	priv = (struct sk_udp_priv *) sock->sk->sk_user_data;

	if (try_wait_for_completion(&priv->compl))
		mask = EPOLLIN | EPOLLRDNORM;

	return mask;
}

static int write_sock(struct socket *sock, void *buff, int len)
{
	if (sock->sk->sk_type == SOCK_STREAM) {
		struct tcp_sock *tp = tcp_sk(sock->sk);
		int mss = tp->mss_cache;
		int wnd = tp->snd_wnd - (tp->write_seq - tp->snd_una);
		return bspeed_write_accelerated(sock, buff, len, mss, wnd);
	} else
		return bspeed_write_accelerated_udp(sock, buff, len);
}

void *bspeed_get_sock_sk(void *sock)
{
	return ((struct socket *)sock)->sk;
}

int bspeed_sock_sendmsg(void *sock, void *buff, int len)
{
	struct msghdr msg;
	struct iovec iov;
	int err;

	err = import_single_range(WRITE, buff, len, &iov, &msg.msg_iter);
	if (unlikely(err))
		return err;
	msg.msg_name = NULL;
	msg.msg_control = NULL;
	msg.msg_controllen = 0;
	msg.msg_namelen = 0;
	msg.msg_flags = MSG_MORE | MSG_DONTWAIT | MSG_NOSIGNAL;
	return sock_sendmsg(sock, &msg);
}

int bspeed_recv(struct iperf_driver_message __user *msg)
{
	int ret;
	struct socket *sock;

	sock = sockfd_lookup(msg->fd, &ret);
	if (!sock) {
		pr_err("%s ERROR: sockfd_lookup failed for fd %llu\n", __func__, msg->fd);
		return ret;
	}
	if (sock->sk->sk_type == SOCK_STREAM) {
		ret = bspeed_read_sock(sock->sk, msg->len);
	} else if (sock->sk->sk_type == SOCK_DGRAM) {
		ret = bspeed_read_udp_sock(sock->sk, msg);
	} else {
		sockfd_put(sock);
		pr_err("%s ERROR: sock type not a stream\n", __func__);
		return -ESOCKTNOSUPPORT;
	}
	sockfd_put(sock);
	return ret;
}

int bspeed_command(struct iperf_driver_message __user *msg)
{
	int ret = -ESOCKTNOSUPPORT;
	struct socket *sock;

	sock = sockfd_lookup(msg->fd, &ret);
	if (!sock)
		return ret;

	if (msg->command == KERNEL_DRIVER_WRITE) {
#ifdef CONFIG_COMPAT
		if (is_compat_task())
			ret = write_sock(sock, compat_ptr((uintptr_t)msg->data), msg->len);
		else
#endif
			ret = write_sock(sock, msg->data, msg->len);
	}
	else if (msg->command == KERNEL_DRIVER_CONNECT) {
		if (sock->sk->sk_type == SOCK_STREAM)
			ret = add_nethook_client(sock->sk, msg->rate,
						 msg->sender ? true : false,
						 0, 0);
		else if (sock->sk->sk_type == SOCK_DGRAM) {
			if (msg->len >= 8) {
				ret = add_nethook_client(sock->sk, msg->rate,
						msg->sender ? true : false,
						(int)((u64 *) msg->data)[0],
						((u64 *) msg->data)[1]);
			} else
				ret = add_nethook_client(sock->sk, msg->rate,
						msg->sender ? true : false,
						1, 0);

			if (msg->sender == false) {
				struct sk_udp_priv *priv;

				priv = kzalloc(sizeof(struct sk_udp_priv), GFP_KERNEL);
				priv->new_ops = kmemdup(sock->ops,
							sizeof(struct proto_ops),
							GFP_KERNEL);

				priv->new_ops->poll = bspeed_udp_poll;
				priv->old_ops = sock->ops;
				sock->ops = priv->new_ops;
				sock->sk->sk_user_data = priv;
				init_completion(&priv->compl);
			}
		}

		if (ret == 0)
			ret = sizeof(struct iperf_driver_message);
	}
	else if (msg->command == KERNEL_DRIVER_DISCONNECT) {
		pr_info("disconnect socket rcvlowat %d rcvbuf %d\n",
			sock->sk->sk_rcvlowat, sock->sk->sk_rcvbuf);
		ret = del_nethook_client(sock->sk);
		if (ret == 0)
			ret = sizeof(struct iperf_driver_message);
		if ((sock->sk->sk_type == SOCK_DGRAM) &&
			(msg->sender == false)){
			struct sk_udp_priv *priv;

			priv = sock->sk->sk_user_data;
			if (priv) {
				complete(&priv->compl);
				sock->ops = priv->old_ops;

				kfree(priv->new_ops);
				kfree(priv);
			}
		}
	}
	sockfd_put(sock);
	return ret;
}

void bspeed_schedule(void)
{
	schedule();
}

void __init bspeed_client_init(void)
{
	BUILD_BUG_ON(sizeof(struct bspeed_fpm_buff) != sizeof(struct fpm_buff));
	BUILD_BUG_ON(offsetof(struct bspeed_fpm_buff, data) !=
		     offsetof(struct fpm_buff, data));
	BUILD_BUG_ON(offsetof(struct bspeed_fpm_buff, dev_out) !=
		     offsetof(struct fpm_buff, dev_out));
}
