/****************************************************************************
*
* Broadcom Proprietary and Confidential. (c) 2018 Broadcom.  All rights reserved.
* The term “Broadcom” refers to Broadcom Limited and/or its subsidiaries.
*
* Unless you and Broadcom execute a separate written software license
* agreement governing use of this software, this software is licensed to
* you under the terms of the GNU General Public License version 2 (the
* "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
* the following added to such license:
*
* As a special exception, the copyright holders of this software give you
* permission to link this software with independent modules, and to copy
* and distribute the resulting executable under terms of your choice,
* provided that you also meet, for each linked independent module, the
* terms and conditions of the license of that module. An independent
* module is a module which is not derived from this software. The special
* exception does not apply to any modifications of the software.
*
* Notwithstanding the above, under no circumstances may you combine this
* software in any way with any other Broadcom software provided under a
* license other than the GPL, without Broadcom's express prior written
* consent.
*
****************************************************************************
* Author: Peter Sulc <peter.sulc@broadcom.com>
* Filename: bspeed_proc.c
* Description: Proc controls for bspeed
****************************************************************************/

#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/ctype.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/inet.h>
#include <linux/version.h>
#include "bspeed.h"

#define PROC_DIR		"driver/bspeed"
#define STATUS_FILE		"status"
#define STATS_FILE		"stats"

struct bspeed_proc_val {
	const char *fname;
	const char *pform;
	u32 *value;
	struct proc_dir_entry *proc_file;
	void (*callback)(u32 value);
};

static void rtt_callback(u32 value);

static struct bspeed_proc_val bspeed_proc_val_table[] = {
	{
		.fname = "ack_delay_start",
		.pform = "%u\n",
		.value = &bspeed_ack_delay_start
	},
	{
		.fname = "ack_delay_limit",
		.pform = "%u\n",
		.value = &bspeed_ack_delay_limit
	},
	{
		.fname = "validate_tcp_checksum",
		.pform = "%u\n",
		.value = &bspeed_validate_checksum
	},
	{
		.fname = "no_accel",
		.pform = "%u\n",
		.value = &bspeed_noaccel
	},
	{
		.fname = "nspp",
		.pform = "%u\n",
		.value = &bspeed_nspp
	},
	{
		.fname = "nspp_static",
		.pform = "%u\n",
		.value = &bspeed_nspp_static
	},
	{
		.fname = "retrans_gap",
		.pform = "%u\n",
		.value = &bspeed_retrans_gap
	},
	{
		.fname = "rcvlowat",
		.pform = "%u\n",
		.value = &bspeed_rcvlowat
	},
	{
		.fname = "rtt_stats",
		.pform = "%u\n",
		.value = &bspeed_rtt_stats,
		.callback = rtt_callback
	},
	{
		.fname = "rtt_max",
		.pform = "%u\n",
		.value = &bspeed_rtt_max
	},
	{
		.fname = "udp_pkt_thresh",
		.pform = "%u\n",
		.value = &bspeed_udp_pkt_thresh
	},
	{
		.fname = "bspeed_no_schedule",
		.pform = "%u\n",
		.value = &bspeed_no_schedule
	},
};

static int   status_proc_open(struct inode *inode, struct file *file);
static int   status_proc_show(struct seq_file *seq, void *v);

static int   stats_proc_open(struct inode *inode, struct file *file);
static int   stats_proc_show(struct seq_file *seq, void *v);

static ssize_t value_read(struct file *, char __user *, size_t, loff_t *);
static ssize_t value_write(struct file *, const char __user *, size_t, loff_t *);
static ssize_t stats_proc_write(struct file *, const char __user *, size_t, loff_t *);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 14, 0)
static const struct proc_ops status_fops = {
	.proc_open		= status_proc_open,
	.proc_read		= seq_read,
	.proc_lseek		= seq_lseek,
	.proc_release		= single_release,
};

static const struct proc_ops stats_fops = {
	.proc_open		= stats_proc_open,
	.proc_read		= seq_read,
	.proc_write		= stats_proc_write,
	.proc_lseek		= seq_lseek,
	.proc_release		= single_release,
};

static const struct proc_ops value_fops = {
	.proc_read		= value_read,
	.proc_write		= value_write,
};
#else /* 5.4 kernel */
static const struct file_operations status_fops = {
	.owner		= THIS_MODULE,
	.open		= status_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= seq_release,
};

static const struct file_operations stats_fops = {
	.owner		= THIS_MODULE,
	.open		= stats_proc_open,
	.read		= seq_read,
	.write		= stats_proc_write,
	.llseek		= seq_lseek,
	.release	= seq_release,
};

static const struct file_operations value_fops = {
	.owner		= THIS_MODULE,
	.read		= value_read,
	.write		= value_write,
};
#endif

static int status_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, status_proc_show, NULL);
}

static int status_proc_show(struct seq_file *seq, void *v)
{
	seq_printf(seq, "ack_delay_start %d\n", bspeed_ack_delay_start);
	seq_printf(seq, "ack_delay_limit %d\n", bspeed_ack_delay_limit);
	seq_printf(seq, "tcp_checksum    %d\n", bspeed_validate_checksum);
	return 0;
}

static int stats_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, stats_proc_show, NULL);
}

static ssize_t
stats_proc_write(struct file *f, const char __user *buf, size_t size, loff_t *ppos)
{
	bspeed_reset_stats();
	return size;
}

static int stats_proc_show(struct seq_file *seq, void *v)
{
	bspeed_print_stats_proc(seq);
	return 0;
}

static struct bspeed_proc_val *get_proc_val(const char *name)
{
	int i;
	int num = sizeof(bspeed_proc_val_table) / sizeof(struct bspeed_proc_val);

	for (i = 0; i < num; i++) {
		if (strcmp(name, bspeed_proc_val_table[i].fname) == 0)
			return &bspeed_proc_val_table[i];
	}
	return NULL;
}

static ssize_t value_read(struct file *f, char __user *buf,
			  size_t size, loff_t *ppos)
{
	int num;
	struct bspeed_proc_val *pv;

	if (*ppos)
		return 0;

	pv = get_proc_val(f->f_path.dentry->d_name.name);
	if (!pv)
		return 0;

	snprintf(buf, size, pv->pform, *pv->value);
	num = strlen(buf);
	*ppos += num;
	return num;
}

static ssize_t value_write(struct file *f, const char __user *buf,
			   size_t size, loff_t *ppos)
{
	char mybuf[16];
	unsigned long value, i;
	struct bspeed_proc_val *pv = get_proc_val(f->f_path.dentry->d_name.name);
	if (!pv)
		return 0;

	for (i = 0; i < sizeof(mybuf); i++) {
		if (buf[i] <= ' ' || buf[i] >= 0x7f) {
			mybuf[i] = 0;
			break;
		}
		mybuf[i] = buf[i];
	}
	if (kstrtoul(mybuf, 0, &value)) {
		pr_err("parse error: %s\n", mybuf);
		return -EINVAL;
	}
	*pv->value = value;

	if (pv->callback)
		pv->callback(value);

	*ppos += size;
	return size;
}

static void rtt_callback(u32 value)
{
	init_net.ipv4.sysctl_tcp_timestamps = value;
}

int __init bspeed_proc_init(void)
{
	int i, num;
	struct bspeed_proc_val *pv;
	struct proc_dir_entry *proc_dir;
	struct proc_dir_entry *proc_file;

	proc_dir = proc_mkdir(PROC_DIR, NULL);
	if (!proc_dir) {
		pr_err("Failed to create PROC directory %s\n", PROC_DIR);
		return -EIO;
	}
	proc_file = proc_create(STATUS_FILE, S_IRUGO, proc_dir, &status_fops);
	if (!proc_file) {
		pr_err("Failed to create %s\n", STATUS_FILE);
		return -EIO;
	}
	proc_file = proc_create(STATS_FILE, S_IRUGO, proc_dir, &stats_fops);
	if (!proc_file) {
		pr_err("Failed to create %s\n", STATS_FILE);
		return -EIO;
	}
	num = sizeof(bspeed_proc_val_table) / sizeof(struct bspeed_proc_val);

	for (i = 0, pv = &bspeed_proc_val_table[0]; i < num; i++, pv++) {
		pv->proc_file = proc_create(pv->fname, S_IRUGO|S_IWUGO,
					    proc_dir, &value_fops);
		if (!pv->proc_file) {
			pr_err("Failed to create %s\n", pv->fname);
			return -EIO;
		}
	}
	return 0;
}

void __exit bspeed_proc_exit(void)
{
	remove_proc_subtree(PROC_DIR, NULL);
}
