/*
 * Copyright (C) 2022 Broadcom Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <linux/version.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/printk.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/ctype.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/kthread.h>
#include <bcmcache.h>

#include "fpm.h"
#include "dqnet.h"
#include "bcmnethooks.h"
#include "bspeed.h"

#include <linux/kfifo.h>

#define FPM_FIFO_DEPTH	64
#define FPM_FIFO_FULL	60

typedef struct {
	struct hrtimer		hrt;
	u32			interval;
	u32			ticks;
	u32			congested;
	u32			uncongested;
	u32			empty;
	u32			samples;
	u32			accum;
	u32			packets;
	bool			hrt_started;
	DECLARE_KFIFO(fpm_fifo, struct bspeed_fpm_buff, FPM_FIFO_DEPTH);
} hrt_control;

static enum hrtimer_restart hrtick(struct hrtimer *timer)
{
	struct bspeed_fpm_buff fb;
	hrt_control *hc = container_of(timer, hrt_control, hrt);
	int n = kfifo_out(&hc->fpm_fifo, &fb, 1);
	if (n) {
		if (!bspeed_tx_fpm_buff(&fb))
			pr_err("%s error sending fpm\n", __func__);
		hc->packets++;
	}
	hc->ticks++;
	if (hc->interval) {
		hrtimer_forward_now(timer, ns_to_ktime(hc->interval));
		return HRTIMER_RESTART;
	}
	while (kfifo_out(&hc->fpm_fifo, &fb, 1))
		if (!bspeed_tx_fpm_buff(&fb))
			pr_err("%s error sending fpm\n", __func__);
	hrtimer_forward_now(timer, ns_to_ktime(100));
	return HRTIMER_RESTART;
}

void *bspeed_hrtimer_create(void)
{
	hrt_control *hc = kmalloc(sizeof(hrt_control), GFP_KERNEL);
	if (!hc)
		return NULL;
	memset(hc, 0, sizeof(*hc));
	INIT_KFIFO(hc->fpm_fifo);
	return hc;
}

void bspeed_hrtimer_destroy(void *vhc)
{
	hrt_control *hc = (hrt_control *)vhc;
	pr_info("Timer Stats:\n");
	pr_info("      Ticks: %u\n", hc->ticks);
	pr_info("    Packets: %u\n", hc->packets);
	pr_info("  Congested: %u\n", hc->congested);
	pr_info("Uncongested: %u\n", hc->uncongested);
	pr_info("      Empty: %u\n", hc->empty);
	pr_info("    Samples: %u\n", hc->samples);
	pr_info("    Average: %u\n", hc->accum / hc->samples);
	pr_info("     On CPU: %u\n", hc->hrt.base? hc->hrt.base->cpu_base->cpu : 0);
	kfree(vhc);
}

void bspeed_hrtimer_init(void *vhc)
{
	hrt_control *hc = (hrt_control *)vhc;
	hrtimer_init(&hc->hrt, CLOCK_MONOTONIC, HRTIMER_MODE_REL_PINNED);
	hc->hrt.function = hrtick;
}

void bspeed_hrtimer_start(void *vhc, u32 interval)
{
	hrt_control *hc = (hrt_control *)vhc;
	hc->interval = interval;
	hrtimer_start(&hc->hrt, ns_to_ktime(interval), HRTIMER_MODE_REL_PINNED);
	hc->hrt_started = true;
}

void bspeed_hrtimer_stop(void *vhc)
{
	struct bspeed_fpm_buff fb;
	hrt_control *hc = (hrt_control *)vhc;
	hc->interval = 0;
	if(hc->hrt_started) {
		hrtimer_cancel(&hc->hrt);
		hc->hrt_started = false;
	}
	while (kfifo_out(&hc->fpm_fifo, &fb, 1))
		if (bcm_nethook_tx_fpm((struct fpm_buff *)&fb, fb.dev_out))
			pr_err("%s error sending fpm\n", __func__);
}

void bspeed_hrtimer_set_interval(void *vhc, u32 interval)
{
	hrt_control *hc = (hrt_control *)vhc;
	hc->interval = interval;
}

bool bspeed_hrtimer_tx_congested(void *vhc)
{
	u32 accum;
	hrt_control *hc = (hrt_control *)vhc;
	int n = kfifo_len(&hc->fpm_fifo);
	accum = hc->accum;
	accum += n;
	if (accum > hc->accum) {
		hc->accum = accum;
		hc->samples++;
	}
	if (n >= FPM_FIFO_FULL) {
		hc->congested++;
		return true;
	}
	hc->uncongested++;
	if (!n)
		hc->empty++;
	return false;
}

bool bspeed_hrtimer_tx_empty(void *vhc)
{
	hrt_control *hc = (hrt_control *)vhc;
	return kfifo_is_empty(&hc->fpm_fifo);
}

bool bspeed_hrtimer_tx_fpm(void *vhc, struct bspeed_fpm_buff *fb)
{
	hrt_control *hc = (hrt_control *)vhc;
	return (kfifo_in(&hc->fpm_fifo, fb, 1) == 1);
}
