/*
 * Copyright (C) 2017 Broadcom Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/miscdevice.h>
#include <linux/printk.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include <linux/ctype.h>
#include <linux/fs.h>

#include "bspeed.h"

static int ref_count = 0;

void *bspeed_get_mso_feat_symbol(void)
{
	return __symbol_get("get_mso_feat");
}

void bspeed_put_mso_feat_symbol(void)
{
	__symbol_put("get_mso_feat");
}

static int bspeed_file_open(struct inode *inode, struct file *file)
{
	if (ref_count)
		return -EBUSY;
	ref_count++;
	return 0;
}

static int bspeed_file_release(struct inode *inode, struct file *file)
{
	if (!ref_count)
		return -ENODEV;
	ref_count--;
	return 0;
}

static ssize_t bspeed_file_read(struct file *file,
			       char __user *buf,
			       size_t count,
			       loff_t *ppos)
{
	struct iperf_driver_message __user *msg = (struct iperf_driver_message *)buf;

	if (msg->command == KERNEL_DRIVER_READ) {
		return bspeed_recv(msg);
	}
	return 0;
}

static ssize_t bspeed_file_write(struct file *file,
				const char __user *buf,
				size_t count,
				loff_t *ppos)
{
	return bspeed_command((struct iperf_driver_message __user *)buf);
}


static const struct file_operations bspeed_fops = {
	.owner =		THIS_MODULE,
	.open =			bspeed_file_open,
	.release =		bspeed_file_release,
	.read =			bspeed_file_read,
	.write =		bspeed_file_write,
};

static struct miscdevice bspeed_device = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = "iperf0",
	.fops = &bspeed_fops,
};

void __exit bspeed_exit(void)
{
	misc_deregister(&bspeed_device);
	bspeed_proc_exit();
}


int __init bspeed_init(void)
{
	if (misc_register(&bspeed_device))
		pr_err("failed to initialize miscdevice %s\n", KERNEL_DRIVER_DEV);

	bspeed_client_init();
	bspeed_proc_init();
	bspeed_fpm_init();

	pr_info("%s initialized\n", bspeed_version_string);
	return 0;
}

module_init(bspeed_init);
module_exit(bspeed_exit);
MODULE_LICENSE("GPL");
