 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential.
 * (c) 2019 Broadcom. All rights reserved.
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Jayesh Patel <jayeshp@broadcom.com>
 ****************************************************************************/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/inetdevice.h>
#include "bcmflow.h"
#include "dqnet_priv.h"
#include "dqnet_fap.h"
#include "flowmgr.h"
#include "flowmgr_fap_ops.h"
#include "rdpa_types.h"
#include "rdpa_cpu_helper.h"
#include "rdpa_ip_class.h"
#include "autogen/rdpa_ag_ip_class.h"
#include "rdpa_iptv.h"
#include "rdpa_wlan_mcast.h"
#include "autogen/rdpa_ag_wlan_mcast.h"
#include "autogen/rdpa_ag_iptv.h"
#include "rdp_init.h"
#include "bdmf_dev.h"
#if defined(CONFIG_BCM_MSO_FEAT_MODULE)
#include "mso_feat.h"
#endif

DEFINE_SPINLOCK(rfap_ops_lock);

int l2flow_nataction_enable;
module_param_named(l2flow_nataction_enable, l2flow_nataction_enable, int, 0644);
MODULE_PARM_DESC(l2flow_nataction_enable, "Enable/Disable NAT action for l2 flows");

static bdmf_object_handle ip_class;
static int ip_class_created;
static bdmf_object_handle iptv;
static int iptv_created;
static bdmf_object_handle wlan_mcast;
static int wlan_mcast_created;
static u32 enable_flag;
extern struct dqnet_fap_ops dqnet_rfap;
atomic_t flow_exist_err_cnt;
atomic_t flow_expected_exist_err_cnt;
atomic_t flowring_err_cnt1;
atomic_t flowring_err_cnt2;
atomic_t netdev_in_down_cnt;
atomic_t unconfig_gre_defer_cnt;
atomic_t mso_feat_err[mf_max];
get_mso_feat_fptr get_mso_feat;

#define MAX_GRE_TUNNELS (RDPA_MAX_GRE_TUNNELS)
static int gre_tunnel_mask;
int max_gre_tunnels = MAX_GRE_TUNNELS;
static int param_set_max_gre(const char *val, const struct kernel_param *kp)
{
	int ret = 0;
	u32 max_gre;

	ret = kstrtouint(val, 10, &max_gre);
	if (ret)
		return ret;
	if (gre_tunnel_mask) {
		pr_err("gre_tunnel_mask must be zero to change max gre value\n");
		return ret;
	}

	if ((max_gre < 1) || (max_gre > MAX_GRE_TUNNELS)) {
		pr_err("Out of recommended range %u, between %u-%u\n",
		       max_gre,
		       1, MAX_GRE_TUNNELS);
		ret = -EINVAL;
	}
	max_gre = clamp(max_gre, (u32)1, (u32)MAX_GRE_TUNNELS);
	max_gre_tunnels= max_gre;
	return ret;
}

static int param_get_max_gre(char *buffer, const struct kernel_param *kp)
{
	u32 max_gre;

	max_gre = *(u32 *)kp->arg;
	/* Buffer can be max 4k */
	return snprintf(buffer, 32, "%u\n", max_gre);
}

static const struct kernel_param_ops param_ops_max_gre = {
	.set = param_set_max_gre,
	.get = param_get_max_gre,
};

module_param_cb(max_gre, &param_ops_max_gre, &max_gre_tunnels, 0644);
MODULE_PARM_DESC(max_gre, "MAX GRE Tunnels Supported");

#define UNCONFIG_GRE_DELAY_MIN 1000
u32 unconfig_gre_delay = UNCONFIG_GRE_DELAY_MIN*10;
static int param_set_unconfig_gre_delay(const char *val, const struct kernel_param *kp)
{
	int ret = 0;
	u32 delay;

	ret = kstrtouint(val, 10, &delay);
	if (ret)
		return ret;

	if (delay < UNCONFIG_GRE_DELAY_MIN) {
		pr_err("Dealy %u must be > %u\n",
		       delay,
		       1000);
		ret = -EINVAL;
		return ret;
	}
	unconfig_gre_delay = delay;
	return ret;
}

static int param_get_unconfig_gre_delay(char *buffer, const struct kernel_param *kp)
{
	u32 delay;

	delay = *(u32 *)kp->arg;
	/* Buffer can be max 4k */
	return snprintf(buffer, 32, "%u\n", delay);
}

static const struct kernel_param_ops param_ops_unconfig_gre_delay = {
	.set = param_set_unconfig_gre_delay,
	.get = param_get_unconfig_gre_delay,
};

module_param_cb(unconfig_gre_delay, &param_ops_unconfig_gre_delay, &unconfig_gre_delay, 0644);
MODULE_PARM_DESC(unconfig_gre_delay, "GRE Unconfig Defer Delay");

#define MAX_MAPT_DOMAINS (RDPA_MAX_MAPT_DOMAINS)

#define VLAN_TAG_LEN	(4)

/* Tag is constructed and desconstructed using byte by byte access
 * because the tag is placed after the MAC Source Address, which does
 * not make it 4-bytes aligned, so this might cause unaligned accesses
 * on most systems where this is used.
 */

/* Ingress and egress opcodes */
#define BRCM_OPCODE_SHIFT	5
#define BRCM_OPCODE_MASK	0x7
#define BRCM_OPCODE_0		0x0
#define BRCM_OPCODE_1		0x1

/* Ingress fields */
/* 1st byte in the tag */
#define BRCM_IG_TC_SHIFT	2
#define BRCM_IG_TC_MASK		0x7
/* 2nd byte in the tag */
#define BRCM_IG_TE_MASK		0x3
#define BRCM_IG_TS_SHIFT	7
/* 3rd byte in the tag */
#define BRCM_IG_DSTMAP2_MASK	1
#define BRCM_IG_DSTMAP1_MASK	0xff

/* Egress fields */

/* 2nd byte in the tag */
#define BRCM_EG_CID_MASK	0xff

/* 3rd byte in the tag */
#define BRCM_EG_RC_MASK		0xff
#define  BRCM_EG_RC_RSVD	(3 << 6)
#define  BRCM_EG_RC_EXCEPTION	(1 << 5)
#define  BRCM_EG_RC_PROT_SNOOP	(1 << 4)
#define  BRCM_EG_RC_PROT_TERM	(1 << 3)
#define  BRCM_EG_RC_SWITCH	(1 << 2)
#define  BRCM_EG_RC_MAC_LEARN	(1 << 1)
#define  BRCM_EG_RC_MIRROR	(1 << 0)
#define BRCM_EG_TC_SHIFT	5
#define BRCM_EG_TC_MASK		0x7
#define BRCM_EG_PID_MASK	0x1f

static void cleanup_rdpa_objects(void)
{
	if (ip_class) {
		if (ip_class_created)
			bdmf_destroy(ip_class);
		else
			bdmf_put(ip_class);
	}
	if (iptv) {
		if (iptv_created)
			bdmf_destroy(iptv);
		else
			bdmf_put(iptv);
	}
	if (wlan_mcast) {
		if (wlan_mcast_created)
			bdmf_destroy(wlan_mcast);
		else
			bdmf_put(wlan_mcast);
	}
}

static int fap_enable(bool enable)
{
	enable_flag = enable;
	return 0;
}

static int ip_class_config(void)
{
	int rc = -1;
	if (rdpa_ip_class_drv())
	{
		rc = rdpa_ip_class_get(&ip_class);
		if (rc) {
			BDMF_MATTR(ip_class_attrs, rdpa_ip_class_drv());
			rc = bdmf_new_and_set(rdpa_ip_class_drv(), NULL,
					      ip_class_attrs, &ip_class);
			if (rc) {
				pr_err("%s: Error - rdpa ip_class object does not exist\n",
				       __func__);
				return rc;
			}
		}
		ip_class_created = 1;
		bdmf_trace_level_set(rdpa_ip_class_drv(), bdmf_trace_level_none);
	}
	else
		ip_class_created = 0;
	return rc;
}

static int iptv_config(void)
{
	int rc = -1;
	if (rdpa_iptv_drv())
	{
		rc = rdpa_iptv_get(&iptv);
		if (rc) {
			BDMF_MATTR(iptv_attrs, rdpa_iptv_drv());
			rdpa_iptv_lookup_method_set(iptv_attrs,
				iptv_lookup_method_group_ip_src_ip);
			rc = bdmf_new_and_set(rdpa_iptv_drv(), NULL,
					      iptv_attrs, &iptv);
			if (rc) {
				pr_err("%s : Error - rdpa iptv object does not exist\n",
				       __func__);
				return rc;
			}
		}
		iptv_created = 1;
	}
	else
		iptv_created = 0;

	rc = rdpa_wlan_mcast_get(&wlan_mcast);
	if (rc) {
		BDMF_MATTR(wlan_mcast_attrs, rdpa_wlan_mcast_drv());
		rc = bdmf_new_and_set(rdpa_wlan_mcast_drv(), NULL,
				      wlan_mcast_attrs, &wlan_mcast);
		if (rc) {
			pr_err("%s : Error rdpa wlan_mcast object does not exist\n",
			       __func__);
			return rc;
		}
	}
	wlan_mcast_created = 1;

	return rc;
}

static int fap_config(char *pri_lan_dev_name, char *wan_dev_name,
		      char *sec_lan_dev_name)
{
	int rc = ip_class_config();
	rc = rc ? rc : iptv_config();
	if (rc)
		cleanup_rdpa_objects();

	return rc;
}

static int fap_config_gre(void *cfg, int oindex)
{
	int rc = -1;
#if defined(CONFIG_BCM_MSO_FEAT_MODULE)
	struct net_device *dev;
	struct dqnet_netdev *ndev;
	struct flow_gre_tunnel_params *params;
	rdpa_gre_tunnel_cfg_t gre = {};
	int id;
	get_mso_feat_fptr get_mso_feat;
	struct mso_feat *feat;

	dev = __dev_get_by_index(&init_net, oindex);
	if (!dev)
		return -1;
	ndev = netdev_priv(dev);
	if (!ndev)
		return -1;

	spin_lock(&rfap_ops_lock);

	params = (struct flow_gre_tunnel_params *) cfg;
	if (!params->tunnel_id) {
		for (id = 0; id < max_gre_tunnels; id++) {
			if (!(gre_tunnel_mask & (1<<id)))
				break;
		}

		if (id == max_gre_tunnels) {
			net_dbg_ratelimited("FLOWMGR: All HW GRE tunnel resources are used\n");
			rc = -1;
			goto done;
		}
	} else
		id = params->tunnel_id-1;

	get_mso_feat = get_mso_feat_symbol();
	if (!get_mso_feat) {
		atomic_inc(&mso_feat_err[0]);
		net_dbg_ratelimited("FLOWMGR: MSO Feat Module not loaded\n");
		goto done;
	}

	feat = get_mso_feat(mf_langre);
	if (!feat) {
		atomic_inc(&mso_feat_err[mf_langre]);
		net_dbg_ratelimited("FLOWMGR: MSO Feat GRE APIs not supported\n");
		goto done;
	}

	if (!feat->gre) {
		atomic_inc(&mso_feat_err[mf_langre]);
		net_dbg_ratelimited("FLOWMGR: MSO GRE Module not defined\n");
		goto done;
	}
	if (!feat->gre->ops1) {
		atomic_inc(&mso_feat_err[mf_langre]);
		net_dbg_ratelimited("FLOWMGR: MSO GRE Module ops1 not defined\n");
		goto done;
	}
	rc = feat->gre->ops1(params, sizeof(struct flow_gre_tunnel_params),
						 &gre, sizeof(rdpa_gre_tunnel_cfg_t),
						 (BCM_NETDEVICE_GROUP_TYPE(dev->group) == BCM_NETDEVICE_GROUP_WAN) ? 1 : 0);
	if (rc) {
		atomic_inc(&mso_feat_err[mf_langre]);
		net_dbg_ratelimited("FLOWMGR: MSO GRE Module ops1 failed %d %s:%d\n", rc,
							dev->name, dev->group);
		goto done;
	}

	if ((ndev->if_id >= rdpa_if_lan0) &&
	    (ndev->if_id <= rdpa_if_lan5)) {
		gre.tag  = (ndev->brcmtag_opc_uc << BRCM_OPCODE_SHIFT);
		gre.tag  <<= 24;
		gre.tag  |= (1 << rdpa_if_to_switch_port(ndev->if_id)) &
			BRCM_IG_DSTMAP1_MASK;
		/* BRCM Tag 4 bytes */
		gre.mtu += 4;
	} else if (params->vlan_tag)
		gre.tag = (0x8100 << 16) | params->vlan_tag;
	gre.mtu += params->encap_mtu;
	if (params->type == ft_ipv6)
		pr_alert("FLOWMGR: gre tunnel %d: type %d smac %pM dmac %pM sip %pI6c dip %pI6c vid %x mtu %d\n",
			 id, gre.tunnel_type, &gre.src_mac, &gre.dst_mac,
			 params->local_ip, params->remote_ip, gre.tag,
			 gre.mtu);
	else
		pr_alert("FLOWMGR: gre tunnel %d: type %d smac %pM dmac %pM sip %pI4 dip %pI4 vid %x mtu %d\n",
			 id, gre.tunnel_type, &gre.src_mac, &gre.dst_mac,
			 &params->local_ip[0], &params->remote_ip[0], gre.tag,
			 gre.mtu);


	rc = rdpa_ip_class_gre_tunnels_set(ip_class, id, &gre);
	if ( rc == 0) {
		gre_tunnel_mask |= (1<<id);
		params->tunnel_id = id+1; /* Return 1 based index */
	}
done:
	spin_unlock(&rfap_ops_lock);
#endif
	return rc;
}


struct unconfig_gre_work {
	struct work_struct work;
	int id;
	u32 delay;
};

static int unconfig_gre(int id)
{
	rdpa_gre_tunnel_cfg_t gre;
	int rc = 0;

	id = id -1; /* Convert it to 0 based index */

	if ((id < 0) || (id >= max_gre_tunnels)) {
		pr_err("FLOWMGR: GRE tunnel %d invalid\n", id);
		return -1;
	}

	spin_lock(&rfap_ops_lock);
	if (!(gre_tunnel_mask & (1<<id))) {
		pr_err("FLOWMGR: GRE tunnel %d not configured\n", id);
		rc = -1;
		goto done;
	}

	memset(&gre, 0, sizeof(rdpa_gre_tunnel_cfg_t));
	gre.mtu = 1; /* Runner API require this to be non zero */
	rc = rdpa_ip_class_gre_tunnels_set(ip_class, id, &gre);
	if ( rc == 0)
		gre_tunnel_mask &= ~(1<<id);
	else
		net_dbg_ratelimited("FLOWMGR: HW GRE tunnel id %d not removed\n", id);
done:
	spin_unlock(&rfap_ops_lock);
	return rc;
}

static int force_unconfig_gre(int id)
{
	int rc;
	rdpa_ip_flow_info_t f = {};
	bdmf_index i;
	int delcnt = 0;

	for (i = 0; i < 65536; i++) {
		rc = rdpa_ip_class_flow_find(ip_class, &i, &f);
		if (rc == BDMF_ERR_OK) {
			if ((f.result.action_vec & rdpa_fc_action_l2gre_decap) &&
			    (f.result.gre_tunnel_idx == (id-1))) {
				rc = rdpa_ip_class_flow_delete(ip_class, i);
				if (rc < 0) {
					net_err_ratelimited("%s: Flow delete Error %d: flow id %ld\n",
							    __FUNCTION__, rc, i);
				} else
					delcnt++;
			}
		}
	}
	pr_alert("FLOWMGR: HW GRE tunnel id %d flows %d removed\n", id, delcnt);
	rc = unconfig_gre(id);
	if (rc)
		pr_err("FLOWMGR: HW GRE tunnel id %d force delete Error %d\n", id, rc);
	else
		pr_alert("FLOWMGR: HW GRE tunnel id %d force delete Success\n", id);
	return rc;
}

void unconfig_gre_work(struct work_struct *work)
{
	struct unconfig_gre_work *w;
	int rc = 0;

	w = container_of(work, struct unconfig_gre_work, work);
	atomic_inc(&unconfig_gre_defer_cnt);
	msleep(w->delay);
	rc = unconfig_gre(w->id);
	if (rc < 0)
		rc = force_unconfig_gre(w->id);
	kfree(w);
	module_put(THIS_MODULE);
}

int schedule_unconfig_gre_work(int id)
{
	struct unconfig_gre_work *w;

	if (!try_module_get(THIS_MODULE))
		goto fail;
	w = kzalloc(sizeof(*w), GFP_ATOMIC);
	if (w) {
		INIT_WORK(&w->work, unconfig_gre_work);
		w->id = id;
		w->delay = unconfig_gre_delay;
		pr_alert("FLOWMGR: Defer deleting HW GRE tunnel id %d\n", id);
		schedule_work(&w->work);

		return 0;
	}
	module_put(THIS_MODULE);
fail:
	pr_err("FLOWMGR: Error %s\n", __func__);
	return -1;
}

static int fap_unconfig_gre(int id)
{
	int rc = 0;

	rc = unconfig_gre(id);
	if (rc < 0)
		schedule_unconfig_gre_work(id);
	return rc;
}

static int fap_config_mapt(void *params, int oindex)
{
	int rc = -1;
#if defined(CONFIG_BCM_MSO_FEAT_MODULE)
	struct flow_mapt_params *mapt;
	rdpa_mapt_xlation_cfg_t mapt_x;
	int id;
	get_mso_feat_fptr get_mso_feat;
	struct mso_feat *feat;

	mapt = (struct flow_mapt_params *)params;
	if (!mapt)
		return -1;

	spin_lock(&rfap_ops_lock);

	id = (int)mapt->domain_index;
	if (id > MAX_MAPT_DOMAINS) {
		pr_err("FLOWMGR: MAPT domain %d invalid\n", id);
		goto done;
	}

	get_mso_feat = get_mso_feat_symbol();
	if (!get_mso_feat) {
		atomic_inc(&mso_feat_err[0]);
		net_dbg_ratelimited("FLOWMGR: MSO Feat Module not loaded\n");
		goto done;
	}

	feat = get_mso_feat(mf_mapt);
	if (!feat) {
		atomic_inc(&mso_feat_err[mf_mapt]);
		net_dbg_ratelimited("FLOWMGR: MSO Feat MAPT APIs not supported\n");
		goto done;
	}

	if (!feat->mapt || !feat->mapt->ops1) {
		atomic_inc(&mso_feat_err[mf_mapt]);
		net_dbg_ratelimited("FLOWMGR: MSO MAPT Module ops1 not defined\n");
		goto done;
	}
	rc = feat->mapt->ops1(mapt, sizeof(struct flow_mapt_params),
			      &mapt_x, sizeof(rdpa_mapt_xlation_cfg_t));
	if (rc) {
		atomic_inc(&mso_feat_err[mf_mapt]);
		net_dbg_ratelimited("FLOWMGR: MSO MAPT Module ops1 failed %d\n", rc);
		goto done;
	}

	rc = rdpa_ip_class_mapt_xlations_set(ip_class, id, &mapt_x);

done:
	spin_unlock(&rfap_ops_lock);
#endif
	pr_debug("%s %d CONFIG MAPT %d\n", __func__, __LINE__, rc);
	return rc;
}

static int __ssid_add(uint16_t *ssid_vector_p, uint8_t ssid)
{
	uint16_t ssid_mask = 1 << ssid;

	if (*ssid_vector_p & ssid_mask) {
		pr_debug("SSID %d already joined the group (0x%04X)",
			 ssid, *ssid_vector_p);
		return BDMF_ERR_PARM;
	}

	*ssid_vector_p |= ssid_mask;

	return BDMF_ERR_OK;
}

static int __ssid_delete(uint16_t *ssid_vector_p, uint8_t ssid)
{
	uint16_t ssid_mask = 1 << ssid;

	if (!(*ssid_vector_p & ssid_mask)) {
		pr_debug("SSID %d did not join the group (0x%04X)",
			 ssid, *ssid_vector_p);
		return BDMF_ERR_PARM;
	}

	*ssid_vector_p &= ~ssid_mask;

	return BDMF_ERR_OK;
}

static int __wlan_mcast_xlate(rdpa_if port,
			      rdpa_wlan_mcast_fwd_table_t *fwd_table_p,
			      int add)
{
	int ret = BDMF_ERR_OK;

	pr_debug("rdpa_if %d add %d\n", port, add);

	fwd_table_p->wfd_tx_priority = 0;
	fwd_table_p->dhd_station_index =
		RDPA_WLAN_MCAST_DHD_STATION_INDEX_INVALID;

	switch ((port-rdpa_if_ssid0)>>4) {
	case 0:
		if (add) {
			fwd_table_p->wfd_0_priority = 0;
			ret = __ssid_add(&fwd_table_p->wfd_0_ssid_vector,
					 port-rdpa_if_ssid0);
		} else {
			ret = __ssid_delete(&fwd_table_p->wfd_0_ssid_vector,
					    port-rdpa_if_ssid0);
		}
		break;

	case 1:
		if (add) {
			fwd_table_p->wfd_1_priority = 0;
			ret = __ssid_add(&fwd_table_p->wfd_1_ssid_vector,
					 port-rdpa_if_ssid16);
		} else {
			ret = __ssid_delete(&fwd_table_p->wfd_1_ssid_vector,
					    port-rdpa_if_ssid16);
		}
		break;

	case 2:
		if (add) {
			fwd_table_p->wfd_2_priority = 0;
			ret = __ssid_add(&fwd_table_p->wfd_2_ssid_vector,
					 port-rdpa_if_ssid32);
		} else {
			ret = __ssid_delete(&fwd_table_p->wfd_2_ssid_vector,
					    port-rdpa_if_ssid32);
		}
		break;

	default:
		return BDMF_ERR_PARM;
	}

	return ret;
}

static int wlan_mcast_add(rdpa_if port, bdmf_index *fwd_table_index_p)
{
	rdpa_wlan_mcast_fwd_table_t fwd_table;
	int ret;

	if (*fwd_table_index_p == RDPA_WLAN_MCAST_FWD_TABLE_INDEX_INVALID) {
		/* New FWD table entry */

		memset(&fwd_table, 0, sizeof(rdpa_wlan_mcast_fwd_table_t));

		ret = __wlan_mcast_xlate(port, &fwd_table, 1);
		if (ret)
			return ret;

		ret = rdpa_wlan_mcast_fwd_table_add(wlan_mcast,
						    fwd_table_index_p,
						    &fwd_table);
		if (ret)
			pr_err("FLOWMGR: Could not rdpa_wlan_mcast_fwd_table_add");

		return ret;
	} else {
		/* Existing FWD table entry */

		ret = rdpa_wlan_mcast_fwd_table_get(wlan_mcast,
						    *fwd_table_index_p,
						    &fwd_table);
		if (ret)
			return ret;

		ret = __wlan_mcast_xlate(port, &fwd_table, 1);
		if (ret)
			return ret;

		ret = rdpa_wlan_mcast_fwd_table_set(wlan_mcast,
						    *fwd_table_index_p,
						    &fwd_table);
		if (ret)
			pr_err("FLOWMGR: Could not rdpa_wlan_mcast_fwd_table_set");
	}

	pr_debug("tbl index %ld ssid vector %x %x %x\n",
		 *fwd_table_index_p,
		 fwd_table.wfd_0_ssid_vector,
		 fwd_table.wfd_1_ssid_vector,
		 fwd_table.wfd_2_ssid_vector);

	return ret;
}

static int wlan_mcast_delete(rdpa_if port, bdmf_index *fwd_table_index_p)
{
	rdpa_wlan_mcast_fwd_table_t fwd_table;
	int ret;

	ret = rdpa_wlan_mcast_fwd_table_get(wlan_mcast,
					    *fwd_table_index_p,
					    &fwd_table);
	if (ret) {
		pr_err("FLOWMGR: Could not rdpa_wlan_mcast_fwd_table_get");
		return ret;
	}

	ret = __wlan_mcast_xlate(port, &fwd_table, 0);
	if (ret) {
		pr_err("FLOWMGR: Could not __wlan_mcast_xlate");
		return ret;
	}

	if (!fwd_table.wfd_0_ssid_vector &&
	    !fwd_table.wfd_1_ssid_vector &&
	    !fwd_table.wfd_2_ssid_vector) {
		ret = rdpa_wlan_mcast_fwd_table_delete(wlan_mcast,
						       *fwd_table_index_p);

		*fwd_table_index_p =
			RDPA_WLAN_MCAST_FWD_TABLE_INDEX_INVALID;

		return ret;
	} else
		return rdpa_wlan_mcast_fwd_table_set(wlan_mcast,
						     *fwd_table_index_p,
						     &fwd_table);
}

static int fap_flow_iptv_add(struct flow_params *params, int *flow_id)
{
	struct net_device *dev;
	struct dqnet_netdev *ndev;
	rdpa_iptv_channel_request_t request = {};
	rdpa_channel_req_key_t request_key;
	bdmf_index wlan_mcast_idx = RDPA_WLAN_MCAST_FWD_TABLE_INDEX_INVALID;
	rdpa_iptv_channel_t channel;
	bdmf_index tmp;

	int rc = -1;
	if (!iptv_created)
		return -1;
	dev = __dev_get_by_index(&init_net,
				 params->tx.interface);
	if (!dev)
		return -1;
	ndev = netdev_priv(dev);
	if (!ndev)
		return -1;

	if (params->multicast.group_id[0]) {
		request.key.mcast_group.l3.gr_ip.family =
			bdmf_ip_family_ipv6;
		memcpy(request.key.mcast_group.l3.gr_ip.addr.ipv6.data,
		       params->multicast.group_id, 16);
	} else {
		request.key.mcast_group.l3.gr_ip.family =
			bdmf_ip_family_ipv4;
		request.key.mcast_group.l3.gr_ip.addr.ipv4 =
			ntohl(params->multicast.group_id[3]);
	}

	request.mcast_result.egress_port = ndev->if_id;
	if ((ndev->if_id >= rdpa_if_ssid0) &&
	    (ndev->if_id <= rdpa_if_ssid47)) {

		if (!wlan_mcast)
			return -1;

		channel.key = request.key;
		rc = rdpa_iptv_channel_find(iptv, &tmp, &channel);
		if (!rc) {
			/* In case channel already exist use existing
			   wlan_mcast entry */
			wlan_mcast_idx = channel.wlan_mcast_index;
		}
		pr_debug("port %d mcast_idx %ld rc %d\n",
		       request.mcast_result.egress_port, wlan_mcast_idx, rc);
		rc = wlan_mcast_add(ndev->if_id, &wlan_mcast_idx);
		if (rc == BDMF_ERR_NOENT) {
			wlan_mcast_idx =
				RDPA_WLAN_MCAST_FWD_TABLE_INDEX_INVALID;
			rc = wlan_mcast_add(ndev->if_id, &wlan_mcast_idx);
		} else if (rc < 0) {
			pr_debug(
			   "FLOWMGR: Entry exists: port %d mcast_idx %ld rc %d\n",
			   request.mcast_result.egress_port,
			   wlan_mcast_idx, rc);
			return 1;
		}
		if (rc < 0) {
			pr_err("FLOWMGR: Error port %d mcast_idx %ld rc %d\n",
			       request.mcast_result.egress_port,
			       wlan_mcast_idx, rc);
			return rc;
		}
	}
	request.wlan_mcast_index = wlan_mcast_idx;

	rc = rdpa_iptv_channel_request_add(iptv, &request_key, &request);

	pr_debug("port %d mcast_idx %ld rc %d\n",
		 request.mcast_result.egress_port, wlan_mcast_idx, rc);

	if (rc < 0)
		return rc;
	*flow_id = (int)request_key.channel_index;
	return 0;
}


static inline
void flow_info_key_ipv4_dump(rdpa_ip_flow_info_t *ip_flow)
{
	__be32 ip1, ip2;
	ip1 = htonl(ip_flow->key.src_ip.addr.ipv4);
	ip2 = htonl(ip_flow->key.dst_ip.addr.ipv4);
	pr_alert("FLOWMGR: Runner key={src_ip=%pI4,dst_ip=%pI4,prot=%d,src_port=%d,dst_port=%d,dir=%s}\n",
		 &ip1,
		 &ip2,
		 ip_flow->key.prot,
		 ip_flow->key.src_port,
		 ip_flow->key.dst_port,
		 ip_flow->key.dir ? "us" : "ds");
}

static inline
void flow_info_key_ipv6_dump(rdpa_ip_flow_info_t *ip_flow)
{
	pr_alert("FLOWMGR: Runner key={src_ip=%pI6c,dst_ip=%pI6c,prot=%d,src_port=%d,dst_port=%d,dir=%s}\n",
		 ip_flow->key.src_ip.addr.ipv6.data,
		 ip_flow->key.dst_ip.addr.ipv6.data,
		 ip_flow->key.prot,
		 ip_flow->key.src_port,
		 ip_flow->key.dst_port,
		 ip_flow->key.dir ? "us" : "ds");
}

static inline
void flow_info_result_short_dump(rdpa_ip_flow_info_t *ip_flow)
{
	pr_alert("FLOWMGR: Runner result={action=%d|%x,trap_reason=%d,dscp_value=%d,"
		 "port=%d,phy_port=%d,queue_id=%d,wan_flow=%d}\n",
		 ip_flow->result.action,
		 ip_flow->result.action_vec,
		 ip_flow->result.trap_reason,
		 ip_flow->result.dscp_value,
		 ip_flow->result.port,
		 ip_flow->result.phy_port,
		 ip_flow->result.queue_id,
		 ip_flow->result.wan_flow);
}

static inline
void flow_info_result_dump(rdpa_ip_flow_info_t *ip_flow)
{
	__be32 ip1, ip2;
	uint8_t l2_header[2*RDPA_L2_HEADER_SIZE+1] = {};
	hex_dump_to_buffer(ip_flow->result.l2_header,
			   ip_flow->result.l2_header_size,
			   32, 1,
			   l2_header, 2*RDPA_L2_HEADER_SIZE,
			   false);
	flow_info_result_short_dump(ip_flow);
	if (ip_flow->result.action_vec & rdpa_fc_action_nat) {
		ip1 = htonl(ip_flow->result.nat_ip.addr.ipv4);
		pr_alert("FLOWMGR: Runner result={nat_port=%d,nat_ip=%pI4}\n",
			 ip_flow->result.nat_port,
			 &ip1);
	}
	pr_alert("FLOWMGR: Runner result={l2_offset=%d,l2_head_size=%d,l2_header=%s}\n",
		 ip_flow->result.l2_header_offset,
		 ip_flow->result.l2_header_size,
		 l2_header);
	pr_cont("FLOWMGR: Runner result={wl_accel_type=%d,wl_metadata=0x%x,gre_tunnel=%d}\n",
		 ip_flow->result.wl_accel_type,
		 ip_flow->result.wl_metadata,
		 ip_flow->result.gre_tunnel_idx);
	if (ip_flow->result.action_vec & rdpa_fc_action_map_xlate) {
		ip1 = htonl(ip_flow->result.src_ip4);
		ip2 = htonl(ip_flow->result.dst_ip4);
		pr_alert("FLOWMGR: Runner result={mapt_domain=%d,dst_port=%d,"
			 "src_ipv4=%pI4,dst_ip4=%pI4,"
			 "src_ip6=%pI6c,dst_ip6=%pI6c}\n",
			 ip_flow->result.mapt_domain,
			 ip_flow->result.dst_port,
			 &ip1,
			 &ip2,
			 ip_flow->result.src_ip6.data,
			 ip_flow->result.dst_ip6.data);
		ip1 = htonl(ip_flow->result.nat_ip.addr.ipv4);
		pr_alert("FLOWMGR: Runner result={nat_port=%d,nat_ip=%pI4}\n",
			 ip_flow->result.nat_port,
			 &ip1);
	}
}

static void set_l2_ether_type(rdpa_ip_flow_result_t *result, __be16 type)
{
	result->l2_header[result->l2_header_size] = type>>8;
	result->l2_header[result->l2_header_size + 1] = type&0xFF;
	result->l2_header_size += 2;
}

static int handle_flow_exist_err(struct flow_params *params,
				 struct net_device *dev_in,
				 struct net_device *dev,
				 rdpa_ip_flow_info_t *ip_flow,
				 int *flow_id)
{
	int rc;
	rdpa_ip_flow_info_t rdpa_flow = {};
	bdmf_index index = -1;
	bdmf_number nflows;

	rc = rdpa_ip_class_flow_find(ip_class, &index, ip_flow);
	if (rc == BDMF_ERR_NOENT) {
		/* If flow is not found with the key, then add the flow
		   to increment the internal runner counter */
		rc = rdpa_ip_class_flow_add(ip_class, &index, ip_flow);
		return rc;
	}
	if (ip_flow->key.src_ip.family == bdmf_ip_family_ipv4)
		goto flow_v4;
	else if (ip_flow->key.src_ip.family == bdmf_ip_family_ipv6)
		goto flow_v6;
	return BDMF_ERR_OK;

flow_v4:
	*flow_id = (int)index;
	rdpa_ip_class_nflows_get(ip_class, &nflows);
	net_dbg_ratelimited("%s: Flow exist %d: IPv4 Key: %pI4 %pI4 %05d %05d %d %d type %d %s - > %s:"
			    " Active flows %lld flow id %d\n",
			    __FUNCTION__, rc,
			    &params->ipv4.src_ip,
			    &params->ipv4.dst_ip,
			    ip_flow->key.src_port,
			    ip_flow->key.dst_port,
			    ip_flow->key.prot,
			    ip_flow->key.dir,
			    params->ipv4.type,
			    dev_in ? dev_in->name : "null",
			    dev ? dev->name : "null",
			    nflows, *flow_id);
	rc = rdpa_ip_class_flow_get(ip_class, index, &rdpa_flow);
	if (rc < 0) {
		/* If flow is not found with the index, then add the flow
		   to increment the internal runner counter */
		rc = rdpa_ip_class_flow_add(ip_class, &index, ip_flow);
		return rc;
	}

	params->ipv4.src_ip = ntohl(rdpa_flow.key.src_ip.addr.ipv4);
	params->ipv4.dst_ip = ntohl(rdpa_flow.key.dst_ip.addr.ipv4);
	net_dbg_ratelimited("%s: Flow exist: Runner IPv4 Key: %pI4 %pI4 %05d %05d %d %d\n",
			    __FUNCTION__,
			    &params->ipv4.src_ip,
			    &params->ipv4.dst_ip,
			    rdpa_flow.key.src_port,
			    rdpa_flow.key.dst_port,
			    rdpa_flow.key.prot,
			    rdpa_flow.key.dir);
	goto flow_result;

flow_v6:
	*flow_id = (int)index;
	rdpa_ip_class_nflows_get(ip_class, &nflows);
	net_dbg_ratelimited("%s: Flow exist %d: IPv6 Key: %pI6 %pI6 %05d %05d %d %d type %d %s - > %s:"
			    " Active flows %lld flow id %d\n",
			    __FUNCTION__, rc,
			    params->ipv6.src_ip,
			    params->ipv6.dst_ip,
			    ip_flow->key.src_port,
			    ip_flow->key.dst_port,
			    ip_flow->key.prot,
			    ip_flow->key.dir,
			    params->ipv6.type,
			    dev_in ? dev_in->name : "null",
			    dev ? dev->name : "null",
			    nflows, *flow_id);
	rc = rdpa_ip_class_flow_get(ip_class, index, &rdpa_flow);
	if (rc < 0) {
		/* If flow is not found with the index, then add the flow
		   to increment the internal runner counter */
		rc = rdpa_ip_class_flow_add(ip_class, &index, ip_flow);
		return rc;
	}

	net_dbg_ratelimited("%s: Flow exist: Runner IPv6 Key: %pI6 %pI6 %05d %05d %d %d\n",
			    __FUNCTION__,
			    rdpa_flow.key.src_ip.addr.ipv6.data,
			    rdpa_flow.key.dst_ip.addr.ipv6.data,
			    rdpa_flow.key.src_port,
			    rdpa_flow.key.dst_port,
			    rdpa_flow.key.prot,
			    rdpa_flow.key.dir);

flow_result:
	net_dbg_ratelimited("%s: Flow exist: Runner Result: port %d action %d trap_reason %d "
			    "wan_flow %d l2_header_offset %d l2_header_size %d "
			    "action_vec %x gre_tunnel_idx %d "
			    "wl_accel_type %d phy_port %d queue_id %d\n",
			    __FUNCTION__,
			    rdpa_flow.result.port,
			    rdpa_flow.result.action,
			    rdpa_flow.result.trap_reason,
			    rdpa_flow.result.wan_flow,
			    rdpa_flow.result.l2_header_offset,
			    rdpa_flow.result.l2_header_size,
			    rdpa_flow.result.action_vec,
			    rdpa_flow.result.gre_tunnel_idx,
			    rdpa_flow.result.wl_accel_type,
			    rdpa_flow.result.phy_port,
			    rdpa_flow.result.queue_id);
	if (ip_flow->result.action == rdpa_forward_action_host)
		atomic_inc(&flow_expected_exist_err_cnt);
	else
		atomic_inc(&flow_exist_err_cnt);
	return BDMF_ERR_ALREADY;
}

static int hw_accel_check(struct flow_params *params)
{
	struct mso_feat *feat;
	int rc, mf_id = 0;

	if (!get_mso_feat) {
		get_mso_feat = get_mso_feat_symbol();
		if (!get_mso_feat) {
			atomic_inc(&mso_feat_err[0]);
			net_dbg_ratelimited("FLOWMGR: MSO Feat Module not loaded\n");
			return -EACCES;
		}
	}

	feat = get_mso_feat(mf_check);
	if (!feat || !feat->check || !feat->check->ops0) {
		atomic_inc(&mso_feat_err[mf_check]);
		net_dbg_ratelimited("FLOWMGR: MSO Feat Check APIs not supported\n");
		return -EACCES;
	}

	rc = feat->check->ops0((void *)params, sizeof(struct flow_params),
			       (void *)&mf_id);
	if (rc) {
		atomic_inc(&mso_feat_err[mf_id]);
		return rc;
	}

	return rc;
}

static int hw_accel_check_post(struct flow_params *params,
							   rdpa_ip_flow_info_t *ip_flow)
{
	struct mso_feat *feat;
	int rc, mf_id = 0;

	if (!get_mso_feat) {
		get_mso_feat = get_mso_feat_symbol();
		if (!get_mso_feat) {
			atomic_inc(&mso_feat_err[0]);
			net_dbg_ratelimited("FLOWMGR: MSO Feat Module not loaded\n");
			return -EACCES;
		}
	}

	feat = get_mso_feat(mf_check);
	if (!feat || !feat->check || !feat->check->ops1) {
		atomic_inc(&mso_feat_err[mf_check]);
		net_dbg_ratelimited("FLOWMGR: MSO Feat Check APIs not supported\n");
		return -EACCES;
	}

	rc = feat->check->ops1((void *)params, sizeof(struct flow_params),
						   (void *)ip_flow, sizeof(rdpa_ip_flow_info_t));
	if (rc) {
		atomic_inc(&mso_feat_err[mf_id]);
		return rc;
	}

	return rc;
}

static int fap_flow_ip_add(struct flow_params *params, int *flow_id)
{
	struct net_device *dev;
	struct dqnet_netdev *ndev;
	struct net_device *dev_in;
	struct dqnet_netdev *ndev_in;
	uint8_t *l2_header;
	__u8  *rep_mac_dst = NULL;
	__u8  *rep_mac_src = NULL;
	int vlantaglen = VLAN_TAG_LEN;
	u8  vlantag[VLAN_TAG_LEN];
	bool lan2lan = 0;
	rdpa_ip_flow_info_t ip_flow = {};
	bdmf_index index = 0;
	bdmf_number nflows;
	int rc = -1;
	__be16 ethtype = ETH_P_IP;

	if (!ip_class_created)
		return -1;

	dev = __dev_get_by_index(&init_net,
				 params->tx.interface);
	if (!dev)
		return -1;
	ndev = netdev_priv(dev);
	if (!ndev)
		return -1;

	dev_in = __dev_get_by_index(&init_net,
				    params->rx.interface);
	if (!dev_in)
		return -1;
	ndev_in = netdev_priv(dev_in);
	if (!ndev_in)
		return -1;

	if (!netif_running(dev_in) || !netif_oper_up(dev_in)) {
		atomic_inc(&netdev_in_down_cnt);
		return -1;
	}

	if ((BCM_NETDEVICE_GROUP_TYPE(dev->group) == BCM_NETDEVICE_GROUP_WAN) &&
	    ndev->if_id == rdpa_if_wan1) {
		params->tx.control.ethwan = 1;
	}

	if ((BCM_NETDEVICE_GROUP_TYPE(dev_in->group) == BCM_NETDEVICE_GROUP_WAN) &&
	    ndev_in->if_id == rdpa_if_wan1) {
		params->rx.control.ethwan = 1;
	}

	rc = hw_accel_check(params);
	if (rc)
		return rc;

	if (params->type == ft_ipv4) {
		rep_mac_dst = params->ipv4.replacement_mac_dst;
		rep_mac_src = params->ipv4.replacement_mac_src;
	} else if (params->type == ft_ipv6) {
		rep_mac_dst = params->ipv6.replacement_mac_dst;
		rep_mac_src = params->ipv6.replacement_mac_src;
	} else {
		return -1;
	}

	/* Make sure dqnet initializes this with appropriate rdpa_if */
	ip_flow.result.port = ndev->if_id;
	ip_flow.result.queue_id = 0; /* Single Queue. No TC */
	if (ndev->if_id > rdpa_if_ssid47) {
		return -1;
	} else if (ndev->dhdol_get_flow) {
		u32 flowring_idx;
		int ssid = (ndev->if_id - rdpa_if_ssid0) % 16;
		int radio_idx = (ndev->if_id - rdpa_if_ssid0) / 16;
		/* Get flowring and intialize wl metadeta */
		flowring_idx = params->wifi.flowring;

		if (flowring_idx == 0) {
			atomic_inc(&flowring_err_cnt1);
			return -1;
		}

		if (flowring_idx == WL_METADATA_FLOWRING_MASK) {
			atomic_inc(&flowring_err_cnt2);
			return -1;
		}

		ip_flow.result.wl_accel_type = RDPA_WL_ACCEL_DHD_OFFLOAD;
		ip_flow.result.rnr.is_wfd = 0;
		ip_flow.result.rnr.radio_idx = radio_idx;
		ip_flow.result.rnr.ssid = ssid;
		ip_flow.result.rnr.flowring_idx = flowring_idx;
		ip_flow.result.queue_id = params->wifi.pri;
	} else if (ndev->if_id >= rdpa_if_ssid32) {
		ip_flow.result.queue_id = 4;
#if defined(CONFIG_BCM_NIO)
		ip_flow.result.wl_accel_type = RDPA_WL_ACCEL_BRM_OFFLOAD;
		ip_flow.result.rnr.is_wfd = 0;
		ip_flow.result.rnr.radio_idx = 2;
		ip_flow.result.rnr.is_brm = 1;
#else
		ip_flow.result.wl_accel_type = RDPA_WL_ACCEL_WFD;
#endif
	} else if (ndev->if_id >= rdpa_if_ssid16) {
		ip_flow.result.queue_id = 2;
#if defined(CONFIG_BCM_NIO)
		ip_flow.result.wl_accel_type = RDPA_WL_ACCEL_BRM_OFFLOAD;
		ip_flow.result.rnr.is_wfd = 0;
		ip_flow.result.rnr.radio_idx = 1;
		ip_flow.result.rnr.is_brm = 1;
#else
		ip_flow.result.wl_accel_type = RDPA_WL_ACCEL_WFD;
#endif
	} else if (ndev->if_id >= rdpa_if_ssid0) {
		ip_flow.result.queue_id = 0;
#if defined(CONFIG_BCM_NIO)
		ip_flow.result.wl_accel_type = RDPA_WL_ACCEL_BRM_OFFLOAD;
		ip_flow.result.rnr.is_wfd = 0;
		ip_flow.result.rnr.radio_idx = 0;
		ip_flow.result.rnr.is_brm = 1;
#else
		ip_flow.result.wl_accel_type = RDPA_WL_ACCEL_WFD;
#endif
	}

	/* L2 Info */
	ip_flow.result.l2_header_offset = 0;
	ip_flow.result.l2_header_size = 2*ETH_ALEN; /* exc ETHTYPE(2 bytes) */
	l2_header = ip_flow.result.l2_header;
	memcpy(l2_header, rep_mac_dst, ETH_ALEN);
	l2_header += ETH_ALEN;
	memcpy(l2_header, rep_mac_src, ETH_ALEN);
	l2_header += ETH_ALEN;

	if ((ndev->if_id >= rdpa_if_lan0) &&
	    (ndev->if_id <= rdpa_if_lan5)) {
		/* If packet is destined to Switch add BRCM tag */
		int pri = params->tx.control.priority;
		/* TC to QID mapping as follows
		   TC 1,0 -> QID 1
		   TC 3,2 -> QID 3
		   TC 5,4 -> QID 5
		   TC 7,6 -> QID 7
		   to match switch QOS settings*/
		if ((pri % 2) == 0)
			ip_flow.result.queue_id = pri + 1;
		else
			ip_flow.result.queue_id = pri;
		ip_flow.result.phy_port =
			dqnet_rfap.get_imp_lag_port(dev);
		params->tx.control.lag = ip_flow.result.phy_port;

		if (params->type == ft_ipv4) {
			if ((params->ipv4.type == ft_ipv4_gre_encap_nat_src) ||
			    (params->ipv4.type == ft_ipv4_gre_encap_nat_dst) ||
			    (params->ipv4.type == ft_ipv4_gre_encap) ||
			    (params->ipv4.type == ft_ipv4_dslite2gre))
				goto done_brcmtag;
		} else if (params->type == ft_ipv6) {
			if (params->ipv6.type == ft_ipv6_gre_encap)
				goto done_brcmtag;
		}
		ip_flow.result.l2_header_offset -= vlantaglen;
		ip_flow.result.l2_header_size += vlantaglen;
		vlantag[0] = (ndev->brcmtag_opc_uc << BRCM_OPCODE_SHIFT) |
			((pri & BRCM_IG_TC_MASK) << BRCM_IG_TC_SHIFT);
		vlantag[1] = 0;
		vlantag[2] = 0;
		vlantag[3] = (1 << rdpa_if_to_switch_port(ndev->if_id)) &
			BRCM_IG_DSTMAP1_MASK;
		memcpy(l2_header, vlantag, vlantaglen);
		l2_header += vlantaglen;
	}
done_brcmtag:
	if ((ndev_in->if_id >= rdpa_if_lan0) &&
	    (ndev_in->if_id <= rdpa_if_lan5)) {
		ip_flow.result.l2_header_offset += vlantaglen;
	}

	if (params->tx.control.vlan_tci) {
		ip_flow.result.l2_header_offset -= vlantaglen;
		ip_flow.result.l2_header_size += vlantaglen;
		vlantag[0] = ETH_P_8021Q >> 8;
		vlantag[1] = ETH_P_8021Q & 0xFF;
		vlantag[2] = params->tx.control.vlan_tci >> 8;
		vlantag[3] = params->tx.control.vlan_tci & 0xFF;
		memcpy(l2_header, vlantag, vlantaglen);
		l2_header += vlantaglen;
	}

	if (params->rx.control.vlan_untag)
		ip_flow.result.l2_header_offset += vlantaglen;

	if (ndev->if_id == rdpa_if_wan1) /* BYOI flow */
		ip_flow.result.wan_flow = 1;

	if (BCM_NETDEVICE_GROUP_TYPE(dev->group) ==
	    BCM_NETDEVICE_GROUP_LAN) {
		if (BCM_NETDEVICE_GROUP_TYPE(dev_in->group) ==
		    BCM_NETDEVICE_GROUP_LAN)
			lan2lan = 1;
	}

	ip_flow.key.dir = rdpa_dir_ds;
	if (BCM_NETDEVICE_GROUP_TYPE(dev->group) ==
	    BCM_NETDEVICE_GROUP_WAN) {
		ip_flow.key.dir = rdpa_dir_us;
	}
	if (params->type == ft_mac_bridge) {
		/* Not implemented */
	} else if (params->type == ft_ipv4) {
		if (params->ipv4.type == ft_ipv4_nat_dst) {
			set_l2_ether_type(&(ip_flow.result), ETH_P_IP);
			ip_flow.result.action_vec |= rdpa_fc_action_nat;
			ip_flow.result.action_vec |= rdpa_fc_action_ttl;
			ip_flow.result.nat_ip.addr.ipv4 =
				ntohl(params->ipv4.replacement_ip);
			ip_flow.result.nat_port =
				ntohs(params->ipv4.replacement_port);
			ip_flow.result.nat_ip.family =
				bdmf_ip_family_ipv4;
		} else if (params->ipv4.type == ft_ipv4_nat_src) {
			set_l2_ether_type(&(ip_flow.result), ETH_P_IP);
			ip_flow.result.action_vec |= rdpa_fc_action_nat;
			ip_flow.result.action_vec |= rdpa_fc_action_ttl;
			ip_flow.result.nat_ip.addr.ipv4 =
				ntohl(params->ipv4.replacement_ip);
			ip_flow.result.nat_port =
				ntohs(params->ipv4.replacement_port);
			ip_flow.result.nat_ip.family =
				bdmf_ip_family_ipv4;
		} else if (params->ipv4.type == ft_ipv4_bridge) {
			/* WLAN to LAN or LAN to WLAN */
			ip_flow.key.dir = rdpa_dir_us;
			set_l2_ether_type(&(ip_flow.result), ETH_P_IP);
			if (l2flow_nataction_enable) {
				ip_flow.result.action_vec |= rdpa_fc_action_nat;
				ip_flow.result.action_vec |= rdpa_fc_action_ttl;
				ip_flow.result.nat_ip.addr.ipv4 =
					ntohl(params->ipv4.src_ip);
				ip_flow.result.nat_port =
					ntohs(params->ipv4.src_port);
				ip_flow.result.nat_ip.family =
					bdmf_ip_family_ipv4;
			}
		} else if (params->ipv4.type == ft_ipv4_bridge_ds) {
			set_l2_ether_type(&(ip_flow.result), ETH_P_IP);
			if (l2flow_nataction_enable) {
				ip_flow.result.action_vec |= rdpa_fc_action_nat;
				ip_flow.result.action_vec |= rdpa_fc_action_ttl;
				ip_flow.result.nat_ip.addr.ipv4 =
					ntohl(params->ipv4.dst_ip);
				ip_flow.result.nat_port =
					ntohs(params->ipv4.dst_port);
				ip_flow.result.nat_ip.family =
					bdmf_ip_family_ipv4;
			}
		} else if (params->ipv4.type == ft_ipv4_dslite_encap) {
			ip_flow.key.dir = rdpa_dir_us;
			ip_flow.result.action_vec |=
				rdpa_fc_action_dslite_tunnel;
			ip_flow.result.action_vec |=
				rdpa_fc_action_ttl;
			l2_header = ip_flow.result.l2_header;
			memcpy(l2_header,
			       params->encap_tun.dslite.remote_mac, ETH_ALEN);
			l2_header += ETH_ALEN;
			memcpy(l2_header,
			       params->encap_tun.dslite.local_mac, ETH_ALEN);
			/* IPv6 packet */
			set_l2_ether_type(&(ip_flow.result), ETH_P_IPV6);
			memcpy(&ip_flow.result.ds_lite_src,
			       params->encap_tun.dslite.local_ip,
			       sizeof(bdmf_ipv6_t));
			memcpy(&ip_flow.result.ds_lite_dst,
			       params->encap_tun.dslite.remote_ip,
			       sizeof(bdmf_ipv6_t));
			ip_flow.result.l2_header_offset -= 40;
			/*
			ip_flow.result.action_vec |= rdpa_fc_action_nat;
			ip_flow.result.nat_ip.addr.ipv4 =
				ntohl(params->ipv4.replacement_ip);
			ip_flow.result.nat_port =
				ntohs(params->ipv4.replacement_port);
			ip_flow.result.nat_ip.family =
				bdmf_ip_family_ipv4;
			*/
		} else if (params->ipv4.type == ft_ipv4_dslite_decap) {
			ip_flow.key.dir = rdpa_dir_ds;
			ip_flow.result.action_vec |= rdpa_fc_action_ttl;
			set_l2_ether_type(&(ip_flow.result), ETH_P_IP);
			/*
			ip_flow.result.action_vec |= rdpa_fc_action_nat;
			ip_flow.result.nat_ip.addr.ipv4 =
				ntohl(params->ipv4.replacement_ip);
			ip_flow.result.nat_port =
				ntohs(params->ipv4.replacement_port);
			ip_flow.result.nat_ip.family =
				bdmf_ip_family_ipv4;
			*/
		} else if (params->ipv4.type == ft_ipv4_gre_encap) {
			/*ip_flow.key.dir = rdpa_dir_us;*/
			if (lan2lan)
				ip_flow.key.dir = rdpa_dir_us;
			ip_flow.result.action_vec |=
				rdpa_fc_action_l2gre_encap;
			ip_flow.result.gre_tunnel_idx =
				params->encap_tun.gre.tunnel_id - 1;
			/* add ether_type & increment ethernet header size */
			set_l2_ether_type(&(ip_flow.result), ETH_P_IP);
			/* 14 Ethernet + 20/40 outer IPv4/6 header + 4 Gre */
			if (params->encap_tun.gre.type == ft_ipv6)
				ip_flow.result.l2_header_offset -= 58;
			else
				ip_flow.result.l2_header_offset -= 38;
		} else if ((params->ipv4.type == ft_ipv4_gre_encap_nat_src) ||
			   (params->ipv4.type == ft_ipv4_gre_encap_nat_dst)) {
			/*ip_flow.key.dir = rdpa_dir_us;*/
			if (lan2lan)
				ip_flow.key.dir = rdpa_dir_us;
			ip_flow.result.action_vec |= rdpa_fc_action_nat;
			ip_flow.result.nat_ip.addr.ipv4 =
				ntohl(params->ipv4.replacement_ip);
			ip_flow.result.nat_port =
				ntohs(params->ipv4.replacement_port);
			ip_flow.result.action_vec |=
				rdpa_fc_action_l2gre_encap;
			ip_flow.result.gre_tunnel_idx =
				params->encap_tun.gre.tunnel_id - 1;
			/* add ether_type & increment ethernet header size */
			set_l2_ether_type(&(ip_flow.result), ETH_P_IP);
			/* 14 Ethernet + 20/40 outer IPv4/6 header + 4 Gre */
			if (params->encap_tun.gre.type == ft_ipv6)
				ip_flow.result.l2_header_offset -= 58;
			else
				ip_flow.result.l2_header_offset -= 38;
		} else if (params->ipv4.type == ft_ipv4_gre_decap) {
			/*ip_flow.key.dir = rdpa_dir_ds;*/
			if (lan2lan)
				ip_flow.key.dir = rdpa_dir_us;
			ip_flow.result.action_vec |=
				rdpa_fc_action_l2gre_decap;
			ip_flow.result.gre_tunnel_idx =
				params->decap_tun.gre.tunnel_id - 1;
			set_l2_ether_type(&(ip_flow.result), ETH_P_IP);
		} else if ((params->ipv4.type == ft_ipv4_gre_decap_nat_src) ||
			   (params->ipv4.type == ft_ipv4_gre_decap_nat_dst)) {
			/*ip_flow.key.dir = rdpa_dir_ds;*/
			if (lan2lan)
				ip_flow.key.dir = rdpa_dir_us;
			ip_flow.result.action_vec |= rdpa_fc_action_nat;
			ip_flow.result.nat_ip.addr.ipv4 =
				ntohl(params->ipv4.replacement_ip);
			ip_flow.result.nat_port =
				ntohs(params->ipv4.replacement_port);
			ip_flow.result.action_vec |=
				rdpa_fc_action_l2gre_decap;
			ip_flow.result.gre_tunnel_idx =
				params->decap_tun.gre.tunnel_id - 1;
			set_l2_ether_type(&(ip_flow.result), ETH_P_IP);
		} else if (params->ipv4.type == ft_ipv4_routed_src) {
			if (lan2lan)
				ip_flow.key.dir = rdpa_dir_us;
			ip_flow.result.action_vec |= rdpa_fc_action_ttl;
			set_l2_ether_type(&(ip_flow.result), ETH_P_IP);
		} else if (params->ipv4.type == ft_ipv4_routed_dst) {
			ip_flow.result.action_vec |= rdpa_fc_action_ttl;
			set_l2_ether_type(&(ip_flow.result), ETH_P_IP);
		} else if (params->ipv4.type == ft_ipv4_mapt_map) {
			ip_flow.result.action_vec |= rdpa_fc_action_map_xlate;
			set_l2_ether_type(&(ip_flow.result), ETH_P_IPV6);
			ip_flow.result.nat_ip.addr.ipv4 =
				ntohl(params->ipv4.replacement_ip);
			ip_flow.result.nat_port =
				ntohs(params->ipv4.replacement_port);
			ip_flow.result.nat_ip.family =
				bdmf_ip_family_ipv4;
			ip_flow.result.mapt_domain = params->ipv4.mapt.domain_index;
			memcpy(&ip_flow.result.src_ip6, &params->ipv4.mapt.ipv6_src_ip,
				sizeof(bdmf_ipv6_t));
			memcpy(&ip_flow.result.dst_ip6, &params->ipv4.mapt.ipv6_dst_ip,
				sizeof(bdmf_ipv6_t));
		} else if (params->ipv4.type == ft_ipv4_gre2dslite) {
			ip_flow.key.dir = rdpa_dir_us;
			ip_flow.result.action_vec |=
				rdpa_fc_action_dslite_tunnel;
			ip_flow.result.action_vec |=
				rdpa_fc_action_ttl;
			l2_header = ip_flow.result.l2_header;
			memcpy(l2_header,
			       params->encap_tun.dslite.remote_mac, ETH_ALEN);
			l2_header += ETH_ALEN;
			memcpy(l2_header,
			       params->encap_tun.dslite.local_mac, ETH_ALEN);
			/* IPv6 packet */
			set_l2_ether_type(&(ip_flow.result), ETH_P_IPV6);
			memcpy(&ip_flow.result.ds_lite_src,
			       params->encap_tun.dslite.local_ip,
			       sizeof(bdmf_ipv6_t));
			memcpy(&ip_flow.result.ds_lite_dst,
			       params->encap_tun.dslite.remote_ip,
			       sizeof(bdmf_ipv6_t));
			ip_flow.result.l2_header_offset -= 40;
			ip_flow.result.action_vec |=
				rdpa_fc_action_l2gre_decap;
			ip_flow.result.gre_tunnel_idx =
				params->decap_tun.gre.tunnel_id - 1;
			if (params->encap_tun.gre.type == ft_ipv6)
				ip_flow.result.l2_header_offset += 58;
			else
				ip_flow.result.l2_header_offset += 38;
		} else if (params->ipv4.type == ft_ipv4_dslite2gre) {
			ip_flow.key.dir = rdpa_dir_ds;
			ip_flow.result.action_vec |= rdpa_fc_action_ttl;
			ip_flow.result.l2_header_offset += 40;
			ip_flow.result.action_vec |=
				rdpa_fc_action_l2gre_encap;
			ip_flow.result.gre_tunnel_idx =
				params->encap_tun.gre.tunnel_id - 1;
			set_l2_ether_type(&(ip_flow.result), ETH_P_IP);
			if (params->encap_tun.gre.type == ft_ipv6)
				ip_flow.result.l2_header_offset -= 58;
			else
				ip_flow.result.l2_header_offset -= 38;
		} else if (params->ipv4.type == ft_ipv4_mapt_gre2map) {
			ip_flow.result.action_vec |= rdpa_fc_action_map_xlate;
			ip_flow.result.action_vec |= rdpa_fc_action_l2gre_decap;
			ip_flow.result.gre_tunnel_idx =
				params->decap_tun.gre.tunnel_id - 1;
			ip_flow.result.l2_header_offset -= 20;
			set_l2_ether_type(&(ip_flow.result), ETH_P_IPV6);
			ip_flow.result.nat_ip.addr.ipv4 =
				ntohl(params->ipv4.replacement_ip);
			ip_flow.result.nat_port =
				ntohs(params->ipv4.replacement_port);
			ip_flow.result.nat_ip.family =
				bdmf_ip_family_ipv4;
			ip_flow.result.mapt_domain = params->ipv4.mapt.domain_index;
			memcpy(&ip_flow.result.src_ip6, &params->ipv4.mapt.ipv6_src_ip,
				sizeof(bdmf_ipv6_t));
			memcpy(&ip_flow.result.dst_ip6, &params->ipv4.mapt.ipv6_dst_ip,
				sizeof(bdmf_ipv6_t));
		}

		ip_flow.key.prot = params->ipv4.ip_prot;
		ip_flow.key.src_ip.addr.ipv4 = ntohl(params->ipv4.src_ip);
		ip_flow.key.dst_ip.addr.ipv4 = ntohl(params->ipv4.dst_ip);
		if (params->ipv4.ip_prot == IPPROTO_ESP) {
			u32 spi = params->ipsec.spi;
			ip_flow.key.dst_port = ntohl(spi) & 0xFFFF;
			ip_flow.key.src_port = ntohl(spi) >> 16;
		} else if (params->ipv4.ip_prot == IPPROTO_AH) {
			ip_flow.key.src_port = 0;
			ip_flow.key.dst_port = 0;
		} else {
			ip_flow.key.src_port = ntohs(params->ipv4.src_port);
			ip_flow.key.dst_port = ntohs(params->ipv4.dst_port);
		}
		ip_flow.key.src_ip.family = bdmf_ip_family_ipv4;
		ip_flow.key.dst_ip.family = bdmf_ip_family_ipv4;

		if (params->tx.control.dscp_mark) {
			ip_flow.result.action_vec |= rdpa_fc_action_dscp_remark;
			ip_flow.result.dscp_value = params->tx.control.dscp_val;
		}

		if (ip_flow.result.action_vec & rdpa_fc_action_nat) {
			if ((ip_flow.result.nat_ip.addr.ipv4 == ip_flow.key.src_ip.addr.ipv4) ||
			    (ip_flow.result.nat_ip.addr.ipv4 == ip_flow.key.dst_ip.addr.ipv4)) {
				net_dbg_ratelimited("%s: Error Nat %d: IPv4 Key: %pI4 %pI4 %05d %05d %d %d %d %s - > %s\n",
						    __FUNCTION__, rc,
						    &params->ipv4.src_ip,
						    &params->ipv4.dst_ip,
						    ip_flow.key.src_port,
						    ip_flow.key.dst_port,
						    ip_flow.key.prot,
						    ip_flow.key.dir,
						    params->ipv4.type,
						    dev_in->name,
						    dev->name);
				return BDMF_ERR_PARM;
			}
		}

		rc = rdpa_ip_class_flow_add(ip_class, &index, &ip_flow);
		if (params->debug) {
			flow_info_key_ipv4_dump(&ip_flow);
			flow_info_result_dump(&ip_flow);
			pr_alert("Runner return code=%d\n",rc);
		}
		if (rc == BDMF_ERR_ALREADY) {
			return handle_flow_exist_err(params, dev_in, dev,
						     &ip_flow, flow_id);
		} else if (rc < 0) {
			rdpa_ip_class_nflows_get(ip_class, &nflows);
			if (params->debug)
				pr_alert("%s: Error %d: Active flows %lld\n",
					 __FUNCTION__, rc,
					 nflows);
			return rc;
		}
		*flow_id = (int)index;
		return 0;
	} else if (params->type == ft_ipv6) {
		ethtype = ETH_P_IPV6;

		if (params->ipv6.type == ft_ipv6_routed) {
			/* TTL decrement action */
			if (lan2lan)
				ip_flow.key.dir = rdpa_dir_us;
			ip_flow.result.action_vec |= rdpa_fc_action_ttl;
		} else if (params->ipv6.type == ft_ipv6_bridge) {
			/* WLAN to LAN or LAN to WLAN */
			ip_flow.key.dir = rdpa_dir_us;
		} else if (params->ipv6.type == ft_ipv6_bridge_ds) {
			ip_flow.key.dir = rdpa_dir_ds;
		} else if (params->ipv6.type == ft_ipv6_mapt_unmap) {
			ip_flow.result.action_vec |= rdpa_fc_action_map_xlate;
			ethtype = ETH_P_IP;
			ip_flow.result.nat_ip.addr.ipv4 =
				ntohl(params->ipv6.mapt.dst_ip);
			ip_flow.result.nat_port =
				ntohs(params->ipv6.mapt.dst_port);
			ip_flow.result.nat_ip.family =
				bdmf_ip_family_ipv4;
			ip_flow.result.mapt_domain = params->ipv6.mapt.domain_index;
			ip_flow.result.src_ip4 =
				ntohl(params->ipv6.mapt.src_ip);
			ip_flow.result.dst_ip4 =
				ntohl(params->ipv6.mapt.dst_ip);
		} else if (params->ipv6.type == ft_ipv6_gre_encap) {
			if (lan2lan)
				ip_flow.key.dir = rdpa_dir_us;
			ip_flow.result.action_vec |=
				rdpa_fc_action_l2gre_encap;
			ip_flow.result.gre_tunnel_idx =
				params->encap_tun.gre.tunnel_id - 1;
			/* 14 Ethernet + 20 outer IPv4 header + 4 Gre */
			if (params->encap_tun.gre.type == ft_ipv6)
				ip_flow.result.l2_header_offset -= 58;
			else
				ip_flow.result.l2_header_offset -= 38;
		} else if (params->ipv6.type == ft_ipv6_gre_decap) {
			if (lan2lan)
				ip_flow.key.dir = rdpa_dir_us;
			ip_flow.result.action_vec |=
				rdpa_fc_action_l2gre_decap;
			ip_flow.result.gre_tunnel_idx =
				params->decap_tun.gre.tunnel_id - 1;
		} else if (params->ipv6.type == ft_ipv6_mapt_unmap2gre) {
			ip_flow.result.action_vec |= rdpa_fc_action_map_xlate;
			ip_flow.result.action_vec |= rdpa_fc_action_l2gre_encap;
			ip_flow.result.gre_tunnel_idx =
				params->encap_tun.gre.tunnel_id - 1;
			/* 14 Ethernet + 20 outer IPv4 header + 4 Gre */
			if (params->encap_tun.gre.type == ft_ipv6)
				ip_flow.result.l2_header_offset -= 58;
			else
				ip_flow.result.l2_header_offset -= 38;
			ethtype = ETH_P_IP;
			ip_flow.result.nat_ip.addr.ipv4 =
				ntohl(params->ipv6.mapt.dst_ip);
			ip_flow.result.nat_port =
				ntohs(params->ipv6.mapt.dst_port);
			ip_flow.result.nat_ip.family =
				bdmf_ip_family_ipv4;
			ip_flow.result.mapt_domain = params->ipv6.mapt.domain_index;
			ip_flow.result.src_ip4 =
				ntohl(params->ipv6.mapt.src_ip);
			ip_flow.result.dst_ip4 =
				ntohl(params->ipv6.mapt.dst_ip);
		} else {
		}
		ip_flow.key.prot = params->ipv6.ip_prot;
		memcpy(&ip_flow.key.src_ip.addr.ipv6, params->ipv6.src_ip,
		       sizeof(bdmf_ipv6_t));
		memcpy(&ip_flow.key.dst_ip.addr.ipv6, params->ipv6.dst_ip,
		       sizeof(bdmf_ipv6_t));

		if (params->ipv6.ip_prot == IPPROTO_ESP) {
			u32 spi = params->ipsec.spi;
			ip_flow.key.dst_port = ntohl(spi) & 0xFFFF;
			ip_flow.key.src_port = ntohl(spi) >> 16;
		} else if (params->ipv6.ip_prot == IPPROTO_AH) {
			ip_flow.key.src_port = 0;
			ip_flow.key.dst_port = 0;
		} else {
			ip_flow.key.src_port = ntohs(params->ipv6.src_port);
			ip_flow.key.dst_port = ntohs(params->ipv6.dst_port);
		}

		ip_flow.key.src_ip.family = bdmf_ip_family_ipv6;
		ip_flow.key.dst_ip.family = bdmf_ip_family_ipv6;
		set_l2_ether_type(&(ip_flow.result), ethtype);

		if (params->tx.control.dscp_mark) {
			ip_flow.result.action_vec |= rdpa_fc_action_dscp_remark;
			ip_flow.result.dscp_value = params->tx.control.dscp_val;
		}

		rc = hw_accel_check_post(params, &ip_flow);
		if (rc)
			return rc;

		rc = rdpa_ip_class_flow_add(ip_class, &index, &ip_flow);
		if (params->debug) {
			flow_info_key_ipv6_dump(&ip_flow);
			flow_info_result_dump(&ip_flow);
			pr_alert("Runner return code=%d\n",rc);
		}
		if (rc == BDMF_ERR_ALREADY) {
			return handle_flow_exist_err(params, dev_in, dev,
						     &ip_flow, flow_id);
		} else if (rc < 0) {
			rdpa_ip_class_nflows_get(ip_class, &nflows);
			if (params->debug)
				pr_alert("%s: Error %d: Active flows %lld\n",
					 __FUNCTION__, rc,
					 nflows);
			return rc;
		}
		*flow_id = (int)index;
		return 0;
	}
	return -1;
}

static int fap_flow_add_expected(void *ptr, int *flow_id)
{
	struct flow_params *params = (struct flow_params *) ptr;
	rdpa_ip_flow_info_t ip_flow = {};
	struct net_device *dev_in;
	bdmf_index index = 0;
	bdmf_number nflows;
	int rc = -1;

	if (!ip_class_created)
		return -1;

	dev_in = __dev_get_by_index(&init_net,
				    params->rx.interface);
	if (!dev_in)
		return -1;

	ip_flow.result.port = rdpa_if_cpu;
	/*ip_flow.result.dqm_queue = dqnet_rfap.get_expected_queue(dev_in);*/
	ip_flow.result.action = rdpa_forward_action_host;
	ip_flow.result.trap_reason = rdpa_cpu_rx_reason_direct_queue_0 +
		dqnet_rfap.get_expected_queue(dev_in);

	if (params->type == ft_ipv4) {
		ip_flow.key.prot = params->ipv4.ip_prot;
		ip_flow.key.src_ip.addr.ipv4 = ntohl(params->ipv4.src_ip);
		ip_flow.key.dst_ip.addr.ipv4 = ntohl(params->ipv4.dst_ip);
		ip_flow.key.src_port = ntohs(params->ipv4.src_port);
		ip_flow.key.dst_port = ntohs(params->ipv4.dst_port);
		ip_flow.key.src_ip.family = bdmf_ip_family_ipv4;
		ip_flow.key.dst_ip.family = bdmf_ip_family_ipv4;

		rc = rdpa_ip_class_flow_add(ip_class, &index, &ip_flow);
		if (params->debug) {
			flow_info_key_ipv4_dump(&ip_flow);
			flow_info_result_short_dump(&ip_flow);
			pr_alert("Runner return code=%d\n",rc);
		}
		if (rc == BDMF_ERR_ALREADY) {
			return handle_flow_exist_err(params, dev_in, NULL,
						     &ip_flow, flow_id);
		} else if (rc < 0) {
			rdpa_ip_class_nflows_get(ip_class, &nflows);
			if (params->debug)
				pr_alert("%s: Error %d: Active flows %lld\n",
					 __FUNCTION__, rc,
					 nflows);
			return rc;
		}
		*flow_id = (int)index;
		return 0;
	} else if (params->type == ft_ipv6) {
		ip_flow.key.prot = params->ipv6.ip_prot;
		memcpy(&ip_flow.key.src_ip.addr.ipv6, params->ipv6.src_ip,
		       sizeof(bdmf_ipv6_t));
		memcpy(&ip_flow.key.dst_ip.addr.ipv6, params->ipv6.dst_ip,
		       sizeof(bdmf_ipv6_t));
		ip_flow.key.src_port = ntohs(params->ipv6.src_port);
		ip_flow.key.dst_port = ntohs(params->ipv6.dst_port);
		ip_flow.key.src_ip.family = bdmf_ip_family_ipv6;
		ip_flow.key.dst_ip.family = bdmf_ip_family_ipv6;

		rc = rdpa_ip_class_flow_add(ip_class, &index, &ip_flow);
		if (params->debug) {
			flow_info_key_ipv6_dump(&ip_flow);
			flow_info_result_short_dump(&ip_flow);
			pr_alert("Runner return code=%d\n",rc);
		}
		if (rc == BDMF_ERR_ALREADY) {
			return handle_flow_exist_err(params, dev_in, NULL,
						     &ip_flow, flow_id);
		} else if (rc < 0) {
			rdpa_ip_class_nflows_get(ip_class, &nflows);
			if (params->debug)
				pr_alert("%s: Error %d: Active flows %lld\n",
					 __FUNCTION__, rc,
					 nflows);
			return rc;
		}
		*flow_id = (int)index;
		return 0;
	}
	return -1;
}

static int fap_flow_add(void *ptr, int *flow_id)
{
	struct flow_params *params;
	if (!enable_flag)
		return 0;
	params = (struct flow_params *)ptr;
	if (params->type == ft_multicast)
		return fap_flow_iptv_add(params, flow_id);
	else
		return fap_flow_ip_add(params, flow_id);
}

static int fap_flow_remove(int flow_id)
{
	int rc = -1;
	if (!ip_class_created)
		return -1;

	rc = rdpa_ip_class_flow_delete(ip_class, flow_id);
	if (rc < 0) {
		net_dbg_ratelimited("%s: Flow delete Error %d: flow id %d\n",
				    __FUNCTION__, rc, flow_id);
	}
	return rc;
}

static int fap_flow_remove_mcast(int flow_id, __be32 *group_id, int oindex)
{
	struct net_device *dev;
	struct dqnet_netdev *ndev;
	rdpa_channel_req_key_t request_key;
	bdmf_index wlan_mcast_idx = RDPA_WLAN_MCAST_FWD_TABLE_INDEX_INVALID;
	rdpa_iptv_channel_t channel;
	int rc = -1;

	if (!iptv_created)
		return -1;

	dev = __dev_get_by_index(&init_net, oindex);
	if (!dev)
		return -1;
	ndev = netdev_priv(dev);
	if (!ndev)
		return -1;

	request_key.port = ndev->if_id;
	request_key.channel_index = flow_id;

	if ((ndev->if_id >= rdpa_if_ssid0) &&
	    (ndev->if_id <= rdpa_if_ssid47)) {

		if (!wlan_mcast)
			return -1;

		rc = rdpa_iptv_channel_get(iptv, flow_id, &channel);
		if (!rc)
			wlan_mcast_idx = channel.wlan_mcast_index;
		if (wlan_mcast_idx != RDPA_WLAN_MCAST_FWD_TABLE_INDEX_INVALID)
			wlan_mcast_delete(request_key.port, &wlan_mcast_idx);
		if (wlan_mcast_idx == RDPA_WLAN_MCAST_FWD_TABLE_INDEX_INVALID)
			request_key.port = rdpa_if_wlan0;
	}
	rc = rdpa_iptv_channel_request_delete(iptv, &request_key);
	pr_debug(
	   "rdpa_iptv_channel_request_delete: port %d wlan_mcast_idx %ld rc %d\n",
	   request_key.port, wlan_mcast_idx, rc);
	return rc;
}

static int fap_flow_get_counter(int flow_id, u32 *packets, u32 *bytes,
				bool reset)
{
	int rc;
	rdpa_stat_t flow_stat = {};

	if (!enable_flag) {
		*packets = 0;
		*bytes = 0;
		return 0;
	}

	if (!ip_class_created)
		return -1;

	rc = rdpa_ip_class_flow_stat_get(ip_class, flow_id, &flow_stat);
	if (rc == 0) {
		*packets = flow_stat.packets;
		*bytes = flow_stat.bytes;
	}
	return rc;
}

static int fap_flow_get_mcast_counter(int flow_id, u32 *packets, u32 *bytes,
				bool reset)
{
	int rc = -1;
	rdpa_stat_t channel_stat = {};

	if (!enable_flag) {
		*packets = 0;
		*bytes = 0;
		return 0;
	}

	if (!iptv_created)
		return -1;

	rc = rdpa_iptv_channel_pm_stats_get(iptv, flow_id, &channel_stat);
	if (rc == 0) {
		*packets = channel_stat.packets;
		*bytes = channel_stat.bytes;
	}

	return rc;
}

static int fap_dump(struct seq_file *s)
{
	bdmf_number nflows;
	int i;

	rdpa_ip_class_nflows_get(ip_class, &nflows);
	pr_seq(s, "--- RFAP debug dump ----\n");
	pr_seq(s, "enable_flag            : %d\n", enable_flag);
	pr_seq(s, "obj ip_class|iptv|wlan : %d|%d|%d\n",
		ip_class_created, iptv_created, wlan_mcast_created);
	pr_seq(s, "gre_tunnel_mask        : 0x%x\n", gre_tunnel_mask);
	pr_seq(s, "Active flows           : %llu\n", nflows);
	pr_seq(s, "Flows Exist Err         : %u\n",
		atomic_read(&flow_exist_err_cnt));
	pr_seq(s, "Flows Expected Exist Err: %u\n",
		atomic_read(&flow_expected_exist_err_cnt));
	pr_seq(s, "Flow Ring Zero Err      : %u\n",
		atomic_read(&flowring_err_cnt1));
	pr_seq(s, "Flow Ring Uninit Err    : %u\n",
		atomic_read(&flowring_err_cnt2));
	pr_seq(s, "Netdev in down Err      : %u\n",
		atomic_read(&netdev_in_down_cnt));
	pr_seq(s, "Unconfig GRE defer      : %u\n",
		atomic_read(&unconfig_gre_defer_cnt));
	for (i = 0; i < mf_max; i++)
		pr_seq(s, "mso_feat_err[%d]         : %u\n", i, atomic_read(&mso_feat_err[i]));
	dqnet_rfap.dump(s);
	return 0;
}

#ifdef CONFIG_BCM_DHD_RUNNER
static int fap_flow_get_dhdol(struct sk_buff *skb,
			      const struct net_device *dev,
			      int *wifi_flowring,
			      u8 *wifi_pri)
{
	struct dqnet_netdev *ndev;
	int radio, flowring = -1, ssid;
	unsigned char pri = 0;
	int status = -1;
	ndev = netdev_priv(dev);
	ssid = (ndev->if_id - rdpa_if_ssid0) % 16;
	radio = (ndev->if_id - rdpa_if_ssid0) / 16;
	pri = skb->priority & 0x7;
	if(ndev->dhdol_get_flow)
		status = ndev->dhdol_get_flow(ndev->dhd_cntx, (u8 *) skb,
				      skb->len, pri, ssid, &radio, &flowring);
	if (!status) {
		*wifi_pri = (flowring & WL_METADATA_PRIORITY_MASK) >> WL_METADATA_PRIORITY_SHIFT;
		*wifi_flowring = flowring & WL_METADATA_FLOWRING_MASK;
		pr_debug("wifi: flowring %d pri %d\n", *wifi_flowring,
			 *wifi_pri );
		BUG_ON(*wifi_flowring  == 0);
	}

	return status;
}
#endif

static int fap_flow_max(void)
{
	return 65536;
}

static int fap_config_dslite(void *params, int oindex)
{
	int rc = 0;
#if defined(CONFIG_BCM_MSO_FEAT_MODULE)
	bdmf_index index = 0;
	rdpa_ip_flow_info_t ip_flow = {};
	get_mso_feat_fptr get_mso_feat;
	struct mso_feat *feat;

	get_mso_feat = get_mso_feat_symbol();
	if (!get_mso_feat) {
		atomic_inc(&mso_feat_err[0]);
		net_dbg_ratelimited("FLOWMGR: MSO Feat Module not loaded\n");
		goto done;
	}

	feat = get_mso_feat(mf_dslite);
	if (!feat) {
		atomic_inc(&mso_feat_err[mf_dslite]);
		net_dbg_ratelimited("FLOWMGR: MSO Feat DSLITE APIs not supported\n");
		goto done;
	}

	if (!feat->dslite || !feat->dslite->ops3) {
		atomic_inc(&mso_feat_err[mf_dslite]);
		net_dbg_ratelimited("FLOWMGR: MSO DSLITE Module ops3 not defined\n");
		goto done;
	}

	rc = feat->dslite->ops3(params, sizeof(struct flow_dslite_tunnel_params),
							&ip_flow, sizeof(rdpa_ip_flow_info_t), 0);
	if (rc) {
		atomic_inc(&mso_feat_err[mf_dslite]);
		net_dbg_ratelimited("FLOWMGR: MSO DSLITE Module ops3 failed %d\n", rc);
		goto done;
	}

	rc = rdpa_ip_class_flow_add(ip_class, &index, &ip_flow);
	if (rc < 0) {
		pr_debug("%s: Error %d: Adding first dslite encap flow\n",
			 __FUNCTION__, rc);
	} else {
		rc = feat->dslite->ops3(params, sizeof(struct flow_dslite_tunnel_params),
								&ip_flow, sizeof(rdpa_ip_flow_info_t), index);
		if (rc) {
			atomic_inc(&mso_feat_err[mf_dslite]);
			net_dbg_ratelimited("FLOWMGR: MSO DSLITE Module ops3 failed %d\n", rc);
		}
	}
done:
#endif
	return rc;
}

static int fap_unconfig_dslite(int id)
{
	return fap_flow_remove(id);
}

static struct fap_ops fapops = {
	.type = RFAP,
	.name = "HW FAP - Runner",
	.features = (FLOW_F_L4_ALL | FLOW_F_GRETAP4WAN | FLOW_F_GRETAP4LAN | FLOW_F_GRETAP6LAN |
		     FLOW_F_GRETAP4ETHWAN | FLOW_F_GRETAP6WAN | FLOW_F_MAPT |
		     FLOW_F_DSLITE | FLOW_F_MCAST | FLOW_F_DSLITE_GRE| FLOW_F_MAPT_GRE),
	.dump = fap_dump,
	.enable = fap_enable,
	.config = fap_config,
	.config_dslite = fap_config_dslite,
	.unconfig_dslite = fap_unconfig_dslite,
	.config_gre = fap_config_gre,
	.config_mapt = fap_config_mapt,
	.unconfig_gre = fap_unconfig_gre,
	.flow_add = fap_flow_add,
	.flow_add_expected = fap_flow_add_expected,
	.flow_remove = fap_flow_remove,
	.flow_remove_mcast = fap_flow_remove_mcast,
	.flow_get_counter = fap_flow_get_counter,
	.flow_get_mcast_counter = fap_flow_get_mcast_counter,
#ifdef CONFIG_BCM_DHD_RUNNER
	.flow_get_dhdol = fap_flow_get_dhdol,
#endif
	.flow_max = fap_flow_max,
};

struct fap_ops *rfap(void)
{
	return &fapops;
}

static int __init flowmgr_rfap_init(void)
{
	flowmgr_register_fap(rfap());
	/* Disable Multicast to Unicast */
	flowmgr_mc2uc_enable(0);
	return 0;
}

static void __exit flowmgr_rfap_exit(void)
{
	flowmgr_unregister_fap(rfap());
}

module_init(flowmgr_rfap_init);
module_exit(flowmgr_rfap_exit);
MODULE_LICENSE("GPL");
MODULE_ALIAS("flowmgr_rfap");
