 /****************************************************************************
 *
 * Copyright (c) 2015-2018 Broadcom. All rights reserved
 * The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************
 * Author: Tim Ross <tross@broadcom.com>
 *****************************************************************************/
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/if_link.h>
#include <linux/netdevice.h>
#include <linux/jhash.h>
#include <linux/random.h>
#include <net/netfilter/nf_conntrack_core.h>
#include <net/netfilter/nf_conntrack_offload.h>
#include "dqnet_priv.h"
#include "dqnet_fap.h"
#include "dqnet_qmsg.h"
#include "dqnet_brcmtag.h"
#include "fpm.h"
#include <rdpa_api.h>
#include <rdpa_cpu_helper.h>
#include <rdp_init.h>
#include <autogen/rdpa_ag_iptv.h>
#include "ethsw.h"
#include "ethsw_core.h"
#include "ethsw_priv.h"
#define EGRESS_Q_THRESHOLD_LAN2LAN	0
/*
   Setting queue threshold to 1536 to avoid data loss in bursty downstream
   traffic.
   Total pd's available in runner for downstream traffic is 2048.
*/
#define EGRESS_Q_THRESHOLD_IMP2LAN      1536
#define EGRESS_Q_THRESHOLD_OTHER        128
#define EGRESS_Q_COUNT_LAN_PORT         8
#define EGRESS_Q_COUNT_NON_LAN_PORT     1
#define RUNNER_FIRST_DQM_QUEUE          48

#define IMP_MAX_PORTS  3

#define ALTWAN_DISABLED 0
#define ALTWAN_ENABLED  1
#define ALTWAN_RF rdpa_emac_none
#define ALTWAN_SWITCHID_8 0
#define ALTWAN_SWITCHID_5 1
#define ALTWAN_SWITCHID_7 2

#define ALTWAN_DEF_IMP ALTWAN_SWITCHID_5;

struct imp_port {
	u8  runner_id; /* Runner port ID */
	u8  switch_id; /* SF2 port ID */
	u8  enable;    /* 0 or 1 */
	u8  if_mask;
	u8  reserve_if_mask;
	u64 usage;
};

static struct imp_port imp_lag[IMP_MAX_PORTS] = {
	{.runner_id = 0, .switch_id = 8},
	{.runner_id = 1, .switch_id = 5},
	{.runner_id = 2, .switch_id = 7},
};
static u8 imp_port_rr_idx = IMP_MAX_PORTS;

static int8_t imp_lag_static_config[DQNET_IMP_LAG_ACT_MAX+1][ETHSW_PORT_MAX] =
{
	[DQNET_IMP_LAG_ACT_DEFAULT]  = { 0, 0, 0, 0, 0, -1, -1, 1, -1 },
	[DQNET_IMP_LAG_ACT_OPT1]     = { 0, 0, 0, 0, 1, -1, -1, 1, -1 },
	[DQNET_IMP_LAG_ACT_OPT2_S0]  = { 1, 0, 0, 0, 1, -1, -1, 1, -1 },
	[DQNET_IMP_LAG_ACT_OPT2_S1]  = { 0, 1, 0, 0, 1, -1, -1, 1, -1 },
	[DQNET_IMP_LAG_ACT_OPT2_S2]  = { 0, 0, 1, 0, 1, -1, -1, 1, -1 },
	[DQNET_IMP_LAG_ACT_OPT2_S3]  = { 0, 0, 0, 1, 1, -1, -1, 1, -1 },
	[DQNET_IMP_LAG_ACT_OPT3]     = { 1, 0, 0, 0, 1, -1, -1, 1, -1 },

	/* Static configs for non-MoCA platforms below */
	[DQNET_IMP_LAG_ACT_DEFAULT_NOMOCA]  = { 0, 0, 0, 0, -1, -1, -1, 1, -1 },
	[DQNET_IMP_LAG_ACT_OPT1_NM]         = { 1, 0, 0, 0, -1, -1, -1, 1, -1 },
	[DQNET_IMP_LAG_ACT_OPT2_S0_NM]      = { 1, 1, 0, 0, -1, -1, -1, 1, -1 },
	[DQNET_IMP_LAG_ACT_OPT2_S1_NM]      = { 1, 1, 0, 0, -1, -1, -1, 1, -1 },
	[DQNET_IMP_LAG_ACT_OPT2_S2_NM]      = { 1, 0, 1, 0, -1, -1, -1, 1, -1 },
	[DQNET_IMP_LAG_ACT_OPT2_S3_NM]      = { 1, 0, 0, 1, -1, -1, -1, 1, -1 },
	[DQNET_IMP_LAG_ACT_OPT3_NM]         = { 1, 1, 0, 0, -1, -1, -1, 1, -1 },
};

static void set_imp_lag_port_mask(int port, int if_id, bool reserve)
{
	int i;

	pr_debug("imp:%d rsvp %d: -->\n", port, reserve);
	for (i = 0; i < IMP_MAX_PORTS; i++) {
		imp_lag[i].if_mask &= ~(1 << if_id);
		imp_lag[i].reserve_if_mask &= ~(1 << if_id);
	}
	imp_lag[port].if_mask |= (1 << if_id);
	if (reserve)
		imp_lag[port].reserve_if_mask = (1 << if_id);
}

static int dqnet_rfap_get_imp_lag_port(struct net_device *dev)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	int port = ndev ? ndev->swport_imp : 0;
	if (port < 0)
		port = 0;
	if (port < IMP_MAX_PORTS)
		imp_lag[port].usage++;
	return port;
}

static int dqnet_rfap_set_imp_lag_port(struct net_device *dev,
				       int port, int assign)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	int i, p;

	if (assign == DQNET_SET_IMP_RESET) {
		imp_port_rr_idx = IMP_MAX_PORTS;
		for (i = 0; i < IMP_MAX_PORTS; i++) {
			imp_lag[i].if_mask = 0;
			imp_lag[i].reserve_if_mask = 0;
		}
		return 0;
	}

	if (!ndev || ndev->if_id > rdpa_if_lan5)
		return -1;

	p = ndev->if_id - rdpa_if_lan0;
	if (assign == DQNET_SET_IMP_AUTO) {
		/* Auto assignment */
		for (i = 0; i < IMP_MAX_PORTS; i++) {
			imp_port_rr_idx++;
			if (imp_port_rr_idx >= IMP_MAX_PORTS)
				imp_port_rr_idx = 0;
			/* Skip reserved ports */
			if (imp_lag[imp_port_rr_idx].enable &&
			    !imp_lag[imp_port_rr_idx].reserve_if_mask) {
				ndev->swport_imp = imp_port_rr_idx;
				set_imp_lag_port_mask(imp_port_rr_idx, p, false);
				return 0;
			}
		}
	} else if (assign == DQNET_SET_IMP_STATIC) {
		/* Static assignment */
		if ((port < 0) || (port >= IMP_MAX_PORTS))
			return -1;
		if (imp_lag[port].enable &&
		    !imp_lag[port].reserve_if_mask) {
			ndev->swport_imp = port;
			set_imp_lag_port_mask(port, p, false);
			return 0;
		}
	} else if (assign == DQNET_SET_IMP_RESERVE) {
		/* Static reserve assignment */
		if ((port < 0) || (port >= IMP_MAX_PORTS))
			return -1;
		if (imp_lag[port].enable) {
			ndev->swport_imp = port;
			set_imp_lag_port_mask(port, p, true);
			return 0;
		}
	}

	return -1;
}

static int dqnet_rfap_reset_imp_lag_port(struct dqnet_netdev *ndevs)
{
	int status = 0;
	struct dqnet_netdev *ndev = NULL;

	rcu_read_lock();
	list_for_each_entry_rcu(ndev, &ndevs->list, list) {
		if (ndev->link_type != DQNET_LINK_TYPE_SWITCH)
			continue;

		status = dqnet_rfap_set_imp_lag_port(ndev->dev, 0,
				DQNET_SET_IMP_RESET);
	}
	rcu_read_unlock();

	return status;
}

static int dqnet_rfap_set_static_imp_lag_config(struct dqnet_netdev *ndevs,
						int action, int poi)
{
	int i, config, status = 0;
	struct dqnet_netdev *ndev = NULL;

	if (poi != -1) {
		config = action + poi;
	} else {
		config = action;
	}

	pr_debug("Setting static IMP LAG config %d\n", config);

	rcu_read_lock();
	list_for_each_entry_rcu(ndev, &ndevs->list, list) {
		if (ndev->link_type != DQNET_LINK_TYPE_SWITCH)
			continue;

		for (i = 0; i < ETHSW_PORT_MAX; i++) {
			int imp_port = imp_lag_static_config[config][i];

			if (imp_port == -1) {
				continue;
			}

			if (ndev->if_sub_id == i) {
				status = dqnet_rfap_set_imp_lag_port(ndev->dev,
					   imp_port, DQNET_SET_IMP_STATIC);

				if (status) {
					pr_err("Fail to map swport %d to impport %d\n",
						i, imp_port);
				}
			}

		}
	}
	rcu_read_unlock();

	return status;
}

static int dqnet_rfap_tc_init(void)
{
	int status = 0;
	int tc;
	/* Following map should match table in ethsw_flow_control.c */
	int tc_map[] = { 1, 1, 3 ,3, 5 ,5, 7, 7 };
	int tc_map_size = sizeof(tc_map) / sizeof(int);
	bdmf_object_handle tc_to_q_obj = NULL;
	BDMF_MATTR(qos_mattrs, rdpa_tc_to_queue_drv());

	pr_debug("%s: -->\n", __func__);

	status = bdmf_new_and_set(rdpa_tc_to_queue_drv(), NULL, qos_mattrs, &tc_to_q_obj);
	if (status) {
		pr_err("%s: bdmf_new_and_set tc_to_queue obj failed\n",
		       __func__);
		goto done;
	}

	for(tc = 0; tc < tc_map_size; tc ++) {
		status = rdpa_tc_to_queue_tc_map_set(tc_to_q_obj, tc,
						     tc_map[tc]);

		if (status) {
			pr_err("Failed rpa_tc_to_queue_tc_map, tc %d\n",
			       tc);
			goto done;
		}
	}

done:
	pr_debug("%s: <--\n", __func__);
	return status;
}

static int dqnet_rfap_tc_link_port(rdpa_if index)
{
	int status = 0;
	BDMF_MATTR(rdpa_port_attrs, rdpa_port_drv());
	bdmf_object_handle rdpa_port_obj = NULL;
	bdmf_object_handle tc_to_q_obj = NULL;
	rdpa_tc_to_queue_key_t key = {0, rdpa_dir_ds};

	pr_debug("%s: -->\n", __func__);

	status = rdpa_tc_to_queue_get(&key, &tc_to_q_obj);
	if (status) {
		pr_err("Failed to get tc_to_queue table for port %d\n", index);
		goto done;
	}

	status = rdpa_port_get(index, &rdpa_port_obj);
	if (status) {
		rdpa_port_index_set(rdpa_port_attrs, index);
		status = bdmf_new_and_set(rdpa_port_drv(), NULL,
					  rdpa_port_attrs,
					  &rdpa_port_obj);
		if (status) {
			pr_err("Failed to create port_obj for port %d\n",
			       index);
			goto done;
		}
	}
	status = bdmf_link(rdpa_port_obj, tc_to_q_obj, NULL);
	if (status) {
		pr_err("Failed to link tc_to_queue table to port %d\n", index);
		goto done;
	}
done:
	if (tc_to_q_obj)
		bdmf_put(tc_to_q_obj);
	pr_debug("%s: <--\n", __func__);
	return status;
}

int dqnet_wlan_mcast_init(void)
{
	bdmf_object_handle wlan_mcast_class;
	int ret;

	BDMF_MATTR(wlan_mcast_attrs, rdpa_wlan_mcast_drv());

	ret = rdpa_wlan_mcast_get(&wlan_mcast_class);
	if (ret) {
		ret = bdmf_new_and_set(rdpa_wlan_mcast_drv(), NULL,
                                      wlan_mcast_attrs, &wlan_mcast_class);
		if (ret) {
			pr_err("rdpa wlan_mcast_class object does not exist ");
			pr_cont("and can't be created.\n");
			return ret;
		}
	}

       return 0;
}

static int rfap_init_wlan_queue(void)
{
	bdmf_object_handle rdpa_cpu_obj;
	int rc;

	rc = rdpa_cpu_get(rdpa_cpu_wlan0, &rdpa_cpu_obj);
	if (rc < 0) {

		BDMF_MATTR(cpu_attrs, rdpa_cpu_drv());

		rdpa_cpu_index_set(cpu_attrs, rdpa_cpu_wlan0);
		rc = bdmf_new_and_set(rdpa_cpu_drv(), NULL, cpu_attrs,
				      &rdpa_cpu_obj);
		if (rc < 0) {
			pr_err("%s: bdmf_new_and_set rdpa_cpu obj failed\n",
			       __func__);
			return rc;
		}
	}

	return rc;
}

int altwan_enable = ALTWAN_DISABLED;
module_param_named(altwan_enable, altwan_enable, int, 0644);
MODULE_PARM_DESC(altwan, "Enable Altwan mode (altwan_enable=0|1");

int altwan_imp = ALTWAN_RF;
module_param_named(altwan_imp, altwan_imp, int, 0644);
MODULE_PARM_DESC(altwan, "Set Altwan IMP (altwan_imp=0|1|2)");

int altwan_support = ALTWAN_DISABLED;
module_param_named(altwan_support, altwan_support, int, 0644);
MODULE_PARM_DESC(altwan_support, "Altwan mode supported (altwan_support=0|1)");

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 14, 0)
int read_altwan_config(void)
{
	return 0;
}
#else
struct file *dqnet_file_open(const char *path, int flags, int rights)
{
	struct file *filp = NULL;
	mm_segment_t oldfs;
	int err = 0;

	oldfs = get_fs();
	set_fs(KERNEL_DS);
	filp = filp_open(path, flags, rights);
	set_fs(oldfs);
	if (IS_ERR(filp)) {
		err = PTR_ERR(filp);
		return NULL;
	}
	return filp;
}

void dqnet_file_close(struct file *file)
{
	filp_close(file, NULL);
}
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5,4,0))
#define vfs_read kernel_read
#define vfs_write kernel_write
#endif

int dqnet_file_read(struct file *file, unsigned long long offset,
		      unsigned char *data, unsigned int size)
{
	mm_segment_t oldfs;
	int ret;

	oldfs = get_fs();
	set_fs(KERNEL_DS);

	ret = vfs_read(file, data, size, &offset);

	set_fs(oldfs);
	return ret;
}

/* Read kernel boot args which can override default settings */
int read_altwan_config(void)
{
	struct file *f;
	unsigned char data[256];
	int ret;

	f = dqnet_file_open("/proc/cmdline",
			      O_RDONLY, 0);
	if (!f)
		return 0;

	ret = dqnet_file_read(f, 0, data, 256);
	dqnet_file_close(f);
	if (ret > 0) {
		if (strstr(data, "altwan=0")) {
			altwan_imp = ALTWAN_SWITCHID_8;
		} else if (strstr(data, "altwan=1")) {
			altwan_imp = ALTWAN_SWITCHID_5;
		} else if (strstr(data, "altwan=2")) {
			altwan_imp = ALTWAN_SWITCHID_7;
		}
	}

	return 0;
}
#endif

int dqnet_rfap_init_system(void)
{
	int i, j, status = 0;
	BDMF_MATTR(rdpa_system_attrs, rdpa_system_drv());
	BDMF_MATTR(rdpa_filter_attrs, rdpa_filter_drv());
	bdmf_object_handle rdpa_system_obj = NULL;
	bdmf_object_handle rdpa_filter_obj = NULL;
	rdpa_system_init_cfg_t sys_init_cfg = {};
	rdpa_system_cfg_t system_cfg = {};
	bdmf_object_handle ip_class_obj;
	bdmf_object_handle iptv_obj;

	pr_debug(" -->\n");
	read_altwan_config();
	for (i = 0; i < IMP_MAX_PORTS; ++i) {
		if (rdp_lag_port_enabled(i)) {
			sys_init_cfg.enabled_emac |= (1 << (rdpa_emac0 + i));
		}
	}

	pr_debug("AltWan config: support=%d enable=%d imp=%d\n",
		 altwan_support, altwan_enable, altwan_imp);

	sys_init_cfg.runner_ext_sw_cfg.type = rdpa_brcm_hdr_opcode_0;
	sys_init_cfg.runner_ext_sw_cfg.emac_id = rdpa_emac0;
	sys_init_cfg.runner_ext_sw_cfg.enabled = 1;
	sys_init_cfg.switching_mode = rdpa_switching_none;
	sys_init_cfg.ip_class_method = rdpa_method_fc;
	sys_init_cfg.alt_wan_feat_enabled = altwan_support;
	sys_init_cfg.alt_wan_port = rdpa_emac0 + altwan_imp;

	status = rdpa_system_init_cfg_set(rdpa_system_attrs, &sys_init_cfg);
	if (status) {
		pr_err("%s: Call to rdpa_system_init_cfg_set failed.\n",
		       __func__);
		status = -EIO;
		goto done;
	}
	status = bdmf_new_and_set(rdpa_system_drv(), NULL, rdpa_system_attrs,
					&rdpa_system_obj);
	if (status) {
		pr_err("%s: Call to create system object failed. Error code=%d\n",
		       __func__, status);
		status = -EIO;
		goto done;
	}
	if (altwan_enable) {
		pr_info("init_system: Configure Runner in Altwan mode [%d]\n",
			altwan_imp);
		status = rdpa_system_cfg_get(rdpa_system_obj, &system_cfg);
		if (status) {
			pr_err("%s: Call to rdpa_system_cfg_get failed. Error code=%d\n",
				__func__, status);
			status = -EIO;
			goto done;
		}
		system_cfg.alt_wan_enabled = 1;
		status = rdpa_system_cfg_set(rdpa_system_attrs, &system_cfg);
		if (status) {
			pr_err("%s: Call to rdpa_system_cfg_set failed. Error code=%d\n",
				__func__, status);
			status = -EIO;
			goto done;
		}
	} else {
		pr_info("init_system: Configure Runner in DOCSIS mode\n");
	}

	status = bdmf_new_and_set(rdpa_filter_drv(), NULL, rdpa_filter_attrs,
				  &rdpa_filter_obj);
	if (status) {
		pr_err("%s: Call to create filter object failed.\n",
		       __func__);
		status = -EIO;
		goto done;
	}
	status = rdpa_filter_etype_udef_set(rdpa_filter_obj, 0,
					    ETH_P_IEEE1905);
	if (status) {
		pr_err("%s: Call to udef set for filter object failed.\n",
		       __func__);
		status = -EIO;
		goto done;
	}
	status = rdpa_filter_etype_udef_set(rdpa_filter_obj, 1,
					    ETH_P_BRCM);
	if (status) {
		pr_err("%s: Call to udef set for filter object failed.\n",
		       __func__);
		status = -EIO;
		goto done;
	}
	status = rdpa_filter_etype_udef_set(rdpa_filter_obj, 2,
					    ETH_P_802_1X);
	if (status) {
		pr_err("%s: Call to udef set for filter object failed.\n",
		       __func__);
		status = -EIO;
		goto done;
	}
	status = rdpa_filter_etype_udef_set(rdpa_filter_obj, 3,
					    ETH_P_802_1X_PREAUTH);
	if (status) {
		pr_err("%s: Call to udef set for filter object failed.\n",
		       __func__);
		status = -EIO;
		goto done;
	}
	if (rdpa_ip_class_drv())
	{
		BDMF_MATTR(ip_class_attrs, rdpa_ip_class_drv());
		status = bdmf_new_and_set(rdpa_ip_class_drv(), NULL,
					  ip_class_attrs, &ip_class_obj);
		if (status) {
			pr_err("%s: Call to create ip_class object failed.\n",
			       __func__);
			status = -EIO;
			goto done;
		}
	}
	if (rdpa_iptv_drv())
	{
		BDMF_MATTR(iptv_attrs, rdpa_iptv_drv());
		rdpa_iptv_lookup_method_set(iptv_attrs,
			iptv_lookup_method_group_ip_src_ip);
		status = bdmf_new_and_set(rdpa_iptv_drv(), NULL,
					  iptv_attrs, &iptv_obj);
		if (status) {
			pr_err("%s: Call to create iptv object failed.\n",
			       __func__);
			status = -EIO;
			goto done;
		}
	}

	status = dqnet_wlan_mcast_init();
	if (status) {
		pr_err("%s: Call to create wlan_mcast object failed.\n",
		       __func__);
		status = -EIO;
		goto done;
	}

	status = dqnet_rfap_tc_init();
	if (status) {
		pr_err("%s: Call to create tc object failed.\n",
		       __func__);
		status = -EIO;
		goto done;
	}

	status = rfap_init_wlan_queue();
	if (status) {
		pr_err("%s: Call to create wlan queue failed.\n",
		       __func__);
		status = -EIO;
		goto done;
	}

	for (i = 0, j = 0; i < IMP_MAX_PORTS; i++) {
		if (rdp_lag_port_enabled(i)) {
			imp_lag[i].enable = 1;
		}
	}

done:
	pr_debug(" <--\n");
	return status;
}

/*
 * Enable or disable diversion of important traffic to an alternate exception
 * queue. Enabling this feature helps to ensure that critical traffic reaches
 * the RG host from Runner despite any traffic congestion.
 */
static int dqnet_rfap_ctl_traffic_to_cpu(int queue_id)
{
	bdmf_object_handle rdpa_cpu_obj;
	rdpa_cpu_reason_cfg_t reason_cfg = {};
	rdpa_cpu_reason_index_t reason_idx = {};
	struct dqnet_cpu_rx_demux_entry {
		rdpa_cpu_reason  reason;
		rdpa_traffic_dir dir;
	} demux_table[] = {
		{rdpa_cpu_rx_reason_dhcp,      rdpa_dir_us},
		{rdpa_cpu_rx_reason_dhcp,      rdpa_dir_ds},
		{rdpa_cpu_rx_reason_etype_arp, rdpa_dir_us},
		{rdpa_cpu_rx_reason_etype_arp, rdpa_dir_ds},
		{rdpa_cpu_rx_reason_icmpv6,    rdpa_dir_us}, /* covers NDP */
		{rdpa_cpu_rx_reason_icmpv6,    rdpa_dir_ds}, /* covers NDP */
		{rdpa_cpu_rx_reason_l4_icmp,   rdpa_dir_us},
		{rdpa_cpu_rx_reason_l4_icmp,   rdpa_dir_ds}
	};
	const struct dqnet_cpu_rx_demux_entry *pde;
	int rc;
	int i;

	rc = rdpa_cpu_get(rdpa_cpu_host, &rdpa_cpu_obj);
	if (rc < 0) {
		pr_err(" rdpa_cpu_get error rc=%d\n", rc);
		return rc;
	}

	reason_cfg.meter = BDMF_INDEX_UNASSIGNED;
	reason_cfg.meter_ports = 0;
	reason_cfg.queue = queue_id;

	for (i = sizeof(demux_table) / sizeof(demux_table[0]); i--; ) {
		pde = demux_table + i;
		reason_idx.reason = pde->reason;
		reason_idx.dir    = pde->dir;
		rc = rdpa_cpu_reason_cfg_set(rdpa_cpu_obj, &reason_idx,
					     &reason_cfg);
		if (rc < 0) {
			pr_err("rdpa_cpu_reason_cfg_set error rc=%d "
			       "demux_table(%d)=(%d, %d, %d)\n", rc, i,
			       queue_id, (int)pde->reason, (int)pde->dir);
			break;
		}
	}

	return rc;
}

static int dqnet_rfap_tcp_traffic_to_cpu(int queue_id)
{
	bdmf_object_handle rdpa_cpu_obj;
	rdpa_cpu_reason_cfg_t reason_cfg = {};
	rdpa_cpu_reason_index_t reason_idx = {};
	int rc;

	rc = rdpa_cpu_get(rdpa_cpu_host, &rdpa_cpu_obj);
	if (rc < 0) {
		pr_err(" rdpa_cpu_get error rc=%d\n", rc);
		return rc;
	}

	reason_cfg.meter = BDMF_INDEX_UNASSIGNED;
	reason_cfg.meter_ports = 0;
	reason_cfg.queue = queue_id;
	reason_idx.reason = rdpa_cpu_rx_reason_tcp_flags;
	reason_idx.dir    = rdpa_dir_ds;

	rc = rdpa_cpu_reason_cfg_set(rdpa_cpu_obj, &reason_idx,
				     &reason_cfg);
	if (rc < 0) {
		pr_err("rdpa_cpu_reason_cfg_set tcp ds error rc=%d ", rc);
	}

	reason_idx.dir    = rdpa_dir_us;
	rc = rdpa_cpu_reason_cfg_set(rdpa_cpu_obj, &reason_idx,
				     &reason_cfg);
	if (rc < 0) {
		pr_err("rdpa_cpu_reason_cfg_set tcp us error rc=%d ", rc);
	}

	return rc;
}

static int dqnet_rfap_direct_queue_to_cpu(int queue_id)
{
	bdmf_object_handle rdpa_cpu_obj;
	rdpa_cpu_reason_cfg_t reason_cfg = {};
	rdpa_cpu_reason_index_t reason_idx = {};
	int rc;

	rc = rdpa_cpu_get(rdpa_cpu_host, &rdpa_cpu_obj);
	if (rc < 0) {
		pr_err(" rdpa_cpu_get error rc=%d\n", rc);
		return rc;
	}

	reason_cfg.meter = BDMF_INDEX_UNASSIGNED;
	reason_cfg.meter_ports = 0;
	reason_cfg.queue = queue_id;
	reason_idx.reason = rdpa_cpu_rx_reason_direct_queue_0 + queue_id;
	reason_idx.dir    = rdpa_dir_ds;

	rc = rdpa_cpu_reason_cfg_set(rdpa_cpu_obj, &reason_idx,
				     &reason_cfg);
	if (rc < 0) {
		pr_err("rdpa_cpu_reason_cfg_set tcp ds error rc=%d ", rc);
	}

	reason_idx.dir    = rdpa_dir_us;
	rc = rdpa_cpu_reason_cfg_set(rdpa_cpu_obj, &reason_idx,
				     &reason_cfg);
	if (rc < 0) {
		pr_err("rdpa_cpu_reason_cfg_set tcp us error rc=%d ", rc);
	}

	return rc;
}

static int dqnet_rfap_init_host_queue(struct net_device *dev, int queue_id)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	rdpa_cpu_rxq_cfg_t rxq_cfg;
	bdmf_object_handle rdpa_cpu_obj;
	bdmf_object_handle rdpa_system_obj;
	rdpa_system_cfg_t system_cfg;
	rdpa_fc_bypass fc_bypass_mask;
	bdmf_object_handle ip_class;
	int rc, q_offset;

	rc = rdpa_cpu_get(rdpa_cpu_host, &rdpa_cpu_obj);
	if (rc < 0) {
		pr_err(" rdpa_cpu_get error:  %s.\n", ndev->name);
		return rc;
	}

	rc = rdpa_cpu_rxq_cfg_get(rdpa_cpu_obj, queue_id, &rxq_cfg);
	if (rc < 0) {
		pr_err(" rdpa_cpu_rxq_cfg_get error %d:  %s.\n", rc, ndev->name);
		return rc;
	}
	if ((ndev->chan->qmsg_fmt == DQNET_QMSG_FMT_RFAP_SKB) &&
	    (rxq_cfg.buff_type != rdpa_buf_type_host)) {
		rxq_cfg.buff_type = rdpa_buf_type_host;
		rc = rdpa_cpu_rxq_cfg_set(rdpa_cpu_obj, queue_id, &rxq_cfg);
	}
	if (rc < 0) {
		pr_err(" rdpa_cpu_rxq_cfg_set error %d:  %s.\n", rc, ndev->name);
		return rc;
	}

	rc = rdpa_cpu_get(rdpa_cpu_wlan0, &rdpa_cpu_obj);
	if (rc < 0) {

		BDMF_MATTR(cpu_attrs, rdpa_cpu_drv());

		rdpa_cpu_index_set(cpu_attrs, rdpa_cpu_wlan0);
		rc = bdmf_new_and_set(rdpa_cpu_drv(), NULL, cpu_attrs,
				      &rdpa_cpu_obj);
		if (rc < 0) {
			pr_err("%s: bdmf_new_and_set rdpa_cpu obj failed\n",
			       __func__);
			return rc;
		}
	}

	q_offset = ndev->chan->rx_q_info[queue_id].q_num - RUNNER_FIRST_DQM_QUEUE;
	if (ndev->chan->rx_q_info[queue_id].q_type == DQNET_QUEUE_DS) {
		rc = rdpa_system_get(&rdpa_system_obj);
		if (rc) {
			pr_err("%s: Call to get system object failed.\n",
			       __func__);
			return rc;
		}
		rc = rdpa_system_cfg_get(rdpa_system_obj, &system_cfg);
		if (rc) {
			pr_err("%s: Call to get system cfg failed.\n",
			       __func__);
			return rc;
		}
		if (system_cfg.ds_dqm_queue != q_offset) {
			system_cfg.ds_dqm_queue = q_offset;
			rc = rdpa_system_cfg_set(rdpa_system_obj, &system_cfg);
			if (rc) {
				pr_err("%s: Call to set system cfg failed.\n",
				       __func__);
				return rc;
			}
			pr_info("WAN Exception Queue = %d\n",
				ndev->chan->rx_q_info[queue_id].q_num);
		}
	} else if (ndev->chan->rx_q_info[queue_id].q_type == DQNET_QUEUE_US) {
		rc = rdpa_system_get(&rdpa_system_obj);
		if (rc) {
			pr_err("%s: Call to get system object failed.\n",
			       __func__);
			return rc;
		}
		rc = rdpa_system_cfg_get(rdpa_system_obj, &system_cfg);
		if (rc) {
			pr_err("%s: Call to get system cfg failed.\n",
			       __func__);
			return rc;
		}
		if (system_cfg.us_dqm_queue != q_offset) {
			system_cfg.us_dqm_queue = q_offset;
			rc = rdpa_system_cfg_set(rdpa_system_obj, &system_cfg);
			if (rc) {
				pr_err("%s: Call to set system cfg failed.\n",
				       __func__);
				return rc;
			}
			pr_info("LAN Exception Queue = %d\n",
				ndev->chan->rx_q_info[queue_id].q_num);
		}
	} else if (ndev->chan->rx_q_info[queue_id].q_type == DQNET_QUEUE_CTL) {
		ip_class = NULL;
		fc_bypass_mask = 0;
		rdpa_ip_class_get(&ip_class);
		rdpa_ip_class_fc_bypass_get(ip_class, &fc_bypass_mask);
		if (!(fc_bypass_mask & RDPA_IP_CLASS_MASK_BP_ALL))
			dqnet_rfap_ctl_traffic_to_cpu(q_offset);
	} else if (ndev->chan->rx_q_info[queue_id].q_type == DQNET_QUEUE_EXP) {
		dqnet_rfap_tcp_traffic_to_cpu(q_offset);
		dqnet_rfap_direct_queue_to_cpu(q_offset);
	}

	return rc;
}

static int dqnet_rfap_init_port(struct net_device *dev)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	BDMF_MATTR(rdpa_port_attrs, rdpa_port_drv());
	bdmf_object_handle switch_port_obj = NULL;
	bdmf_object_handle port_obj = NULL;
	rdpa_port_dp_cfg_t port_cfg = {};
	BDMF_MATTR(tm_attr, rdpa_egress_tm_drv());
	bdmf_object_handle tm_obj = NULL;
	rdpa_port_tm_cfg_t port_tm_cfg = {NULL, rdpa_discard_prty_low};
	bdmf_object_handle filter_obj = NULL;
	rdpa_filter_key_t  filter_key = {};
	rdpa_filter_ctrl_t filter_entry = {1, rdpa_forward_action_host};
	rdpa_traffic_dir dir;
	rdpa_filter lan_filters[] = {RDPA_FILTER_ETYPE_UDEF_0,
		RDPA_FILTER_ETYPE_UDEF_1,
		RDPA_FILTER_ETYPE_UDEF_2,
		RDPA_FILTER_ETYPE_UDEF_3,
		RDPA_FILTER_ETYPE_ARP, RDPA_FILTER_IP_FRAG, RDPA_FILTER_HDR_ERR,
		RDPA_FILTER_BCAST, RDPA_FILTER_MCAST, RDPA_FILTER_DHCP};
	rdpa_tm_queue_cfg_t q_cfg = {};
	int q_id;
	int f_idx;
	int q_count;

	pr_debug("%s:-->\n",dev->name);

	if (ndev->chan->type == DQNET_CHAN_TYPE_POINT_TO_POINT ||
	    ndev->chan->type == DQNET_CHAN_TYPE_POINT_TO_POINT_SWAP)
		return 0;

	/* Don't attempt to create a switch port object for an external switch. */
	status = rdpa_port_get(ndev->if_id, &port_obj);
	if (status) {
		pr_debug("%d Failed to get port object for device %s.\n", status, ndev->name);
	}
	if (port_obj) {
		bdmf_put(port_obj);
		return 0;
	}

	for (q_id = 0; q_id < ndev->chan->rx_q_count; q_id++) {
		status = dqnet_rfap_init_host_queue(dev, q_id);
		if (status) {
			pr_err("Failed to init host queue %d object "
			       "for device %s.\n", q_id, ndev->name);
			goto done;
		}
	}

	port_cfg.enable = 0;
	port_cfg.sal_enable = 0;
	port_cfg.dal_enable = 0;
	port_cfg.sal_miss_action = rdpa_forward_action_none;
	port_cfg.dal_miss_action = rdpa_forward_action_none;
	if (ndev->link_type == DQNET_LINK_TYPE_SWITCH) {
		port_cfg.emac = rdpa_emac0;
		status = rdpa_port_get(rdpa_if_switch, &switch_port_obj);
		if (status) {
			rdpa_port_index_set(rdpa_port_attrs, rdpa_if_switch);
			rdpa_port_cfg_set(rdpa_port_attrs, &port_cfg);
			status = bdmf_new_and_set(rdpa_port_drv(), NULL,
						  rdpa_port_attrs,
						  &switch_port_obj);
			if (status) {
			    pr_err("Failed to create switch port ");
			    pr_cont("object for device %s.\n", ndev->name);
			    goto done;
			}
		}
		rdpa_port_index_set(rdpa_port_attrs, ndev->if_id);
		port_cfg.emac = rdpa_emac_none;
		port_cfg.physical_port = ndev->if_id - rdpa_if_lan0;
		rdpa_port_cfg_set(rdpa_port_attrs, &port_cfg);
		status = bdmf_new_and_set(rdpa_port_drv(), switch_port_obj,
					  rdpa_port_attrs, &port_obj);
		if (status) {
		    pr_err("Failed to create port object for device %s.\n",
			   ndev->name);
		    goto done;
		}

		dqnet_rfap_set_imp_lag_port(dev, 0, DQNET_SET_IMP_AUTO);
		dev->dev_port = (u8)ndev->if_id;

	} else if (ndev->if_id == rdpa_if_wan0 || ndev->if_id == rdpa_if_wan1) {
		port_cfg.emac = rdpa_emac_none;
		rdpa_port_index_set(rdpa_port_attrs, ndev->if_id);
		rdpa_port_cfg_set(rdpa_port_attrs, &port_cfg);
		status = bdmf_new_and_set(rdpa_port_drv(), NULL,
					  rdpa_port_attrs, &port_obj);
		if (status) {
		    pr_err("Failed to create port object for device %s.\n",
			   ndev->name);
		    goto done;
		}
	} else if (ndev->if_id == rdpa_if_ssid0) {
		goto filter_cfg;
	} else {
		goto done;
	}

	dir = ndev->if_id == rdpa_if_wan0 || ndev->if_id == rdpa_if_wan1 ?
		rdpa_dir_us : rdpa_dir_ds;
	rdpa_egress_tm_dir_set(tm_attr, dir);
	rdpa_egress_tm_index_set(tm_attr, ndev->if_id);
	rdpa_egress_tm_level_set(tm_attr, rdpa_tm_level_queue);
	rdpa_egress_tm_mode_set(tm_attr, rdpa_tm_sched_sp);
	status = bdmf_new_and_set(rdpa_egress_tm_drv(), port_obj,
				  tm_attr, &tm_obj);
	if (status) {
	    pr_err("Failed to create egress tm object for device %s.\n",
		   ndev->name);
	    goto done;
	}

	pr_debug("Configuring queues for dev %s\n", dev->name);
	q_cfg.drop_alg = rdpa_tm_drop_alg_dt;
	if (ndev->if_id >= rdpa_if_lan0 && ndev->if_id <= rdpa_if_lan5) {
		q_count = EGRESS_Q_COUNT_LAN_PORT;
	} else {
		q_count = EGRESS_Q_COUNT_NON_LAN_PORT;
	}
	for (q_id = 0; q_id < q_count; ++q_id)
	{
		/* match queues 1, 3, 5, and 7 thresholds to switch queue
		 * thresholds */
		if (ndev->if_id >= rdpa_if_lan0 && ndev->if_id <= rdpa_if_lan5) {
			if (q_id % 2) {
				q_cfg.drop_threshold = EGRESS_Q_THRESHOLD_IMP2LAN;
			} else {
				/* LAN-LAN queues should be ignored runner */
				q_cfg.drop_threshold = EGRESS_Q_THRESHOLD_LAN2LAN;
			}
		} else {
			q_cfg.drop_threshold = EGRESS_Q_THRESHOLD_OTHER;
		}
		pr_debug("Setting queue %d threshold to %d, ifid %u, link_type %u\n",
			q_id, q_cfg.drop_threshold, ndev->if_id, ndev->link_type);
		q_cfg.queue_id = q_id;
		status = rdpa_egress_tm_queue_cfg_set(tm_obj, q_id, &q_cfg);
		if (status) {
			pr_err("Failed to create egress tm object for device %s.\n",
			       ndev->name);
			goto done;
		}
	}

	port_tm_cfg.sched = tm_obj;
	status = rdpa_port_tm_cfg_set(port_obj, &port_tm_cfg);
	if (status) {
	    pr_err("Failed to configure port tm object for device %s.\n",
		   ndev->name);
	    goto done;
	}

	if (dir == rdpa_dir_ds) {
		status = dqnet_rfap_tc_link_port(ndev->if_id);
		if (status)
		    goto done;
	}

filter_cfg:
	status = rdpa_filter_get(&filter_obj);
	if (status) {
	    pr_err("Failed to get rdpa filter object %s.\n",
		   ndev->name);
	    goto done;
	}
	for (f_idx = 0; f_idx < sizeof(lan_filters)/sizeof(rdpa_filter); f_idx++)
	{
		filter_key.filter = lan_filters[f_idx];
		filter_key.ports = rdpa_if_id((ndev->if_id == rdpa_if_ssid0)?rdpa_if_wlan0:ndev->if_id);
		status = rdpa_filter_entry_set(filter_obj, &filter_key, &filter_entry);
		if (status) {
			pr_err("Failed to rdpa_filter_entry_set %s.\n",
				   ndev->name);
			goto done;
		}
	}

done:
	if (switch_port_obj)
		bdmf_put(switch_port_obj);
	if (port_obj)
		bdmf_put(port_obj);
	if (tm_obj)
		bdmf_put(tm_obj);
	if (filter_obj)
		bdmf_put(filter_obj);

	pr_debug("%s:<--\n",dev->name);
	return status;
}

static int dqnet_rfap_get_tx_prio(struct net_device *dev, int *prio)
{
	*prio = 0;
	return 0;
}

static int dqnet_rfap_get_tx_q_type(struct net_device *dev,
				   enum dqnet_q_type *q_type)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct dqnet_channel *chan = ndev->chan;

	switch (chan->type) {
	case DQNET_CHAN_TYPE_FAP_EXCEPT:
		if (ndev->if_id == rdpa_if_wan0 ||
		    ndev->if_id == rdpa_if_wan1)
			*q_type = DQNET_QUEUE_US;
		else
			*q_type = DQNET_QUEUE_DS;
		break;
	case DQNET_CHAN_TYPE_FAP_HOST:
	case DQNET_CHAN_TYPE_POINT_TO_POINT:
	case DQNET_CHAN_TYPE_POINT_TO_POINT_SWAP:
	case DQNET_CHAN_TYPE_SPOOFED:
		*q_type = DQNET_QUEUE_US;
		break;
	default:
		netif_err(ndev, tx_err, dev, "Invalid channel type %d\n",
			  chan->type);
		status = -EINVAL;
		goto done;
	}

done:
	return status;
}

/*
 * Query Runner for stats on the interface
 *
 * Parameters
 *      dev	device to query
 *      stats	statistics struct to fill in
*/
#if LINUX_VERSION_CODE > KERNEL_VERSION(4, 9, 0)
static void dqnet_rfap_get_stats(struct net_device *dev,
				 struct rtnl_link_stats64 *stats)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	int status = 0;
	bdmf_object_handle port_obj = NULL;
	rdpa_port_stat_t rstats;
	__u64 multicast = 0;
	struct pcpu_sw_netstats tmp, sum = { 0 };
	unsigned int cpu;

	pr_debug("%s:-->\n",dev->name);

	memset(stats, 0, sizeof(struct rtnl_link_stats64));

	if (ndev->chan->type == DQNET_CHAN_TYPE_FAP_EXCEPT &&
	    (ndev->if_id == rdpa_if_wan0 || ndev->if_id == rdpa_if_wan1)) {
		status = rdpa_port_get(ndev->if_id, &port_obj);
		if (status) {
			pr_err("Failed to get port ");
			pr_cont("object for device %s.\n", ndev->name);
			goto done;
		}
		status = rdpa_port_stat_get(port_obj, &rstats);
		if (status) {
			pr_err("Failed to get port ");
			pr_cont("stats for device %s.\n", ndev->name);
			goto done;
		}

		if (rdpa_iptv_drv() && ndev->nethooks.flow_mdb_counter_update)
		{
			status = ndev->nethooks.flow_mdb_counter_update(dev);
			if (status)
				pr_debug("Failed to get mcast stat\n");
			else {
				if (ndev->nethooks.offload_stats.mcast_fast) {
					for_each_possible_cpu(cpu) {
						unsigned int start;
						const struct pcpu_sw_netstats *tstats
							= per_cpu_ptr(ndev->nethooks.offload_stats.mcast_fast, cpu);
						do {
							start = u64_stats_fetch_begin(&tstats->syncp);
							memcpy(&tmp, tstats, sizeof(tmp));
						} while (u64_stats_fetch_retry(&tstats->syncp, start));
						multicast += tmp.rx_packets;
					}
				}
			}
		}
		if (ndev->nethooks.offload_stats.fast) {
			for_each_possible_cpu(cpu) {
				unsigned int start;
				const struct pcpu_sw_netstats *tstats
					= per_cpu_ptr(ndev->nethooks.offload_stats.fast, cpu);
				do {
					start = u64_stats_fetch_begin(&tstats->syncp);
					memcpy(&tmp, tstats, sizeof(tmp));
				} while (u64_stats_fetch_retry(&tstats->syncp, start));
				sum.tx_bytes   += tmp.tx_bytes;
				sum.tx_packets += tmp.tx_packets;
				sum.rx_bytes   += tmp.rx_bytes;
				sum.rx_packets += tmp.rx_packets;
			}
		}

		if (ndev->nethooks.offload_stats.slow) {
			for_each_possible_cpu(cpu) {
				unsigned int start;
				const struct pcpu_sw_netstats *tstats
					= per_cpu_ptr(ndev->nethooks.offload_stats.slow, cpu);
				do {
					start = u64_stats_fetch_begin(&tstats->syncp);
					memcpy(&tmp, tstats, sizeof(tmp));
				} while (u64_stats_fetch_retry(&tstats->syncp, start));
				sum.tx_bytes   += tmp.tx_bytes;
				sum.tx_packets += tmp.tx_packets;
				sum.rx_bytes   += tmp.rx_bytes;
				sum.rx_packets += tmp.rx_packets;
			}
		}

		stats->rx_packets = sum.rx_packets;
		stats->rx_bytes = sum.rx_bytes;

		stats->tx_packets = sum.tx_packets;
		stats->tx_bytes = sum.tx_bytes;

		stats->rx_dropped  = ndev->stats.netdev.rx_dropped;
		stats->rx_crc_errors = rstats.rx_crc_error_pkt;
		stats->rx_dropped += rstats.rx_discard_1;
		stats->rx_dropped += rstats.rx_discard_2;

		stats->multicast  = ndev->stats.netdev.multicast;
		stats->multicast += multicast;
		stats->rx_packets += multicast;

		stats->tx_dropped  = ndev->stats.netdev.tx_dropped;
		stats->tx_dropped += rstats.tx_discard;
		stats->tx_errors = ndev->stats.netdev.tx_errors;
		stats->tx_fifo_errors = ndev->stats.netdev.tx_fifo_errors;
		stats->tx_carrier_errors = ndev->stats.netdev.tx_carrier_errors;
	}
done:
	pr_debug("%s:<--\n",dev->name);
}
#else
static void dqnet_rfap_get_stats(struct net_device *dev,
				 struct rtnl_link_stats64 *stats)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	int status = 0;
	bdmf_object_handle port_obj = NULL;
	rdpa_port_stat_t rstats;
	bdmf_object_handle iptv;
	rdpa_iptv_stat_t iptv_stat;
	__u64 multicast = 0;
	struct pcpu_sw_netstats tmp, sum = { 0 };
	unsigned int cpu;

	pr_debug("%s:-->\n",dev->name);

	memset(stats, 0, sizeof(struct rtnl_link_stats64));

	if (ndev->chan->type == DQNET_CHAN_TYPE_FAP_EXCEPT &&
	    (ndev->if_id == rdpa_if_wan0 || ndev->if_id == rdpa_if_wan1)) {
		status = rdpa_port_get(ndev->if_id, &port_obj);
		if (status) {
			pr_err("Failed to get port ");
			pr_cont("object for device %s.\n", ndev->name);
			goto done;
		}
		status = rdpa_port_stat_get(port_obj, &rstats);
		if (status) {
			pr_err("Failed to get port ");
			pr_cont("stats for device %s.\n", ndev->name);
			goto done;
		}

		if (rdpa_iptv_drv())
		{
			status = rdpa_iptv_get(&iptv);
			if (status) {
				pr_err("Failed to get iptv object ");
				pr_cont(" for device %s.\n", ndev->name);
				goto done;
			}
			status = rdpa_iptv_iptv_stat_get(iptv, &iptv_stat);
			if (status) {
				pr_err("Failed to get iptv ");
				pr_cont("stats for device %s.\n", ndev->name);
				goto done;
			}
			multicast = iptv_stat.rx_valid_pkt;
		}
		if (ndev->nethooks.offload_stats) {
			for_each_possible_cpu(cpu) {
				unsigned int start;
				const struct pcpu_sw_netstats *tstats
					= per_cpu_ptr(ndev->nethooks.offload_stats, cpu);
				do {
					start = u64_stats_fetch_begin(&tstats->syncp);
					memcpy(&tmp, tstats, sizeof(tmp));
				} while (u64_stats_fetch_retry(&tstats->syncp, start));
				sum.tx_bytes   += tmp.tx_bytes;
				sum.tx_packets += tmp.tx_packets;
				sum.rx_bytes   += tmp.rx_bytes;
				sum.rx_packets += tmp.rx_packets;
			}
		}
		stats->rx_packets  = ndev->stats.netdev.rx_packets;
		stats->rx_packets += sum.rx_packets;
		stats->rx_bytes  = ndev->stats.netdev.rx_bytes;
		stats->rx_bytes += sum.rx_bytes;
		stats->rx_crc_errors = rstats.rx_crc_error_pkt;
		stats->rx_dropped  = ndev->stats.netdev.rx_dropped;
		stats->rx_dropped += rstats.rx_discard_1;
		stats->rx_dropped += rstats.rx_discard_2;

		stats->multicast  = ndev->stats.netdev.multicast;
		stats->multicast += multicast;
		stats->rx_packets += multicast;

		stats->tx_packets  = ndev->stats.netdev.tx_packets;
		stats->tx_packets += sum.tx_packets;
		stats->tx_bytes  = ndev->stats.netdev.tx_bytes;
		stats->tx_bytes += sum.tx_bytes;
		stats->tx_dropped  = ndev->stats.netdev.tx_dropped;
		stats->tx_dropped += rstats.tx_discard;
		stats->tx_errors = ndev->stats.netdev.tx_errors;
		stats->tx_fifo_errors = ndev->stats.netdev.tx_fifo_errors;
		stats->tx_carrier_errors = ndev->stats.netdev.tx_carrier_errors;
	}
done:
	pr_debug("%s:<--\n",dev->name);
}
#endif

static void dqnet_rfap_clr_stats(struct net_device *dev)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	int status = 0;
	bdmf_object_handle port_obj = NULL;
	rdpa_port_stat_t rstats;
	bdmf_object_handle iptv;
	rdpa_iptv_stat_t iptv_stat;

	pr_debug("%s:-->\n",dev->name);

	if (ndev->chan->type == DQNET_CHAN_TYPE_FAP_EXCEPT &&
	    (ndev->if_id == rdpa_if_wan0 || ndev->if_id == rdpa_if_wan1)) {
		status = rdpa_port_get(ndev->if_id, &port_obj);
		if (status) {
			pr_err("Failed to get port ");
			pr_cont("object for device %s.\n", ndev->name);
			goto done;
		}
		status = rdpa_port_stat_set(port_obj, &rstats);
		if (status) {
			pr_err("Failed to set port ");
			pr_cont("stats for device %s.\n", ndev->name);
			goto done;
		}

		status = rdpa_iptv_get(&iptv);
		if (status) {
			pr_err("Failed to get iptv object ");
			pr_cont(" for device %s.\n", ndev->name);
			goto done;
		}
		status = rdpa_iptv_iptv_stat_set(iptv, &iptv_stat);
		if (status) {
			pr_err("Failed to get iptv ");
			pr_cont("stats for device %s.\n", ndev->name);
			goto done;
		}
	}
done:
	pr_debug("%s:<--\n",dev->name);
}

static struct rtnl_link_stats64 *dqnet_rfap_link_stats(struct net_device *dev,
		struct rtnl_link_stats64 *stats)
{
	if (stats)
		dqnet_rfap_get_stats(dev, stats);
	else
		dqnet_rfap_clr_stats(dev);

	return stats;
}

static int dqnet_rfap_get_tag_len(struct net_device *dev)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	int len = 0;

	if (ndev->chan->type == DQNET_CHAN_TYPE_FAP_EXCEPT &&
	    (ndev->if_id == rdpa_if_wan0 || ndev->if_id == rdpa_if_wan1))
		len = cpu_get_docsis_hdr_len();

	return len;
}

static int dqnet_rfap_add_tag(struct net_device *dev, struct fpm_buff *fb)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);
	struct net_device *dev_in = NULL;
	struct dqnet_netdev *ndev_in = NULL;
	rdpa_if srcport = rdpa_if_cpu;

	if (!fb->fap_tag_len)
		goto done;

	/* This code assumes skb_iif is updated by flowmgr module */
	dev_in =  fb->dev_in;
	if (dev_in && dev_in->group) {
		if (BCM_NETDEVICE_GROUP_CAT(dev_in->group) == BCM_NETDEVICE_GROUP_CAT_PHY) {
			ndev_in = netdev_priv(dev_in);
			srcport = ndev_in->if_id;
		}
	} else if (dev_in) {
		if (strncmp(dev_in->name, "wl0", 3) == 0)
			srcport = rdpa_if_ssid0;
		else if (strncmp(dev_in->name, "wl1", 3) == 0)
			srcport = rdpa_if_ssid16;
		else if (strncmp(dev_in->name, "wl2", 3) == 0)
			srcport = rdpa_if_ssid32;
	}

	if (fb->type == BUF_TYPE_FPM) {
		fb->data -= fb->fap_tag_len;
		fb->len += fb->fap_tag_len;
		fb->offset -= fb->fap_tag_len;
		fpm_set_token_size(&fb->token, fb->len);
		cpu_tx_encode_docsis_hdr(fb->data, fb->offset, fb->token,
					 fb->len - fb->fap_tag_len,
					 ndev->if_id, srcport);
	} else {
		if (skb_headroom(fb->skb) < fb->fap_tag_len) {
			struct sk_buff *new_skb;
			new_skb = skb_realloc_headroom(fb->skb,
						       fb->fap_tag_len);
			dev_kfree_skb(fb->skb);
			if (!new_skb)
				return -ENOMEM;
			fb->skb = new_skb;
		}
		skb_push(fb->skb, fb->fap_tag_len);

		cpu_tx_encode_docsis_hdr(fb->skb->data, 0, 0,
					 fb->skb->len - fb->fap_tag_len,
					 ndev->if_id, srcport);
	}

done:
	return 0;
}

static enum dqnet_qmsg_fmt dqnet_rfap_get_qmsg_fmt(struct net_device *dev)
{
	return DQNET_QMSG_FMT_RFAP_FPM;
}

/*
 * Encode a Runner TX message
 *
 * Parameters
 *      fb	packet buffer descriptor
 *      msgdata	DQM message
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_encode_q_msg_rfap(struct dqnet_netdev *ndev,
				   struct fpm_buff *fb,
				   u32 *msgdata)
{
	int status = 0;
	int pd_done = 0;
	rdpa_cpu_tx_info_t tx_info;
	struct dqnet_channel *chan = ndev->chan;
#ifdef CONFIG_BCM_DHD_RUNNER
	rdpa_dhd_tx_post_info_t dhd_tx_info;
	int radio, flowring = -1, ssid;
	unsigned char pri = 0;

	u8 wifi_pri = 0;
	u16 wifi_flr_idx = 0;

	if (fb->if_id >= rdpa_if_ssid0 &&
	    fb->if_id <= rdpa_if_ssid47 &&
	    ndev->dhdol_get_flow) {
		memset(&dhd_tx_info, 0, sizeof(rdpa_dhd_tx_post_info_t));
		ssid = (ndev->if_id - rdpa_if_ssid0) % 16;
		radio = (ndev->if_id - rdpa_if_ssid0) / 16;
		/* Take 3-bit priority from skb priority */
		if (fb->skb)
			pri = fb->skb->priority & 0x7;
		else
			pri = fb->priority;
		status = ndev->dhdol_get_flow(ndev->dhd_cntx, (u8 *) fb->skb,
					      fb->len, pri, ssid, &radio, &flowring);
		if (!status) {
			/* Stash WiFi metadata */
			if (fb->skb) {
				wifi_pri = (flowring & WL_METADATA_PRIORITY_MASK) >> WL_METADATA_PRIORITY_SHIFT;
				wifi_flr_idx = flowring & WL_METADATA_FLOWRING_MASK;
				nf_ct_offload_update_wifi(fb->skb, wifi_flr_idx,
							  wifi_pri);
			}

			dhd_tx_info.data = fb->token;
			dhd_tx_info.data_offset = fb->offset;
			dhd_tx_info.data_size = fb->len;
			dhd_tx_info.radio_idx = radio;
			dhd_tx_info.flow_ring_id = flowring;
			dhd_tx_info.ssid_if_idx = (uint32_t)fb->if_id;
			status = rdpa_cpu_dhd_pd_set(&dhd_tx_info, msgdata);
			if (status) {
				pr_err("%s: Failure calling rdpa_cpu_dhd_pd_set.",
			       __func__);
				goto done;
			}

			/* Mark PD done */
			pd_done = 1;
		} else {
			/* For dhd_offload, set status = -ECOMM to dqnet to suppress
			 * printing of msg to console and drop the packet.
			 */
			if (radio == -1 && flowring == -1)
				status = -ECOMM;
			pr_debug("%s: Failure calling get_dhdol_info. status %d, radio %d, flowring %d",
				__func__, status, radio, flowring);

			if (fb->skb)
				nf_ct_offload_update_wifi(fb->skb, WL_METADATA_FLOWRING_MASK, 0);

			goto done;
		}
	}
#endif

	if(!pd_done) {
		memset(&tx_info, 0, sizeof(rdpa_cpu_tx_info_t));
		switch (chan->type) {
		case DQNET_CHAN_TYPE_FAP_EXCEPT:
			tx_info.method = rdpa_cpu_tx_port;
			break;
		case DQNET_CHAN_TYPE_FAP_HOST:
			tx_info.method = rdpa_cpu_tx_bridge;
			break;
		default:
			pr_debug("%s: Dqnet Channel type unknown.\n", __func__);
			break;
		}
		tx_info.port = fb->if_id;

		if (fb->type == BUF_TYPE_FPM) {
			tx_info.data = fb->token;
			tx_info.data_offset = fb->offset;
			tx_info.data_size = fb->len;
			tx_info.buff_type = rdpa_buf_type_rdp;
		} else {
			memcpy(&tx_info.data, fb->skb->cb,
				   sizeof(tx_info.data));
			tx_info.context = (uint32_t) fb->skb;
			tx_info.data_offset = 0;
			tx_info.data_size = fb->skb->len;
			tx_info.buff_type = rdpa_buf_type_host;
		}

		if (tx_info.port == rdpa_if_wan0 || tx_info.port == rdpa_if_wan1) {
			tx_info.x.wan.queue_id = tx_info.port;
		} else if ((tx_info.port >= rdpa_if_ssid16) &&
			   (tx_info.port <= rdpa_if_ssid31)) {
			tx_info.x.lan.queue_id = 2;
		} else if ((tx_info.port >= rdpa_if_ssid32) &&
			   (tx_info.port <= rdpa_if_ssid47)) {
			tx_info.x.lan.queue_id = 4;
		} else if ((tx_info.port >= rdpa_if_lan0) &&
			   (tx_info.port <= rdpa_if_lan5)) {
			/* priority to queue_id mapping as follows
			   priority 1,0 -> queue_id 1
			   priority 3,2 -> queue_id 3
			   priority 5,4 -> queue_id 5
			   priority 7,6 -> queue_id 7
			   to match switch QOS settings*/
			if ((fb->priority % 2) == 0) {
				tx_info.x.lan.queue_id = fb->priority + 1;
			} else {
				tx_info.x.lan.queue_id = fb->priority;
			}
			tx_info.x.lan.lag_id =
				dqnet_rfap_get_imp_lag_port(fb->dev_out);
		} else {
			tx_info.x.lan.queue_id = 0;
		}
		status = rdpa_cpu_tx_pd_set(&tx_info, msgdata);
		if (status) {
			pr_err("%s: Failure calling rdpa_cpu_tx_pd_set.\n", __func__);
			goto done;
		}
	}

done:
	return status;
}

u64 rdpa_cpu_reason_cnt[rdpa_cpu_reason__num_of];

/*
 * Decode a Runner RX message.
 *
 * Parameters
 *      msgdata	DQM message
 *      fb	packet buffer descriptor
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_decode_q_msg_rfap(struct dqnet_channel *chan,
				   u32 *msgdata,
				   struct fpm_buff *fb)
{
	int status = 0;
	rdpa_cpu_rx_info_t rx_info;

	memset(&rx_info, 0, sizeof(rdpa_cpu_rx_info_t));
	memset(fb, 0, sizeof(struct fpm_buff));
	status = rdpa_cpu_rx_pd_get(msgdata, &rx_info);
	if (status) {
		pr_err("%s: Failure calling rdpa_cpu_rx_pd_get.\n", __func__);
		goto done;
	}
	rdpa_cpu_reason_cnt[rx_info.reason]++;
	if ((rx_info.buff_type == rdpa_buf_type_rdp) ||
	    (chan->qmsg_fmt == DQNET_QMSG_FMT_RFAP_FPM)) {
		fb->type = BUF_TYPE_FPM;
		fb->token = rx_info.data;
		if (!fpm_is_valid_token(fb->token)) {
			pr_err("%s: Invalid token 0x%08x.\n",
			       __func__, fb->token);
			status = -EIO;
			goto done;
		}
		fpm_track_token_rx(fb->token);
		fb->buf = fpm_token_to_buffer(fb->token);
		fb->offset = rx_info.data_offset;
		fb->data = fb->buf + fb->offset;
		fb->len = rx_info.size;
		if (!fb->len ||
		    fpm_get_token_size(fb->token) != fb->len) {
			pr_err("%s: Token size (%d) does not match pkt ",
			       __func__, fpm_get_token_size(fb->token));
			pr_cont("desc size (%d) %x.\n", fb->len,
				fb->token);
			status = -EIO;
			goto done;
		}
	} else {
		if (rx_info.buff_type == rdpa_buf_type_host) {
			fb->type = BUF_TYPE_SKB;
			fb->skb = (struct sk_buff *)rx_info.data;
			if (!fb->skb) {
				__WARN();
				status = -EIO;
				goto done;
			}
			skb_put(fb->skb, rx_info.size);
			pr_debug("%s skb %p skb->head %p skb->data %p skb->len %d\n",
				 __func__, fb->skb, fb->skb->head, fb->skb->data, fb->skb->len);
		} else if (rx_info.buff_type == rdpa_buf_type_rdp) {
			__WARN();
			status = -EIO;
			goto done;
		}
	}

	fb->if_id = rx_info.src_port;
	if (fb->if_id >= rdpa_if__number_of) {
		pr_err("%s: if_id %d greater than max allowed.\n", __func__,
		       fb->if_id);
		status = -EINVAL;
		goto done;
	}
	if ((fb->if_id >= rdpa_if_lan0) && (fb->if_id <= rdpa_if_lan_max))
		fb->if_sub_id = rdpa_if_to_switch_port(fb->if_id);
	else
		fb->if_sub_id = 0;
	fb->priority = rx_info.cmim_priority;
	fb->cmim = rx_info.cmim_if;
	fb->brcm_tag_len = 0;
	fb->fap_tag_len = 0;
done:
	return status;
}

/*
 * Translate an interface ID/SubID to a switch port
 *
 * Parameters
 *      if_id		interface ID
 *      if_sub_id	interface sub-ID
 *      port		switch port #
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_if_id_subid_to_port_rfap(u32 if_id, u32 if_sub_id, u8 *port)
{
	int status = 0;

	if (unlikely(if_id >= rdpa_if_lan_max)) {
		pr_err("%s: Interface ID greater than max LAN ports allowed.\n",
		       __func__);
		status = -EINVAL;
		goto done;
	}
	*port = rdpa_if_to_switch_port(if_id);

done:
	return status;
}

/*
 * Translate a switch port to an interface ID/SubID.
 *
 * Parameters
 *      port		switch port #
 *      if_id		interface ID
 *      if_sub_id	interface sub-ID
 *
 * Returns
 *	0 for success, < 0 is error code
 */
static int dqnet_port_to_if_id_subid_rfap(u8 port, u32 *if_id, u32 *if_sub_id)
{
	int status = 0;

	*if_id = switch_port_to_rdpa_if(port);

	return status;
}

static int dqnet_max_if_id_rfap(void)
{
	return rdpa_if__number_of;
}

static int dqnet_max_if_subid_rfap(void)
{
	return 2;
}

static char *rdpa_cpu_reason_str(int reason)
{
	char *str = NULL;
	switch (reason) {
	case rdpa_cpu_rx_reason_mcast:
		str = "mcast";
		break;
	case rdpa_cpu_rx_reason_bcast:
		str = "bcast";
		break;
	case rdpa_cpu_rx_reason_igmp:
		str = "igmp";
		break;
	case rdpa_cpu_rx_reason_icmpv6:
		str = "icmpv6";
		break;
	case rdpa_cpu_rx_reason_dhcp:
		str = "dhcp";
		break;
	case rdpa_cpu_rx_reason_non_tcp_udp:
		str = "non_tcp_udp";
		break;
	case rdpa_cpu_rx_reason_ipv6_ext_header:
		str = "ipv6_ext_header";
		break;
	case rdpa_cpu_rx_reason_hdr_err:
		str = "hdr_err";
		break;
	case rdpa_cpu_rx_reason_direct_queue_0:
		str = "direct_queue_0";
		break;
	case rdpa_cpu_rx_reason_direct_queue_1:
		str = "direct_queue_1";
		break;
	case rdpa_cpu_rx_reason_direct_queue_2:
		str = "direct_queue_2";
		break;
	case rdpa_cpu_rx_reason_direct_queue_3:
		str = "direct_queue_3";
		break;
	case rdpa_cpu_rx_reason_etype_udef_0:
		str = "etype_udef_0";
		break;
	case rdpa_cpu_rx_reason_etype_udef_1:
		str = "etype_udef_1";
		break;
	case rdpa_cpu_rx_reason_etype_udef_2:
		str = "etype_udef_2";
		break;
	case rdpa_cpu_rx_reason_etype_udef_3:
		str = "etype_udef_3";
		break;
	case rdpa_cpu_rx_reason_etype_arp:
		str = "etype_arp";
		break;
	case rdpa_cpu_rx_reason_pci_ip_flow_miss_1:
		str = "pci_ip_flow_miss_1";
		break;
	case rdpa_cpu_rx_reason_pci_ip_flow_miss_2:
		str = "pci_ip_flow_miss_2";
		break;
	case rdpa_cpu_rx_reason_pci_ip_flow_miss_3:
		str = "pci_ip_flow_miss_3";
		break;
	case rdpa_cpu_rx_reason_ip_flow_miss:
		str = "ip_flow_miss";
		break;
	case rdpa_cpu_rx_reason_tcp_flags:
		str = "tcp_flags";
		break;
	case rdpa_cpu_rx_reason_ttl_expired:
		str = "ttcl_expired";
		break;
	case rdpa_cpu_rx_reason_mtu_exceeded:
		str = "mtu_exceeded";
		break;
	case rdpa_cpu_rx_reason_l4_icmp:
		str = "l4_icmp";
		break;
	case rdpa_cpu_rx_reason_l4_esp:
		str = "l4_esp";
		break;
	case rdpa_cpu_rx_reason_l4_gre:
		str = "l4_gre";
		break;
	case rdpa_cpu_rx_reason_l4_ah:
		str = "l4_ah";
		break;
	case rdpa_cpu_rx_reason_l4_ipv6:
		str = "l4_ipv6";
		break;
	case rdpa_cpu_rx_reason_icmp_echo:
		str = "icmp_echo";
		break;
	default:
		str=NULL;
		break;
	}
	return str;
}

static int dqnet_rfap_dump(struct seq_file *s)
{
	int i;
	char *reason_str;
	pr_seq(s, "--- DQNET RFAP debug dump ----\n");
	pr_seq(s, "imp_lag_ports              : %d\n", rdp_get_lag_count());
	if (!altwan_enable)
		pr_seq(s, "altwan                     : %s\n", "Disabled");
	else
		pr_seq(s, "altwan                     : rdpa_emac%d\n", altwan_imp);
	for (i = 0; i < IMP_MAX_PORTS; i++) {
		pr_seq(s, "imp[%d] enable[%d] mask(0x%02x|0x%02x)    : %llu\n", i,
			imp_lag[i].enable, imp_lag[i].if_mask,
			imp_lag[i].reserve_if_mask, imp_lag[i].usage);
	}
		pr_seq(s, "exception packet counters  :\n");
	for (i = 0; i < rdpa_cpu_reason__num_of; i++)
		if (rdpa_cpu_reason_cnt[i]) {
			reason_str = rdpa_cpu_reason_str(i);
			if (reason_str)
				pr_seq(s, "  %-24s : %llu\n", reason_str,
					rdpa_cpu_reason_cnt[i]);
			else
				pr_seq(s, "  rdpa_cpu_reason_cnt[%02d]  : %llu\n", i,
				       rdpa_cpu_reason_cnt[i]);
		}

	return 0;
}

static int dqnet_rfap_get_expected_queue(struct net_device *dev)
{
	struct dqnet_netdev *ndev;
	int queue;
	ndev = netdev_priv(dev);
	if (!ndev)
		return 0;
	for (queue = 0; queue < ndev->chan->rx_q_count; queue++) {
		if (ndev->chan->rx_q_info[queue].q_type == DQNET_QUEUE_EXP)
			return ndev->chan->rx_q_info[queue].q_num - RUNNER_FIRST_DQM_QUEUE;
	}
	return 0;
}

int dqnet_rfap_get_q_info(struct net_device *dev,
			  struct dqnet_channel *chan)
{
	strncpy(chan->name, "fap-exception", sizeof(chan->name)-1);
	return 0;
}

static int dqnet_rfap_set_wan(struct dqnet_netdev *ndevs,
			      char *ifname, int enable)
{
	int i, status = 0, unreserved_ports = 0, curr_mode = altwan_enable;
	struct dqnet_netdev *ndev, *ndev_altwan = NULL, *ndev_imp_reserve = NULL;
	int switch_port;
	bdmf_object_handle rdpa_system_obj = NULL;
	rdpa_system_cfg_t system_cfg;

	if (enable && (curr_mode == ALTWAN_DISABLED)) {
		pr_info("Switching WAN mode to AltWan enable\n");
		altwan_enable = ALTWAN_ENABLED;
	} else if (!enable && (curr_mode == ALTWAN_ENABLED)) {
		pr_info("Switching WAN mode to AltWan disable\n");
		altwan_enable = ALTWAN_DISABLED;
	}

	switch_port = imp_lag[altwan_imp].switch_id;
	rcu_read_lock();
	list_for_each_entry_rcu(ndev, &ndevs->list, list) {
		if (ndev->link_type != DQNET_LINK_TYPE_SWITCH)
			continue;

		if (!strncmp(ndev->dev->name, ifname, sizeof(IFNAMSIZ)))
			ndev_altwan = ndev;

		if (ndev->swport_imp_reserve)
			ndev_imp_reserve = ndev;
	}

	if (ndev_altwan) {
		ndev = ndev_altwan;
		status = rdpa_system_get(&rdpa_system_obj);
		if (status) {
			pr_err("%s: Call to get system object failed.\n",
					__func__);
			status = -EIO;
			goto done;
		}
		status = rdpa_system_cfg_get(rdpa_system_obj, &system_cfg);
		if (status) {
			pr_err("%s: Call to get system cfg object failed.\n",
					__func__);
			status = -EIO;
			goto done;
		}
		if (enable) {
			if (BCM_NETDEVICE_GROUP_TYPE(ndev->dev->group) ==
			    BCM_NETDEVICE_GROUP_WAN) {
				status =  -1;
				goto done;
			}

			for (i = 0; i < IMP_MAX_PORTS; i++)
				if (imp_lag[i].enable &&
						!imp_lag[i].reserve_if_mask)
					unreserved_ports++;

			if (unreserved_ports < 2) {
				dqnet_rfap_set_imp_lag_port(ndev->dev, 0,
						DQNET_SET_IMP_RESET);
				ndev_imp_reserve = NULL;
			}

			dqnet_rfap_set_imp_lag_port(ndev->dev, altwan_imp,
						   DQNET_SET_IMP_RESERVE);
			ndev->brcmtag_opc_uc = BRCM_OPCODE_1;
			ndev->dev->group = BCM_NETDEVICE_GROUP_WAN;

			pr_info("set_wan: Configure Runner in Altwan mode [%d]\n",
				altwan_imp);
			system_cfg.alt_wan_enabled = 1;
			status = rdpa_system_cfg_set(rdpa_system_obj, &system_cfg);
			if (status) {
				pr_err("%s: Call to set system cfg object failed.\n",
						__func__);
				status = -EIO;
				goto done;
			}
		} else {
			if (BCM_NETDEVICE_GROUP_TYPE(ndev->dev->group) !=
			    BCM_NETDEVICE_GROUP_WAN) {
				status =  -1;
				goto done;
			}
			dqnet_rfap_set_imp_lag_port(ndev->dev, 0,
						   DQNET_SET_IMP_RESET);
			if (ndev_imp_reserve)
				dqnet_rfap_set_imp_lag_port(
						ndev_imp_reserve->dev, 1,
						DQNET_SET_IMP_RESERVE);

			ndev->brcmtag_opc_uc = BRCM_OPCODE_0;
			ndev->dev->group = BCM_NETDEVICE_GROUP_LAN;

			pr_info("set_wan: Configure Runner in DOCSIS mode\n");
			system_cfg.alt_wan_enabled = 0;
			status = rdpa_system_cfg_set(rdpa_system_obj, &system_cfg);
			if (status) {
				pr_err("%s: Call to set system cfg object failed.\n",
						__func__);
				status = -EIO;
				goto done;
			}
		}

		ethsw_alt_ethwan(switch_port, ndev->if_sub_id , enable);

		pr_info("%s: dqnet_set_imp_lag %d wan %d\n", ndev->dev->name,
			ndev->swport_imp, enable);
	} else
		pr_err("%s: No device found in list ifname:%s enable:%d", __func__, ifname, enable);

	list_for_each_entry_rcu(ndev, &ndevs->list, list) {
		if (enable &&
		    !strncmp(ndev->dev->name, ifname, sizeof(IFNAMSIZ)))
			continue;

		if (ndev_imp_reserve &&
		    !strncmp(ndev->dev->name, ndev_imp_reserve->dev->name,
			    sizeof(IFNAMSIZ)))
			continue;

		if (ndev->link_type != DQNET_LINK_TYPE_SWITCH)
			continue;

		if (BCM_NETDEVICE_GROUP_TYPE(ndev->dev->group) !=
		    BCM_NETDEVICE_GROUP_WAN) {
			/* Only one switch port should act as WAN */
			ndev->dev->group = BCM_NETDEVICE_GROUP_LAN;
			ndev->brcmtag_opc_uc = BRCM_OPCODE_0;
		}
		dqnet_rfap_set_imp_lag_port(ndev->dev, 0,
					   DQNET_SET_IMP_AUTO);

		pr_info("%s: dqnet_set_imp_lag %d\n", ndev->dev->name,
			ndev->swport_imp);

	}
done:
	rcu_read_unlock();

	return status;
}

#if defined(CONFIG_BCM_MSO_FEAT_MODULE)
static int dqnet_rfap_set_isolate(struct net_device *dev,
				  int enable)
{
	struct dqnet_netdev *ndev = netdev_priv(dev);

	if (ndev->link_type != DQNET_LINK_TYPE_SWITCH)
		return -1;

	return ethsw_port_isolate(imp_lag[ndev->swport_imp].switch_id,
				  ndev->if_sub_id,
				  enable);
}
#endif

static int dqnet_rfap_balance_imp_lag_ports(struct dqnet_netdev *ndevs,
					    int action, int poi)
{
	int status = 0;
	enum dqnet_imp_lag_action mapping = action;

	/* If more than 2 IMP ports or EthWan enabled, no remapping needed */
	if ((rdp_get_lag_count() > 2) || (altwan_enable == ALTWAN_ENABLED)) {
		goto done;
	}

	/* Reset IMP port mappings first to remove any reserved mask */
	status = dqnet_rfap_reset_imp_lag_port(ndevs);
	if (status) {
		pr_err("Failed to reset IMP LAG port mappings: %d\n", status);
		goto done;
	}

	switch (mapping) {
	case DQNET_IMP_LAG_ACT_OPT1:
	case DQNET_IMP_LAG_ACT_OPT2:
	case DQNET_IMP_LAG_ACT_OPT3:
	case DQNET_IMP_LAG_ACT_DEFAULT_NOMOCA:
	case DQNET_IMP_LAG_ACT_OPT1_NM:
	case DQNET_IMP_LAG_ACT_OPT2_NM:
	case DQNET_IMP_LAG_ACT_OPT3_NM:
		status = dqnet_rfap_set_static_imp_lag_config(ndevs, mapping, poi);
		break;

	case DQNET_IMP_LAG_ACT_DEFAULT:
	default:
		status = dqnet_rfap_set_static_imp_lag_config(ndevs,
							      DQNET_IMP_LAG_ACT_DEFAULT,
							      poi);
		break;
	}

done:
	pr_debug("status=%d for action (%d) poi (%d)\n", status, mapping, poi);
	return status;
}

static int dqnet_rfap_get_wifi_idx(struct dqnet_netdev *ndev)
{
	if (ndev)
		return ((ndev->if_id - rdpa_if_ssid0) % 16);
	else
		return -1;
}

static int dqnet_rfap_set_stp(struct net_device *dev, int state)
{
	int status = 0;
	struct dqnet_netdev *ndev = netdev_priv(dev);

	pr_debug("%s:-->\n",dev->name);
	status = ethsw_port_set_stp_state(ndev->if_sub_id, state);
	if (status)
		netdev_err(dev, "Failed to set switch port STP state.\n");
	pr_debug("%s:<--\n",dev->name);
	return status;
}


struct dqnet_fap_ops dqnet_rfap = {
	.init_system	= dqnet_rfap_init_system,
	.init_port	= dqnet_rfap_init_port,
	.get_tx_prio	= dqnet_rfap_get_tx_prio,
	.get_tx_q_type	= dqnet_rfap_get_tx_q_type,
	.link_stats	= dqnet_rfap_link_stats,
	.get_tag_len	= dqnet_rfap_get_tag_len,
	.add_tag	= dqnet_rfap_add_tag,
	.get_qmsg_fmt	= dqnet_rfap_get_qmsg_fmt,
	.dump		= dqnet_rfap_dump,
	.get_imp_lag_port = dqnet_rfap_get_imp_lag_port,
	.set_imp_lag_port = dqnet_rfap_set_imp_lag_port,
	.get_q_info	= dqnet_rfap_get_q_info,
	.set_wan	= dqnet_rfap_set_wan,
#if defined(CONFIG_BCM_MSO_FEAT_MODULE)
	.set_isolate	= dqnet_rfap_set_isolate,
#endif
	.get_expected_queue = dqnet_rfap_get_expected_queue,
	.balance_imp_lag_port = dqnet_rfap_balance_imp_lag_ports,
	.get_wifi_idx = dqnet_rfap_get_wifi_idx,
	.set_stp = dqnet_rfap_set_stp,
};
EXPORT_SYMBOL_GPL(dqnet_rfap);

struct dqnet_qmsg_ops dqnet_qmsg_rfap = {
	.encode_qmsg    = dqnet_encode_q_msg_rfap,
	.decode_qmsg    = dqnet_decode_q_msg_rfap,
	.id_to_port     = dqnet_if_id_subid_to_port_rfap,
	.port_to_id     = dqnet_port_to_if_id_subid_rfap,
	.max_if_id 	= dqnet_max_if_id_rfap,
	.max_if_subid 	= dqnet_max_if_subid_rfap,
};
EXPORT_SYMBOL_GPL(dqnet_qmsg_rfap);
MODULE_LICENSE("GPL");
