# $language = "python"
# $interface = "1.0"

# UserIO.py
#
# Description:
#     The user interface to issue command and access file is different between secureCRT and RGLinux,
#     so this module is designed to wrap the different implementation under secureCRT and RGLinux,
#     and provide APIs for different user I/O usages.


# Import modules
import os, time


# Constant defines
# latticeli
PROMPT_DEFAULT = ":RG]#"
CLI_ERROR = "Error"
CLI_SUCCESS = "Success"
CLI_DELIMITER = ": "
CLI_GET = "get"
CLI_SET = "set"

# dmcli
RDKM_PROMPT_DEFAULT = "root@"
DMCLI_SUCCESS = "Execution succeed."
DMCLI_GET = "getv"
DMCLI_SET = "setv"
DMCLI_VALUE = "value: "

# common
STR_ERROR_INVALID_CLI_CMD = "Invalid cli command!"
STR_ERROR_TIMEOUT = "Unexpected output!"
TIMEOUT = 5 # secs
CRT_TRIM_TRAIL = 7

# Global Variables
gIpc = False

# <InitCrt> Init crt module for use, for secureCRT only
def InitCrt(obj):
    global crt
    crt = obj
    return


# <InitLogging> Init logging
# <Mandatory Input> enable: Enable/Disable logging
# <Optional Input> logName: Log file name, should be specified with enable
def InitLogging(enable, logName = ""):
    if os.name == "nt":
        #Turn off logging before setting up our script's logging...
        crt.Session.Log(False)
        if enable == True:
            crt.Session.LogFileName = logName
            crt.Session.Log(True)
    else:
        if enable == True:
            global logFp
            logFp = open(logName, "w")
            if logFp == None:
                MessagePrint("Unable to open test log file", "Error")
        else:
            logFp.close()


# <ConsoleLog> Log console output to file, for RGLinux only
# <Mandatory Input> output: Console output; isCmd: True if print a single line cmd
def ConsoleLog(output, isCmd):
    if isCmd == True:
        currentTime = time.asctime(time.localtime())
        if logFp != None:
            logFp.write("\n" + currentTime + ": ")
            logFp.write(output + "\n")
        print("\n" + currentTime + ": ")
        print(output + "\n")
    else:
        if logFp != None:
            logFp.writelines(output)
        for line in output:
            print(line)


# <CmdInput> Input a command.
# <Mandatory Input> cmd: Command to input
def CmdInput(cmd):
    if os.name == "nt":
        crt.Screen.Synchronous = True
        crt.Screen.Send(cmd + "\n")
    else:
        os.system(cmd)


# <CmdInputGetOutput> Input a command and get the output.
# <Mandatory Input> cmd: Command to input
# <Optional Input> prompt: String to wait for, optional for secureCRT
def CmdInputGetOutput(cmd, prompt = PROMPT_DEFAULT):
    if os.name == "nt":
        crt.Screen.Synchronous = True
        crt.Screen.Send(cmd + "\n")
        result = crt.Screen.WaitForString(cmd + "\n", TIMEOUT)
        if result == 0:
            return STR_ERROR_TIMEOUT
        else:
            return crt.Screen.ReadString(prompt)[:-CRT_TRIM_TRAIL]
    else:
        ConsoleLog(cmd, True)
        output = os.popen(cmd).readlines()
        ConsoleLog(output, False)
        return output


# <LatticeCliSet> Set value to a Lattice node via latticecli
# <Mandatory Input> node: Lattice node; value: value to set
# <Optional Input> prompt: String to wait for, optional for secureCRT
# <1st Return> True for success, False for error
# <2rd Return> The output or the error message
# e.g. LatticeCliSet("System.TelnetEnable", "1")
def LatticeCliSet(node, value, prompt = PROMPT_DEFAULT):
    if os.name == "nt":
        crt.Screen.Synchronous = True
        if gIpc == True:
            crt.Screen.Send("latticeipcclient " + CLI_SET + " " + node + " " + value + "\n")
        else:
            crt.Screen.Send("latticecli -n \"" + CLI_SET + " " + node + " " + value + "\"\n")
        result = crt.Screen.WaitForStrings((CLI_ERROR + CLI_DELIMITER, CLI_SUCCESS), TIMEOUT)
        output = crt.Screen.ReadString(prompt)[:-CRT_TRIM_TRAIL]

        if result == 0:
            return (False, STR_ERROR_TIMEOUT)
        elif result == 1:
            return (False, output)
        elif result == 2:
            return (True, CLI_SUCCESS + output)
    else:
        if gIpc == True:
            cmd = "latticeipcclient " + CLI_SET + " " + node + " " + value
        else:
            cmd = "latticecli -n \"" + CLI_SET + " " + node + " " + value + "\""
        ConsoleLog(cmd, True)
        output = os.popen(cmd).readlines()
        ConsoleLog(output, False)
        found = False
        for line in output:
            if line.find(node) != -1:
                found = True
                break
        if found == False:
            return (False, STR_ERROR_TIMEOUT)
        if line.split()[0] == CLI_SUCCESS:
            return (True, line[:-1])
        else:
            return (False, line[:-1])


# <LatticeCliGet> Get output value from a Lattice node via latticecli
# <Mandatory Input> node: Lattice node
# <Optional Input> prompt: String to wait for, optional for secureCRT
# <1st Return> True for success, False for error
# <2rd Return> The output value or the error message
# e.g. LatticeCliGet("System.SoftwareVersion")
def LatticeCliGet(node, prompt = PROMPT_DEFAULT):
    if os.name == "nt":
        crt.Screen.Synchronous = True
        if gIpc == True:
            crt.Screen.Send("latticeipcclient " + CLI_GET + " " + node + "\n")
        else:
            crt.Screen.Send("latticecli -n \"" + CLI_GET + " " + node + "\"\n")
        result = crt.Screen.WaitForStrings((CLI_ERROR + CLI_DELIMITER, node + CLI_DELIMITER), TIMEOUT)
        output = crt.Screen.ReadString(prompt)[:-CRT_TRIM_TRAIL]
        if result == 2:
            return (True, output)
        else:
            return (False, STR_ERROR_TIMEOUT)
    else:
        if gIpc == True:
            cmd = "latticeipcclient " + CLI_GET + " " + node
        else:
            cmd = "latticecli -n \"" + CLI_GET + " " + node + "\""
        ConsoleLog(cmd, True)
        output = os.popen(cmd).readlines()
        ConsoleLog(output, False)
        found = False
        for line in output:
            if line.find(node) != -1:
                found = True
                break
        if found == False:
            return (False, STR_ERROR_TIMEOUT)
        if line.split()[0] == CLI_SUCCESS:
            return (True, line[(line.find(node) + len(node) + len(CLI_DELIMITER)):-1])
        else:
            return (False, line[:-1])


# <DmCliSet> Set value to an RDKM DM node via dmcli
# <Mandatory Input> node: DM node; value: value string in format "<data type> <value>"
# <Optional Input> prompt: String to wait for, optional for secureCRT
# <1st Return> True for success, False for error
# <2rd Return> The output or the error message
# e.g. DmCliSet("Device.Ethernet.Interface.1.Enable", "bool true")
def DmCliSet(node, value, prompt = RDKM_PROMPT_DEFAULT):
    #if os.name == "nt":
        # TODO
    #else:
        cmd = "dmcli eRT "+ DMCLI_SET + " " + node + " " + value
        ConsoleLog(cmd, True)
        output = os.popen(cmd).readlines()
        ConsoleLog(output, False)
        found = False
        for line in output:
            if found == False and line.find(node) != -1:
                found = True
            elif line.find(DMCLI_SUCCESS) != -1:
                return (True, line[:-1])
        if found == False:
            return (False, output[-2][:-1])
        else:
            return (False, output[-3][:-1])


# <DmCliGet> Get output value from an RDKM DM node via dmcli
# <Mandatory Input> node: Dm node
# <Optional Input> prompt: String to wait for, optional for secureCRT
# <1st Return> True for success, False for error
# <2rd Return> The output value or the error message
# e.g. DmCliGet("Device.Ethernet.Interface.1.Enable")
def DmCliGet(node, prompt = RDKM_PROMPT_DEFAULT):
    #if os.name == "nt":
        # TODO
    #else:
        cmd = "dmcli eRT "+ DMCLI_GET + " " + node
        ConsoleLog(cmd, True)
        output = os.popen(cmd).readlines()
        ConsoleLog(output, False)
        found = False
        success = False
        for line in output:
            if found == False and line.find(node) != -1:
                found = True
            elif success == False and line.find(DMCLI_SUCCESS) != -1:
                success = True
            elif line.find(DMCLI_VALUE) != -1:
                return (True, line[(line.find(DMCLI_VALUE) + len(DMCLI_VALUE)):-1])
        if found == False:
            return (False, output[-2][:-1])
        else:
            return (False, output[-3][:-1])


# Wrapper functions of the actual cli API based on the input "clicmd" type
def CliSet(clicmd, node, value, prompt = "default"):
    if clicmd == "latticecli":
        if prompt == "default":
            return LatticeCliSet(node, value)
        else:
            return LatticeCliSet(node, value, prompt)
    elif clicmd == "dmcli":
        if prompt == "default":
            return DmCliSet(node, value)
        else:
            return DmCliSet(node, value, prompt)
    else:
        MessagePrint("Unknown CLI command " + clicmd + "!", "Error")
        return (False, "Unknown CLI command!")

def CliGet(clicmd, node, prompt = "default"):
    if clicmd == "latticecli":
        if prompt == "default":
            return LatticeCliGet(node)
        else:
            return LatticeCliGet(node, prompt)
    elif clicmd == "dmcli":
        if prompt == "default":
            return DmCliGet(node)
        else:
            return DmCliGet(node, prompt)
    else:
        MessagePrint("Unknown CLI command " + clicmd + "!", "Error")
        return (False, "Unknown CLI command!")

def LatticeIpcSet(ipc):
    global gIpc
    gIpc = ipc

# <FileOpen> Print message
# <Mandatory Input> message: message to print; level: message type
def MessagePrint(message, level):
    if os.name == "nt":
        crt.Dialog.MessageBox(message, level)
    else:
        print(level + ": " + message)


# <WriteHeadlineToReport> Write headline to test report
# <Mandatory Input> fp: File handle; testModule: Test module name
# <Return> startTicks: Test start time ticks
def WriteHeadlineToReport(fp, testModule):
    fp.write("Test Report for " + testModule + "\n")
    startTicks = time.time()
    fp.write("Test starts at " + time.asctime(time.localtime(startTicks)) + "\n\n")
    return startTicks


# <WriteSummaryToReport> Write summary to test report
# <Mandatory Input> fp: File handle; startTicks: Test start time ticks; etc.
def WriteSummaryToReport(fp, startTicks, caseNum, casePassNum, caseFailNum):
    endTicks = time.time()
    fp.write("Test Ends at " + time.asctime(time.localtime(endTicks)) + "\n")
    fp.write("Test Summary\n")
    fp.write("Test duration: %d Secs\n" % round(endTicks - startTicks))
    fp.write("Total Test Case: %d\n" % caseNum)
    fp.write("Total Test Case Pass: %d\n" % casePassNum)
    fp.write("Total Test Case Fail: %d\n" % caseFailNum)


# <WriteTestCaseToReport> Write case to test report
# <Mandatory Input> fp: File handle; caseId: Case ID; caseName: Case name; casePass: Case pass?(True/False)
# <Optional Input> reason: Reason of failed test case
def WriteTestCaseToReport(fp, caseId, caseName, casePass, failReason = ""):
    if casePass == True:
        result = "Test Pass"
    else:
        result = "Test Failed"
    fp.write("(%d) " % caseId + caseName + "\n")
    fp.write(result + ". " + failReason + "\n\n")
