# $language = "python"
# $interface = "1.0"

# AutoTestFromCsv.py
#
# Description:
#     This main routine reads the test case from a test data csv file and conducts the auto test.
#     The script will create test report file and log file under the same path as the csv file.
#
# Usage:
#     Under RGLinux, python3 AutoTestFromCsv.py <Test Data CSV file> <Test Report file> <Test Log file> <ipc (optional)>.
#     Under SecureCRT, specify the test files in the main() and then run the AutoTestFromCsv.py script.


# Import modules
import os, sys, time, re

# Insert the module path to sys path
scriptPath, scriptName = os.path.split(__file__)
if not scriptPath in sys.path:
    sys.path.insert(0, scriptPath)
import UserIO

# Init crt object for secureCRT
if os.name == "nt":
    reload(UserIO)
    UserIO.InitCrt(crt)


# Constant defines
CLI_CMD_DEFAULT = "latticecli"
CLI_GET = "get"
CLI_SET = "set"
CLI_IPC = "ipc"


# <IsValueExpected> Verify the value with the given expected value
# <Mandatory Input> value; expect
# <Return> True/False
# The input expect string can be the following formats to verify the input value:
# Expect value in range: (<min>:<max>)
# Expect value in array: [<e0>;<e1>;...;<en>]
# Expect value not zero for number and not empty for string: !0
# Expect value is valid IPv4 address: {IPv4}
# Expect value is valid IPv6 address: {IPv6}
# Expect value is valid MAC address:  {MAC}
# Full match except the above
def IsValueExpected(value, expect):
    # Empty expect, just return true
    if expect == "":
        return True

    # Expect value in a range
    elif expect[0] == '(' and expect[-1] == ')':
        try:
            num = int(value)
        except ValueError:
            return False
        min = int(expect[1:-1].split(":")[0])
        max = int(expect[1:-1].split(":")[1])
        if num >= min and num <= max:
            return True
        else:
            return False

    # Expect value in array
    elif expect[0] == '[' and expect[-1] == ']':
        array = expect[1:-1].split(";")
        for element in array:
            # UserIO.MessagePrint("value:" + value + ",expect:" +  expect + ",element:" + element, "DEBUG")
            if IsValueExpected(value, element) == True:
                return True
        return False

    # Expect value in not zero or not empty
    elif expect == "!0":
        try:
            num = int(value)
            if num != 0:
                return True
            else:
                return False
        except ValueError:
            if len(value) != 0:
                return True
            else:
                return False

    elif expect == "{IPv4}":
        exp = re.compile(r'''^(?:(?:25[0-5]|2[0-4][0-9]|[1]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[1]?[0-9][0-9]?)$''', re.X)
        if exp.match(value.lower()):
            return True
        else:
            return False

    elif expect == "{IPv6}":
        exp = exp = re.compile(r'''^(?:[a-f0-9]{1,4}:){7}[a-f0-9]{1,4}$''', re.X)
        if exp.match(value.lower()):
            return True
        else:
            return False

    elif expect == "{MAC}":
        exp = exp = re.compile(r'''[0-9a-f]{2}([:])[0-9a-f]{2}(\1[0-9a-f]{2}){4}$''', re.X)
        if exp.match(value.lower()):
            return True
        else:
            return False

    # Full match
    elif expect == value:
        return True
    else:
        return False


# <AutoTestFromCsv> Run auto tests from CSV file
# <Mandatory Input> csvFp: CSV file handle; reportFp: Report file handle
# <Return> Total/Pass/Failed case number
def AutoTestFromCsv(csvFp, reportFp):
    testCaseNum = 0
    testCasePassNum = 0
    testCaseFailNum = 0
    objInstanceNum = 0

    rowOperation = 0
    rowDescription = 1
    rowCli = 2
    rowCommand = 3
    rowContext = 4
    rowNode = 5
    rowValue = 6
    rowExpect = 7
    rowDelay = 8
    opStart = "Start"
    opEnd = "End"
    opAction = "Action"
    opDelay = "Delay"

    cliCmd = CLI_CMD_DEFAULT

    for line in csvFp.readlines():
        string = line.split(",")

        #For Start operation, set the case name and reset the case result.
        if string[rowOperation] == opStart:
            testCaseName = string[rowDescription]
            testPass = True
            testFailedReason = ""
            objInstanceNum = 0
            if string[rowCommand] == CLI_GET:
                if string[rowCli] != "":
                    cliCmd = string[rowCli]
                testPass, output = UserIO.CliGet(cliCmd, string[rowContext] + "." + string[rowNode])
                objInstanceNum = int(output)
                if testPass == True:
                    if objInstanceNum == 0:
                        testPass = False
                        testFailedReason = "Unable to test zero instance object!"
                else:
                    testFailedReason = output
            elif string[rowValue] != "" and string[rowValue].isdigit() == True :
                objInstanceNum = int(string[rowValue])

        #For End operation, write the result to the test report.
        elif string[rowOperation] == opEnd:
            testCaseNum += 1
            if testPass == True:
                testCasePassNum += 1
                UserIO.WriteTestCaseToReport(reportFp, testCaseNum, testCaseName, testPass)
            else:
                testCaseFailNum += 1
                UserIO.WriteTestCaseToReport(reportFp, testCaseNum, testCaseName, testPass, testFailedReason)

        #For Action operation, send the command and verify the result.
        elif string[rowOperation] == opAction:
            if string[rowCli] != "":
                cliCmd = string[rowCli]
            if objInstanceNum > 0:
                for index in range(1, objInstanceNum + 1):
                    #If the test case already fails, do nothing.
                    if testPass == True:
                        # For cli set, only verify the result status
                        fullPath = string[rowContext]  + "." + str(index) + "." + string[rowNode]
                        if string[rowCommand] == CLI_SET:
                            testPass, testFailedReason = UserIO.CliSet(cliCmd, fullPath, string[rowValue].replace(';', ','))
                        # For cli get, verify output value if expected value is available
                        else:
                            testPass, output = UserIO.CliGet(cliCmd, fullPath)
                            if testPass == True:
                                if IsValueExpected(output.strip().replace(',', ';'), string[rowExpect]) == False:
                                    testPass = False
                                    testFailedReason = "Unexpected return value for " + fullPath + ": expect " + string[rowExpect] + " but actual " + output.strip().replace(',', ';')
                            else:
                                testFailedReason = output
            else:
                #If the test case already fails, do nothing.
                if testPass == True:
                    # For cli set, only verify the result status
                    fullPath = string[rowContext] + "." + string[rowNode]
                    if string[rowCommand] == CLI_SET:
                        testPass, testFailedReason = UserIO.CliSet(cliCmd, fullPath, string[rowValue].replace(';', ','))
                    # For cli get, verify output value if expected value is available
                    else:
                        testPass, output = UserIO.CliGet(cliCmd, fullPath)
                        if testPass == True:
                            if IsValueExpected(output.strip().replace(',', ';'), string[rowExpect]) == False:
                                testPass = False
                                testFailedReason = "Unexpected return value for " + fullPath + ": expect " + string[rowExpect] + " but actual " + output.strip().replace(',', ';')
                        else:
                            testFailedReason = output

        #For Delay operation, sleep in ms
        elif string[rowOperation] == opDelay:
            #If the test case already fails, do nothing.
            if testPass == True:
                time.sleep(int(string[rowDelay])/1000)

    return (testCaseNum, testCasePassNum, testCaseFailNum)


# Main routine
def main():
    # Set the test files
    # For secureCRT, specify the test files in the script
    if os.name == "nt":
        testDataFile = "C:\TestData.csv"
        testReportFile = "C:\TestReport.txt"
        testLogFile = "C:\TestLog.log"
    elif len(sys.argv) == 4 or len(sys.argv) == 5:
        if not os.path.exists(sys.argv[1]):
            testDataPath, testDataName = os.path.split(sys.argv[1])
            if scriptPath == "":
                testDataFile = testDataName
            else:
                testDataFile = scriptPath + "/" + testDataName
        else:
            testDataFile = sys.argv[1]

        testReportPath, testReportName = os.path.split(sys.argv[2])
        if testReportPath == "":
            if scriptPath == "":
                testReportFile = testReportName
            else:
                testReportFile = scriptPath + "/" + testReportName
        else:
            testReportFile = sys.argv[2]

        testLogPath, testLogName = os.path.split(sys.argv[3])
        if testLogPath == "":
            if scriptPath == "":
                testLogFile = testLogName
            else:
                testLogFile = scriptPath + "/" + testLogName
        else:
            testLogFile = sys.argv[3]

        if len(sys.argv) == 5 and sys.argv[4] == CLI_IPC:
            UserIO.LatticeIpcSet(True)

    else:
        UserIO.MessagePrint("Invalid input arguments", "Error")
        return

    # Open test data file
    testDataFp = open(testDataFile, "r")
    if testDataFp == None:
        UserIO.MessagePrint("Unable to open test data file", "Error")
        return

    # Get the test module
    testDataPath, testDataName = os.path.split(testDataFile)
    testModule = testDataName[:-4]

    # Create test report file
    testReportFp = open(testReportFile, "w")
    if testReportFp == None:
        UserIO.MessagePrint("Unable to open test report file", "Error")
        return

    # Start logging
    UserIO.InitLogging(True, testLogFile)

    # Write headline to the report file
    startTicks = UserIO.WriteHeadlineToReport(testReportFp, testModule)

    # Run the test
    caseNum, casePassNum, caseFailNum = AutoTestFromCsv(testDataFp, testReportFp)

    # Write Summary to the report file
    UserIO.WriteSummaryToReport(testReportFp, startTicks, caseNum, casePassNum, caseFailNum)

    # Stop logging
    UserIO.InitLogging(False)

    # Close files
    testDataFp.close()
    testReportFp.close()

main()
