 /****************************************************************************
 *
 * Broadcom Proprietary and Confidential. (c) 2017 Broadcom.  All rights reserved.
 * The term "Broadcom" refers to Broadcom Limited and/or its subsidiaries.
 *
 * Unless you and Broadcom execute a separate written software license
 * agreement governing use of this software, this software is licensed to
 * you under the terms of the GNU General Public License version 2 (the
 * "GPL"), available at [http://www.broadcom.com/licenses/GPLv2.php], with
 * the following added to such license:
 *
 * As a special exception, the copyright holders of this software give you
 * permission to link this software with independent modules, and to copy
 * and distribute the resulting executable under terms of your choice,
 * provided that you also meet, for each linked independent module, the
 * terms and conditions of the license of that module. An independent
 * module is a module which is not derived from this software. The special
 * exception does not apply to any modifications of the software.
 *
 * Notwithstanding the above, under no circumstances may you combine this
 * software in any way with any other Broadcom software provided under a
 * license other than the GPL, without Broadcom's express prior written
 * consent.
 *
 ****************************************************************************/
 /*******************************************************************************
 *
 * itc_rpc.c
 * Aug. 29 2013
 * Peter Sulc
 *
 *******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <stdbool.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <pthread.h>
#include <fcntl.h>
#include <errno.h>

#include <linux/bcm_media_gw/itc_rpc/itc_rpc.h>
#include <linux/bcm_media_gw/itc_rpc/itc_user.h>

#define DEVICE_NAME	"/dev/brpc0"

typedef struct
{
    rpc_function	*func_tab;
    int			func_cnt;
} rpc_service;

static rpc_service	rpc_services[RPC_MAX_SERVICES];
static int		itc_fd;
static pthread_t	service_pthread_id;

static void start_service_thread(void);

int rpc_init(void)
{
	itc_fd = open(DEVICE_NAME, O_RDWR);
	if (itc_fd < 0)
		printf("%s: unable to open %s error %d\n", __func__, DEVICE_NAME, itc_fd);
	service_pthread_id = 0;
	return itc_fd;
}

int rpc_uninit(void)
{
    int rc = 0;

    if (service_pthread_id) {
        int status = pthread_cancel(service_pthread_id);
        if (status != 0) {
            printf("Catastrophic error - Unable to stop itc/rpc service thread\n");
            rc = -1;
        }
        else {
            status = pthread_join(service_pthread_id, NULL);
            if(status) {
                printf("Catastrophic error - Unable to join itc/rpc service thread\n");
                rc = -1;
            }
            else {
                service_pthread_id = 0;
            }
        }
    }

    if (itc_fd > 0) {
        close(itc_fd);
        itc_fd = 0;
    }

    return rc;
}

int rpc_get_fifo_tunnel_id(char *name)
{
	return ioctl(itc_fd, ITC_IOCTL_GET_TUNNEL_ID, name);
}

/* register function table */
int rpc_register_functions(int service, rpc_function *func_tab, int func_cnt)
{
	int status;

	if (service >= 0 && service < RPC_MAX_SERVICES) {
		rpc_services[service].func_tab = func_tab;
		rpc_services[service].func_cnt = func_cnt;
		status = ioctl(itc_fd, ITC_IOCTL_SERVICE_REGISTER, service);
		if (status < 0) {
			printf("%s: ITC_IOCTL_SERVICE_REGISTER %d failed\n", __func__, service);
			if (status == -EACCES)
				printf("%s: service %d is already registered\n", __func__, service);
			return -1;
		}
		printf("registered userspace service %d functions\n", service);
		start_service_thread();
		return 0;
	}
	printf("%s: service number %d is invalid\n", __func__, service);
	return -1;
}

/* register a single function */
int rpc_register_function(int service, int func_idx, rpc_function *func)
{
	if (func == NULL) {
		printf("%s: invalid func (NULL)\n", __func__);
		return -1;
	}

	if (func_idx < rpc_services[service].func_cnt) {
		rpc_services[service].func_tab[func_idx] = *func;
	}
	else if (rpc_services[service].func_tab == NULL && func_idx < RPC_MAX_FUNCTIONS) {
		rpc_services[service].func_tab = calloc(RPC_MAX_FUNCTIONS, sizeof(rpc_function));
		rpc_services[service].func_cnt = RPC_MAX_FUNCTIONS;
		rpc_services[service].func_tab[func_idx] = *func;
		start_service_thread();
	}
	else {
		printf("%s: function index %d excedes the table size %d\n",
			   __func__, func_idx, rpc_services[service].func_cnt);
		return -1;
	}
	return 0;
}

int rpc_send_reply(int tunnel, rpc_msg *msg)
{
	rpc_msg_user umsg;
	umsg.tunnel = tunnel;
	umsg.msg = *msg;
	return ioctl(itc_fd, ITC_IOCTL_SEND_REPLY, &umsg);
}

int rpc_send_request(int tunnel, rpc_msg *msg)
{
	int rc;

	rpc_msg_user umsg;
	umsg.tunnel = tunnel;
	umsg.msg = *msg;
	rc = ioctl(itc_fd, ITC_IOCTL_SEND_REQUEST, &umsg);

	if (!rc)
	{
		*msg = umsg.msg;
	}

	return rc;
}

int rpc_send_request_timeout(int tunnel, rpc_msg *msg, int seconds)
{
	int status;
	rpc_msg_user umsg;
	umsg.tunnel = tunnel;
	umsg.msg = *msg;
	umsg.timeout = seconds;
	status = ioctl(itc_fd, ITC_IOCTL_SEND_REQUEST_TIMEOUT, &umsg);
	*msg = umsg.msg;
	return status;
}

int rpc_send_message(int tunnel, rpc_msg *msg, bool wait_for_link)
{
	rpc_msg_user umsg;
	umsg.tunnel = tunnel;
	umsg.msg = *msg;
	return ioctl(itc_fd, ITC_IOCTL_SEND_MESSAGE, &umsg);
}


void rpc_dump_msg(rpc_msg *msg)
{
	printf("msg %p Req %d Rep %d Serv %d Func %d reqcnt %d\n",
	       msg, rpc_msg_request(msg),rpc_msg_reply(msg),
	       rpc_msg_service(msg),rpc_msg_function(msg),rpc_msg_counter(msg));
	printf("%08x %08x %08x %08x\n",
	       msg->data[0], msg->data[1], msg->data[2], msg->data[3]);
}

static void *service_thread(void *arg)
{
	rpc_msg_user	umsg;
	int		status;
	for (;;) {
		status = ioctl(itc_fd, ITC_IOCTL_SERVICE_LISTEN, &umsg);
		if (status == 0) {
			int service = rpc_msg_service(&umsg.msg);
			int function = rpc_msg_function(&umsg.msg);
			int tunnel = umsg.tunnel;
			if (service < 0 || service >= RPC_MAX_SERVICES) {
				printf("%s: invalid service %d\n", __func__, service);
				continue;
			}
			if (function < rpc_services[service].func_cnt &&
				rpc_services[service].func_tab[function].func) {
				if (rpc_msg_version(&umsg.msg) !=
				    rpc_services[service].func_tab[function].version) {
					printf("%s: service %d function %d version mismatch\n",
					       __func__, service, function);
					/*
					 * TODO: Send init service error function (0:1) message to boot CPU
					 */
				} else {
					rpc_services[service].func_tab[function].func(tunnel, &umsg.msg);
				}
			} else {
				printf("%s: invalid function %d\n", __func__, function);
			}
		} else {
			if (errno != EINTR)
				printf("%s: ioctl ret(%d) error(%s)\n", __func__, status, strerror(errno));
		}
	}
	printf("%s is exiting\n", __func__);
	return NULL;
}

static void start_service_thread(void)
{
	if (itc_fd == 0)
		rpc_init();
	if (service_pthread_id == 0) {
		int status = pthread_create(&service_pthread_id, NULL, service_thread, NULL);
		if (status != 0) {
			printf("%s: Catastrophic error - Unable to create service thread\n",
				   __func__);
		}
	}
}
