#include <net-snmp/net-snmp-config.h>
#include <net-snmp/net-snmp-includes.h>
#include <net-snmp/agent/net-snmp-agent-includes.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>

#include <esafeMib.h>
#include <latticeTestMib.h>
#include <brcm_rg_mgmt.h>
#include <brcm_bfc_mgmt.h>
#include <brcm_linux_system.h>
#include <brcm_cm_factory.h>
#include <dot1d_bridge.h>
#include <brcm_wifi_mgmt.h>

#ifdef BUILD_RDKM
#include <syscfg/syscfg.h>
#endif

static int keep_running;

extern void init_vacm_vars(void);
extern void init_usmUser(void);

RETSIGTYPE
stop_server(int a) {
    keep_running = 0;
}

static void SnmpWaitChild(int signo)
{
    pid_t pid;
    int stat;

    while((pid = waitpid(-1, &stat, WNOHANG)) > 0){
        snmp_log(LOG_ERR, "child %d terminated\n", pid);
    }
    return;
}

int
main (int argc, char **argv) {
  int agentx_subagent=1; /* change this if you want to be a SNMP master agent */
  int background = 0; /* change this if you want to run in the background */
  int syslog = 0; /* change this if you want to use syslog */

  /* print log errors to syslog or stderr */
  if (syslog)
    snmp_enable_calllog();
  else
    snmp_enable_stderrlog();

  /* we're an agentx subagent? */
  if (agentx_subagent) {
    /* make us a agentx client. */
    netsnmp_ds_set_boolean(NETSNMP_DS_APPLICATION_ID, NETSNMP_DS_AGENT_ROLE, 1);
#ifdef BUILD_OPENWRT
    netsnmp_ds_set_string(NETSNMP_DS_APPLICATION_ID, NETSNMP_DS_AGENT_X_SOCKET, "/var/run/agentx.sock");
#else
    netsnmp_ds_set_string(NETSNMP_DS_APPLICATION_ID, NETSNMP_DS_AGENT_X_SOCKET, "tcp:127.0.0.1:705");
#endif
  }

  /* run in background, if requested */
  if (background && netsnmp_daemonize(1, !syslog))
      exit(1);

  signal(SIGCHLD, SnmpWaitChild);

  /* initialize tcpip, if necessary */
  SOCK_STARTUP;

  /* initialize the agent library */
  init_agent("snmp_mibs_brcmd");

  /* initialize mib code here */
#if defined(BUILD_OPENWRT) || defined(BUILD_RDKM)
  init_esafeMib();
#endif
  init_latticeTestMib();
  init_brcm_rg_mgmt();
  init_brcm_bfc_mgmt();
  init_brcm_linux_system();
  init_brcm_cm_factory();
  init_dot1d_bridge();
  init_brcm_wifi_mgmt();

#ifdef BUILD_RDKM
  syscfg_init();
#endif

  /* initialize vacm/usm access control  */
  if (!agentx_subagent) {
      init_vacm_vars();
      init_usmUser();
  }

  /* snmp_mibs_brcmd will be used to read snmp_mibs_brcmd.conf files. */
  init_snmp("snmp_mibs_brcmd");

  /* If we're going to be a snmp master agent, initial the ports */
  if (!agentx_subagent)
    init_master_agent();  /* open the port to listen on (defaults to udp:161) */

  /* In case we recevie a request to stop (kill -TERM or kill -INT) */
  keep_running = 1;
  signal(SIGTERM, stop_server);
  signal(SIGINT, stop_server);

  snmp_log(LOG_INFO,"snmp_mibs_brcmd is up and running.\n");

  /* your main loop here... */
  while(keep_running) {
    /* if you use select(), see snmp_select_info() in snmp_api(3) */
    /*     --- OR ---  */
    agent_check_and_process(1); /* 0 == don't block */
  }

  /* at shutdown time */
  snmp_shutdown("snmp_mibs_brcmd");
  SOCK_CLEANUP;

  return 0;
}
