/*
 * mdio-tool: monitor and control the Broadcom MDIO for a network interface
 *
 * Usage:
 *
 * mdio-tool [-IO?] [interface ...]
 *
 * Copyright (C) 2017 Broadcom Ltd.
 *
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <time.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <linux/sockios.h>

#ifndef __GLIBC__
#include <linux/if_arp.h>
#include <linux/if_ether.h>
#endif
#include "mii.h"

/*--------------------------------------------------------------------*/

enum opt_name {
	OPT_READ = 0,
	OPT_WRITE,
	OPT_HELP,
	OPT_MAX,
};

struct option longopts[] = {
	/* { name  has_arg  *flag  val } */
	{"read", 		0, 0, 'O'},	/* mdiobus_read from the phy.  */
	{"write", 		0, 0, 'I'},	/* mdiobus_write to the phy.  */
	{"help", 		0, 0, '?'},	/* Give help */
	{ 0, 0, 0, 0 }
};

static unsigned int opt_mask = 0;
static int skfd = -1;		/* AF_INET socket for ioctl() calls. */
static struct ifreq ifr;
static struct mii_data *mii = (struct mii_data *)&ifr.ifr_data;

/*--------------------------------------------------------------------*/

static int mdio_read(int skfd, int phy_id, int location)
{
	memset(mii, 0, sizeof(struct mii_data));
	mii->phy_id = phy_id;
	mii->reg_num = location & 0xffff;
	mii->val_in = location >> 16;

	if (ioctl(skfd, SIOCGMIIREG, &ifr) < 0) {
		fprintf(stderr, "SIOCGMIIREG on %s failed: %s\n", ifr.ifr_name,
				strerror(errno));
		return -1;
	}

	return mii->val_out;
}

static int mdio_write(int skfd, int phy_id, int location, int value)
{
	memset(mii, 0, sizeof(struct mii_data));
	mii->phy_id = phy_id;
	mii->reg_num = location & 0xffff;
	mii->val_in = value;
	mii->val_out = location >> 16;
	if (ioctl(skfd, SIOCSMIIREG, &ifr) < 0) {
		fprintf(stderr, "SIOCSMIIREG on %s failed: %s\n", ifr.ifr_name,
				strerror(errno));
		return -1;
	}

	return 0;
}

/*--------------------------------------------------------------------*/

const char *usage =
"usage: %s [-IO?] [interface ...]\n"
"       -I, --write [intf] <phy_id> [reg] [val]\n"
"       -O, --read  [intf] <phy_id> [reg]\n"
"       -?, --help\n";

int main(int argc, char **argv)
{
	char *argv1 = NULL;
	int ret, phy_id, reg, val = 0;

	if (argc > 1)
		argv1 = argv[1];

	while ((ret = getopt_long(argc, argv, "IO?", longopts, 0)) != EOF) {
		switch (ret) {
			case 'I': opt_mask |= (1 << OPT_WRITE);		break;
			case 'O': opt_mask |= (1 << OPT_READ);		break;
			case '?': opt_mask |= (1 << OPT_HELP);		break;
		}
	}

	if (!opt_mask && argv[optind]) {
		fprintf(stderr, "%s: invalid option \'%s\'\n", argv[0], argv[optind]);
		opt_mask |= (1 << OPT_HELP);
	}

	if (optind > 1 && strcmp(argv1, argv[optind - 1])) {
		fprintf(stderr, "%s: invalid option \'%s\'\n", argv[0], argv1);
		opt_mask |= (1 << OPT_HELP);
	}

	if (argc == 1 || (opt_mask & (1 << OPT_HELP))) {
		fprintf(stderr, usage, argv[0]);
		return 2;
	}

	/* Open a basic socket. */
	if ((skfd = socket(AF_INET, SOCK_DGRAM,0)) < 0) {
		perror("socket");
		exit(-1);
	}

	memset(&ifr, 0, sizeof(struct ifreq));
	strncpy(ifr.ifr_name, argv[2], sizeof(ifr.ifr_name)-1);

	if ((opt_mask & (1 << OPT_READ)) && argc > 3) {
		if (argc == 4) {
			/* Get the vitals from the interface. */
			if (ioctl(skfd, SIOCGMIIPHY, &ifr) < 0) {
				if (errno != ENODEV)
					fprintf(stderr, "SIOCGMIIPHY on '%s' failed: %s\n",
							argv[2], strerror(errno));
				close(skfd);
				return 1;
			}

			phy_id = mii->phy_id;
			reg = strtoul(argv[3], NULL, 16);
		} else {
			phy_id = strtoul(argv[3], NULL, 16);
			reg = strtoul(argv[4], NULL, 16);
		}

		ret = mdio_read(skfd, phy_id, reg);
		if (ret < 0)
			fprintf(stderr, "mdio_read: err %d: %s phy_id %x reg %x\n",
					ret, argv[2], phy_id, reg);
		else
			printf("mdio_read: %s phy_id %x reg %x -> %x\n",
					argv[2], phy_id, reg, ret & 0xffff);
	} else if ((opt_mask & (1 << OPT_WRITE)) && argc > 4) {
		if (argc == 5) {
			/* Get the vitals from the interface. */
			if (ioctl(skfd, SIOCGMIIPHY, &ifr) < 0) {
				if (errno != ENODEV)
					fprintf(stderr, "SIOCGMIIPHY on '%s' failed: %s\n",
							argv[2], strerror(errno));
				close(skfd);
				return 1;
			}

			phy_id = mii->phy_id;
			reg = strtoul(argv[3], NULL, 16);
			val = strtoul(argv[4], NULL, 16);
		} else {
			phy_id = strtoul(argv[3], NULL, 16);
			reg = strtoul(argv[4], NULL, 16);
			val = strtoul(argv[5], NULL, 16);
		}

		ret = mdio_write(skfd, phy_id, reg, val);
		if (ret < 0)
			fprintf(stderr, "mdio_write: err %d: %s phy_id %x reg %x val %x\n",
					ret, argv[2], phy_id, reg, val);
		else
			printf("mdio_write: %s phy_id %x reg %x <- %x\n",
					argv[2], phy_id, reg, val);
	}

	close(skfd);
	return ret;
}
